package com.netacom.full.extensions

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Canvas
import android.graphics.drawable.Drawable
import android.view.MotionEvent
import android.view.View
import androidx.core.content.ContextCompat
import androidx.recyclerview.widget.ItemTouchHelper
import androidx.recyclerview.widget.RecyclerView
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.logger.Logger
import com.netacom.full.R
import com.netacom.full.ui.main.chat.adapter.ListChatDiffUtilAdapter
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.lite.define.MessageStatusType
import com.netacom.lite.define.MessageType
import com.netacom.lite.entity.ui.message.NeSubMessage
import com.netacom.lite.util.AppUtils
import kotlin.math.abs

class MessageSwipeController(
    private val context: Context,
    private val swipeControllerActions: SwipeControllerActions,
    private val themeHelperImpl: ThemeHelperImpl
) :
    ItemTouchHelper.Callback() {
    interface SwipeControllerActions {
        fun showReplyUI(position: Int)
    }

    private lateinit var mView: View
    private var imageDrawable: Drawable? = null
    private var currentItemViewHolder: RecyclerView.ViewHolder? = null

    private var swipeBack = false
    private var startTracking = false

    private val iconHorizontalMargin: Int = 48
    private lateinit var recyclerView: RecyclerView
    private var iconSize = 0
    private var halfIcon = 0
    private var dX = 0f

    override fun getMovementFlags(recyclerView: RecyclerView, viewHolder: RecyclerView.ViewHolder): Int {
        mView = viewHolder.itemView
        this.recyclerView = recyclerView

        imageDrawable = ContextCompat.getDrawable(context, R.drawable.ic_reply)

        iconSize = imageDrawable?.intrinsicHeight ?: 0
        halfIcon = iconSize / 2

        val pos = viewHolder.layoutPosition

        var posItem: NeSubMessage? = null
        recyclerView.adapter?.itemCount?.let {
            if (it > pos) {
                posItem = (recyclerView.adapter as? ListChatDiffUtilAdapter)?.getData?.get(pos)
            }
        }
        val swipeFlag: Int
        if (posItem?.neMessage != null &&
            (
                posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_TEXT ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_IMAGE ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_VIDEO ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_FILE ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_STICKER ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_REPLY ||
                    posItem?.neMessage?.type == MessageType.MESSAGE_TYPE_AUDIO
                ) &&
            (
                posItem?.neMessage?.status != MessageStatusType.MESSAGE_FAILED &&
                    posItem?.neMessage?.status != MessageStatusType.MESSAGE_SENDING &&
                    posItem?.neMessage?.status != MessageStatusType.MESSAGE_DELETED
                )
        ) {
            swipeFlag = makeMovementFlags(
                ItemTouchHelper.ACTION_STATE_IDLE,
                ItemTouchHelper.LEFT
            )
        } else {
            swipeFlag = 0
        }

        return swipeFlag
    }

    override fun onMove(
        recyclerView: RecyclerView,
        viewHolder: RecyclerView.ViewHolder,
        target: RecyclerView.ViewHolder
    ): Boolean {
        Logger.d("MessageSwipeController - onMove")
        return false
    }

    override fun onSwiped(viewHolder: RecyclerView.ViewHolder, direction: Int) {
        Logger.d("MessageSwipeController - onSwiped")
    }

    override fun convertToAbsoluteDirection(flags: Int, layoutDirection: Int): Int {
        if (swipeBack) {
            swipeBack = false
            return 0
        }
        return super.convertToAbsoluteDirection(flags, layoutDirection)
    }

    override fun onChildDraw(
        c: Canvas,
        recyclerView: RecyclerView,
        viewHolder: RecyclerView.ViewHolder,
        dX: Float,
        dY: Float,
        actionState: Int,
        isCurrentlyActive: Boolean
    ) {
        if (actionState == ItemTouchHelper.ACTION_STATE_SWIPE) {
            setTouchListener(recyclerView, viewHolder)
        }

        if (mView.translationX < AppUtils.pxToDp(context = context, px = 130).toInt() || dX < this.dX) {
            super.onChildDraw(c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive)
            this.dX = dX
            startTracking = true
        }
        currentItemViewHolder = viewHolder
        drawBTN(c)
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun setTouchListener(recyclerView: RecyclerView, viewHolder: RecyclerView.ViewHolder) {
        recyclerView.setOnTouchListener { _, event ->
            swipeBack = event.action == MotionEvent.ACTION_CANCEL || event.action == MotionEvent.ACTION_UP
            if (swipeBack) {
                if (abs(mView.translationX) >= AppUtils.pxToDp(context = context, px = 200).toInt()) {
                    swipeControllerActions.showReplyUI(viewHolder.layoutPosition)
                }
            }
            false
        }
    }

    private fun drawBTN(canvas: Canvas) {
        try {
            if (dX > 0) {
                // Swiping Right
                if (dX > iconHorizontalMargin) {
                    val top =
                        mView.top + ((mView.bottom - mView.top) / 2 - halfIcon)

                    imageDrawable?.apply {
                        setBounds(
                            mView.left + iconHorizontalMargin,
                            top,
                            mView.left + iconHorizontalMargin + intrinsicWidth,
                            top + intrinsicHeight
                        )
                    }
                    imageDrawable?.draw(canvas)
                }
            } else if (dX < 0) {
                // Swiping Left
                val imgLeft: Int
                if (dX < -iconHorizontalMargin) {
                    val halfIcon = iconSize / 2
                    val top =
                        mView.top + ((mView.bottom - mView.top) / 2 - halfIcon)
                    imgLeft =
                        mView.right - iconHorizontalMargin - halfIcon * 2

                    imageDrawable?.apply {
                        setBounds(
                            imgLeft,
                            top,
                            mView.right - iconHorizontalMargin,
                            top + intrinsicHeight
                        )
                    }
                    imageDrawable?.draw(canvas)
                }
            }
        } catch (e: Exception) {
            FirebaseCrashlytics.getInstance().recordException(e)
            e.printStackTrace()
        }
    }
}
