/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.firebase

import androidx.work.WorkManager
import androidx.work.workDataOf
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage
import com.netacom.base.chat.android_utils.AppUtils.isAppForeground
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.base.chat.logger.Logger
import com.netacom.full.extensions.enqueueOneTimeNetworkWorkRequest
import com.netacom.full.ui.sdk.NetAloSDK
import com.netacom.full.worker.call.CallWorker
import com.netacom.full.worker.message.MessageWorker
import com.netacom.lite.define.FirebaseDefine
import com.netacom.lite.define.FirebaseDefine.KEY_GET_DATA
import com.netacom.lite.entity.fcm.FcmMessage
import com.netacom.lite.entity.fcm.FcmNotification
import com.netacom.lite.entity.fcm.FcmPayload
import com.netacom.lite.entity.fcm.FcmSecretPayload
import com.netacom.lite.entity.socket.Call
import com.netacom.lite.local.db.DbManager
import com.netacom.lite.local.prefs.PreferencesHelperImpl
import com.netacom.lite.mapper.Fcm_Firebase_Call_Mapper
import com.netacom.lite.mapper.Fcm_Firebase_Mapper
import com.netacom.lite.socket.IOSocket
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint
import org.json.JSONObject
import javax.inject.Inject

/**
Created by vantoan on 28/Sep/2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

@AndroidEntryPoint
class FirebaseService : FirebaseMessagingService() {
    @Inject
    lateinit var jsonSerializer: JsonSerializer

    @Inject
    lateinit var fcmFirebaseMapper: Fcm_Firebase_Mapper

    @Inject
    lateinit var fcmFirebaseCallMapper: Fcm_Firebase_Call_Mapper

    @Inject
    lateinit var preferencesHelperImpl: PreferencesHelperImpl

    @Inject
    lateinit var dbManager: DbManager

    @Inject
    lateinit var socket: IOSocket

    private val isShowLog = true

    override fun onNewToken(newToken: String) {
        super.onNewToken(newToken)
        if (::preferencesHelperImpl.isInitialized) {
            preferencesHelperImpl.fcmToken = newToken
            if (isShowLog) Logger.e("Fcm -> New token = $newToken")
        }
    }

    override fun onMessageReceived(remoteMessage: RemoteMessage) {
        super.onMessageReceived(remoteMessage)
        if (isShowLog) Logger.e("Fcm -> onMessageReceived${remoteMessage.data}")
        NetAloSDK.eventFireBase(remoteMessage.data)
        if (::jsonSerializer.isInitialized) {
            (remoteMessage.data as? Map<*, *>)?.let {
                (it[KEY_GET_DATA] as? String)?.apply {
                    jsonSerializer.asObject(this, FcmMessage::class.java)?.let { fcmMessage ->
                        if (isShowLog) Logger.e("Fcm -> fcmMessage::$fcmMessage")
                        when (fcmMessage.type) {
                            FirebaseDefine.TYPE_MESSAGE -> handleFcmMessage(fcmMessage.payload)
                            FirebaseDefine.TYPE_CALLING -> handleFcmCalling(fcmMessage.payload)
                            FirebaseDefine.TYPE_MESSAGE_SECRET -> handleFcmMessage(fcmMessage.payload)
                            FirebaseDefine.TYPE_START_SECRET -> handleFcmStartSecret(fcmMessage.payload)
                            FirebaseDefine.TYPE_REGISTER -> handleFcmNewUser(fcmMessage.payload)
                        }
                    }
                }
            }
        }
    }

    private fun handleFcmMessage(fcmPayload: String?) {
        jsonSerializer.asObject(fcmPayload, FcmPayload::class.java)?.apply {
            val fcmNotification = fcmFirebaseMapper.mapToEntity(this)
            if (isShowLog) Logger.e("Fcm -> handleFcmMessage: $fcmNotification")
            if (isShowNotification(fcmNotification) && fcmNotification.isCallSuccess != true) {
                val fcmNotificationJson = jsonSerializer.asJson(fcmNotification, FcmNotification::class.java)
                WorkManager.getInstance(applicationContext).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                    workDataOf(FirebaseDefine.KEY_DATA to fcmNotificationJson)
                )
            }
        }
    }

    private fun handleFcmCalling(fcmPayload: String?) {
        jsonSerializer.asJson(fcmFirebaseCallMapper.mapFromEntity(JSONObject(fcmPayload ?: "").getJSONObject(FirebaseDefine.TYPE_CALLING)), Call::class.java)
            ?.apply {
                WorkManager.getInstance(applicationContext).enqueueOneTimeNetworkWorkRequest<CallWorker>(
                    workDataOf(FirebaseDefine.KEY_DATA to this)
                )
            }
    }

    private fun handleFcmStartSecret(fcmPayload: String?) {
        jsonSerializer.asObject(fcmPayload, FcmSecretPayload::class.java)?.apply {
            val fcmNotification = jsonSerializer.asJson(fcmFirebaseMapper.mapToEntity(this), FcmNotification::class.java)
            if (isShowLog) Logger.e("Fcm -> handleFcmStartSecret: $fcmNotification")
            WorkManager.getInstance(applicationContext).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                workDataOf(FirebaseDefine.KEY_DATA to fcmNotification)
            )
        }
    }

    private fun handleFcmNewUser(fcmPayload: String?) {
        // show notify when socket disconnect
        if (!socket.isConnect()) {
            jsonSerializer.asObject(fcmPayload, FcmPayload::class.java)?.apply {
                fcmFirebaseMapper.mapToEntityNewUser(this).apply {
                    val fcmNotification = jsonSerializer.asJson(
                        this,
                        FcmNotification::class.java
                    )
                    WorkManager.getInstance(applicationContext).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                        workDataOf(FirebaseDefine.KEY_DATA to fcmNotification)
                    )
                }
            }
        }
    }

    override fun onDeletedMessages() {
        super.onDeletedMessages()
        Logger.e("onDeletedMessages")
    }

    /**
     * Condition show notification
     */
    private fun isShowNotification(fcmNotification: FcmNotification): Boolean {
        if (!dbManager.checkIsMute(fcmNotification.group_id.toString())) {
            if (!isAppForeground()) {
                return true
            } else if (fcmNotification.mSenderUin != preferencesHelperImpl.getUserId) {
                if (NetAloSDK.fragmentDisplay != Constants.GROUP_FRAGMENT && NetAloSDK.fragmentDisplay != Constants.BASE_CHAT_FRAGMENT) {
                    return true
                } else {
                    if (NetAloSDK.fragmentDisplay == Constants.GROUP_FRAGMENT) {
                        return false
                    } else if (NetAloSDK.fragmentDisplay == Constants.BASE_CHAT_FRAGMENT) {
                        return NetAloSDK.activeGroup != fcmNotification.group_id.toString()
                    }
                }
            }
        }
        return false
    }
}
