/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.login.otp

import android.view.KeyEvent
import androidx.appcompat.widget.AppCompatEditText
import androidx.core.widget.doAfterTextChanged
import com.netacom.base.chat.android_utils.KeyboardUtils
import com.netacom.base.chat.android_utils.SpanUtils
import com.netacom.base.chat.type.ScreenState
import com.netacom.base.chat.util.DateTimeUtils
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDataFragment
import com.netacom.full.databinding.FragmentOtpBinding
import com.netacom.full.extensions.setFocusable
import com.netacom.full.ui.sdk.NetAloSDK
import com.netacom.full.utils.DialogUtil
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.ObsoleteCoroutinesApi
import kotlinx.coroutines.channels.ticker
import kotlinx.coroutines.launch

/**
Created by vantoan on 03/Aug/2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

@ObsoleteCoroutinesApi
@AndroidEntryPoint
class OtpFragment : BaseSDKDataFragment<FragmentOtpBinding, OtpViewModel>(
    R.layout.fragment_otp,
    OtpViewModel::class.java
) {

    override fun setViewModel(): Int = BR.viewModel
    private val tickerChannel = ticker(delayMillis = 1000, initialDelayMillis = 0)

    override fun initViews() {
        initOtp()
        view?.apply {
            KeyboardUtils.showSoftInput(this)
        }
    }

    private fun initOtp() {
        with(binding) {
            txtOTP01.setFocusable()
            txtOTP01.otpTextChanged(0, txtOTP02)
            txtOTP02.otpTextChanged(1, txtOTP03)
            txtOTP03.otpTextChanged(2, txtOTP04)
            txtOTP04.otpTextChanged(3, txtOTP05)
            txtOTP05.otpTextChanged(4, txtOTP06)
            txtOTP06.doAfterTextChanged {
                it ?: return@doAfterTextChanged
                if (it.isNotEmpty()) {
                    this@OtpFragment.viewModel.otps[5] = it[0]
                    KeyboardUtils.showSoftInput()
                    activity?.currentFocus?.clearFocus()
                    this@OtpFragment.viewModel.confirmRegisterOTP(
                        {
                            this@OtpFragment.viewModel.openUpdateProfile()
                        },
                        {
                            NetAloSDK.openNetAlo(requireContext())
                        }
                    )
                } else {
                    this@OtpFragment.viewModel.otps[5] = '\u0000'
                }
            }

            txtOTP02.otpDeleted(txtOTP01)
            txtOTP03.otpDeleted(txtOTP02)
            txtOTP04.otpDeleted(txtOTP03)
            txtOTP05.otpDeleted(txtOTP04)
            txtOTP06.otpDeleted(txtOTP05)
        }
    }

    override fun initData() {
        SpanUtils.with(binding.tvStatus).append(getString(R.string.otp_send_to)).append(viewModel.getPhone()).setBold().create()
        startCountResendOtp()
        viewModel.enableRetry.observeOnce {
            if (!it) {
                startCountResendOtp()
            }
        }

        viewModel.errorOtp.observeOnce {
            showOTPInvalid()
        }

        viewModel.timeOutOtp.observeOnce {
            showTimeoutPopup()
        }

        viewModel.screenState.observe {
            if (it.state == ScreenState.OTHER || it.state == ScreenState.DATA) {
                view?.apply {
                    KeyboardUtils.hideSoftInput(this)
                }
                if (it.state == ScreenState.OTHER) {
                    activity?.finish()
                }
            }
        }
    }

    private fun startCountResendOtp() {
        launch {
            viewModel.checkTimeout = false
            repeat(viewModel.RESEND_OTP_TIMEOUT) {
                tickerChannel.receive()
                binding.txtTime.text = DateTimeUtils.convertCountTimeToString(
                    viewModel.RESEND_OTP_TIMEOUT.toLong() - it - 1
                )
            }
            binding.txtTime.isEnabled = true
            viewModel.retryOtp(isRetry = true)
            viewModel.checkTimeout = true
        }
    }

    private fun stopCountResendOtp() {
        tickerChannel.cancel()
    }

    private fun AppCompatEditText.otpTextChanged(otpIndex: Int, focusNextItem: AppCompatEditText) {
        this.doAfterTextChanged {
            it ?: return@doAfterTextChanged
            if (it.isNotEmpty()) {
                viewModel.otps[otpIndex] = it[0]
                focusNextItem.setFocusable()
            } else {
                viewModel.otps[otpIndex] = '\u0000'
            }
        }
    }

    private fun AppCompatEditText.otpDeleted(focusNextItem: AppCompatEditText) {
        this.setOnKeyListener { _, keyCode, _ ->
            if (keyCode == KeyEvent.KEYCODE_DEL && this.text.isNullOrEmpty()) {
                focusNextItem.setFocusable()
            }
            false
        }
    }

    private fun resetOtp() {
        with(binding) {
            txtOTP01.text = null
            txtOTP02.text = null
            txtOTP03.text = null
            txtOTP04.text = null
            txtOTP05.text = null
            txtOTP06.text = null
        }
        binding.txtOTP01.requestFocus()
        binding.txtOTP01.setSelectAllOnFocus(true)
    }

    private fun showTimeoutPopup() {
        DialogUtil.showMessage(
            requireContext(),
            message = R.string.popup_otp_expired,
            okFunc = {
                resetOtp()
            },
            okLabel = R.string.popup_confirm
        )
    }

    private fun showOTPInvalid() {
        DialogUtil.showMessage(
            requireContext(),
            message = R.string.popup_otp_not_match,
            okFunc = {
                resetOtp()
            },
            okLabel = R.string.popup_confirm
        )
    }

    override fun syncEvent() {
    }

    override fun setupTheme() {
        themeHelperImpl.setThemeColorForViews(binding.tvChangePhoneNumber)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        stopCountResendOtp()
    }
}
