/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main

import android.Manifest
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.provider.Settings
import androidx.activity.result.ActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.Keep
import androidx.core.content.ContextCompat
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.ViewModelProvider
import com.fondesa.kpermissions.allGranted
import com.fondesa.kpermissions.anyPermanentlyDenied
import com.fondesa.kpermissions.extension.permissionsBuilder
import com.fondesa.kpermissions.extension.send
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.AppUtils
import com.netacom.base.chat.android_utils.BarUtils
import com.netacom.base.chat.android_utils.KeyboardUtils
import com.netacom.base.chat.android_utils.NetworkUtils
import com.netacom.base.chat.base.BaseViewModel
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.getThemeColor
import com.netacom.base.chat.util.isNull
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKActivity
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch

@Keep
abstract class MainSdkActivity<DB : ViewDataBinding, VM : BaseViewModel>(
    layoutID: Int,
    viewModelClass: Class<VM>
) : BaseSDKActivity<DB, VM>(layoutID, viewModelClass) {

    private lateinit var themeHelperImpl: ThemeHelperImpl

    private val sdkViewModel by lazy {
        ViewModelProvider(this).get(MainSdkViewModel::class.java)
    }

    private val starPermissionForResult = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { _: ActivityResult ->
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && !Settings.canDrawOverlays(this)) {
            sdkViewModel.getPreferences.nePermission?.isCallOverlay = true
            showSnackBar(R.string.call_miss_notification)
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        listenerGroupLink()
        initView()
    }

    private fun initView() {
        KeyboardUtils.fixAndroidBug5497(this)
        sdkViewModel.loadStickers()
        observeNetworkChange()
    }

    private fun observeNetworkChange() {
        NetworkUtils.registerNetworkStatusChangedListener(
            object : NetworkUtils.OnNetworkStatusChangedListener {
                override fun onDisconnected() {
                }

                override fun onConnected(networkType: NetworkUtils.NetworkType?) {
                    sdkViewModel.getPreferences.networkChange = networkType?.name
                }
            }
        )
    }

    private fun initColorStatusBar() {
        this.apply {
            BarUtils.setStatusBarLightMode(this, false)
            if (::themeHelperImpl.isInitialized) {
                if (themeHelperImpl.isDarkMode) {
                    BarUtils.setNavBarColor(this, ContextCompat.getColor(this, com.netacom.base.chat.R.color.color_black))
                } else {
                    BarUtils.setNavBarColor(this, getThemeColor(themeHelperImpl.mainColor))
                }
            }
        }
    }

    private fun listenerGroupLink() {
        sdkViewModel.joinGroupWithDeepLink.observeOnce { neGroup ->
            neGroup?.let { _neGroup ->
                Logger.e("joinGroupWithDeepLink==$_neGroup")
                sdkViewModel.openChat(_neGroup)
            }
        }
    }

    private fun joinGroup() {
        launch {
            sdkViewModel.getPreferences.getDeepLinkTokenFlow().collect { deepLinkToken ->
                deepLinkToken?.let { _deepLinkToken ->
                    if (sdkViewModel.getIOSocket.isConnect()) {
                        sdkViewModel.joinGroupWithDeepLink(_deepLinkToken)
                        sdkViewModel.getPreferences.setDeepLinkToken(null)
                    }
                }
            }
        }
    }

    fun checkPermissionAll() {
        try {
            // val checkPermission = AutoStartPermissionHelper.getInstance().isAutoStartPermissionAvailable(this)
            val isSyncContact = sdkViewModel.getPreferences.sdkConfig?.isSyncContact
            if (isSyncContact == true) {
                permissionsBuilder(
                    Manifest.permission.READ_CONTACTS, Manifest.permission.WRITE_CONTACTS, Manifest.permission.READ_EXTERNAL_STORAGE,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE, Manifest.permission.RECORD_AUDIO
                )
            } else {
                permissionsBuilder(
                    Manifest.permission.READ_EXTERNAL_STORAGE,
                    Manifest.permission.WRITE_EXTERNAL_STORAGE, Manifest.permission.RECORD_AUDIO
                )
            }.build()
                .send { status ->
                    when {
                        status.allGranted() -> {
                            Logger.e("checkPermissionAll==allGranted:" + status.map { it.permission })
                            if (status.any { it.permission == Manifest.permission.WRITE_CONTACTS }) {
                                sdkViewModel.getPreferences.isContactPermission = true
                                if (sdkViewModel.listContact.value.isNull) {
                                    sdkViewModel.syncData(isSyncContact = true)
                                }
                            } else {
                                sdkViewModel.getPreferences.isContactPermission = false
                                if (isSyncContact == false) {
                                    sdkViewModel.syncData(isSyncContact = false)
                                }
                            }
                            checkPermissionCall()
                        }
                        status.anyPermanentlyDenied() -> {
                            Logger.e("checkPermissionAll==anyPermanentlyDenied:" + status.map { it.permission })
                            showSnackBar(
                                R.string.permission_storage_denied, R.string.permission_open_setting,
                                {
                                    AppUtils.launchAppDetailsSettings()
                                }
                            )
                            checkPermissionCall()
                        }
                        else -> {
                            Logger.e("checkPermissionAll==else")
                            showSnackBar(R.string.permission_storage_denied)
                            checkPermissionCall()
                        }
                    }
                }
        } catch (e: Exception) {
            e.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(e)
        }
    }

    private fun checkPermissionCall() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            if (!Settings.canDrawOverlays(this) || sdkViewModel.getPreferences.nePermission?.isCallOverlay == false) {
                val intent = Intent(Settings.ACTION_MANAGE_OVERLAY_PERMISSION, Uri.parse("package:${AppUtils.getAppPackageName()}"))
                starPermissionForResult.launch(intent)
            }
        }
    }

    override fun onResume() {
        super.onResume()
        initColorStatusBar()
        // sdkViewModel.reConnectSocket()
        joinGroup()
    }

    fun initTheme(themeHelperImpl: ThemeHelperImpl) {
        try {
            this.themeHelperImpl = themeHelperImpl
        } catch (e: Exception) {
            FirebaseCrashlytics.getInstance().recordException(e)
            e.printStackTrace()
        }
    }
}
