/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main

import android.content.Context
import android.util.Log
import android.widget.Toast
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.navigation.NavController
import androidx.work.WorkManager
import androidx.work.workDataOf
import com.google.android.gms.tasks.OnCompleteListener
import com.google.firebase.FirebaseApp
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.google.firebase.messaging.FirebaseMessaging
import com.netacom.base.chat.android_utils.AppUtils.isAppForeground
import com.netacom.base.chat.android_utils.SpanUtils
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.define.ConfigDef
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.model.ScreenStateObj
import com.netacom.base.chat.network.ApiResponseError
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.network.ResultData
import com.netacom.base.chat.util.getOnlyDigits
import com.netacom.base.chat.util.isNull
import com.netacom.base.chat.util.unAccentText
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKViewModel
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.enqueueOneTimeNetworkWorkRequest
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.full.ui.main.calling.CallingUtils
import com.netacom.full.ui.sdk.NetAloSDK
import com.netacom.full.utils.DialogUtil
import com.netacom.full.utils.convertPhoneNumber
import com.netacom.full.utils.getContactName
import com.netacom.full.utils.getContactNumbers
import com.netacom.full.worker.message.MessageWorker
import com.netacom.lite.define.CallDefine
import com.netacom.lite.define.FirebaseDefine
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MessageStatusType
import com.netacom.lite.define.NotificationType.NOTIFICATION_TYPE_DELETE_SECRET
import com.netacom.lite.define.SocketMessageStatus
import com.netacom.lite.define.SocketMessageStatus.MESSAGE_STATUS_DELETED
import com.netacom.lite.define.SocketState
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.database.attachment.NeDocument
import com.netacom.lite.entity.database.contact.DbContactEntity
import com.netacom.lite.entity.database.group.DbGroupEntity
import com.netacom.lite.entity.fcm.FcmNotification
import com.netacom.lite.entity.socket.Call
import com.netacom.lite.entity.socket.Event
import com.netacom.lite.entity.socket.EventType
import com.netacom.lite.entity.socket.Group
import com.netacom.lite.entity.socket.Message
import com.netacom.lite.entity.ui.contact.ContactInfo
import com.netacom.lite.entity.ui.contact.LocalContactInfo
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.local.LocalFileModel
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.entity.ui.secret.NePublicKey
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.network.model.request.ContactItemRequest
import com.netacom.lite.network.model.response.ContactResponse
import com.netacom.lite.repository.AuthRepository
import com.netacom.lite.repository.ContactRepository
import com.netacom.lite.repository.DownloadRepository
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.repository.MessageRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.repository.StickerRepository
import com.netacom.lite.repository.UserRepository
import com.netacom.lite.repository.base.State
import com.netacom.lite.repository.configs.GroupConfig
import com.netacom.lite.repository.secretChat.ChatSecretUtils
import com.netacom.lite.repository.secretChat.impl.RealmStatic
import com.netacom.lite.socket.request.AcceptSecretMessageRequest
import com.netacom.lite.socket.request.CheckCallRequest
import com.netacom.lite.socket.request.CreateDeepLinkRequest
import com.netacom.lite.socket.request.DisableDeepLinkRequest
import com.netacom.lite.socket.request.EventRequest
import com.netacom.lite.socket.request.JoinGroupWithLinkRequest
import com.netacom.lite.socket.request.ListLinkGroupRequest
import com.netacom.lite.socket.request.StartSecretMessageRequest
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.CallbackStickerResult
import com.netacom.lite.util.Constants
import com.netacom.lite.util.FileUtils
import com.netacom.lite.util.getUrlImage
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.withContext
import okhttp3.ResponseBody
import java.io.File
import javax.inject.Inject
import kotlin.system.measureTimeMillis

/**
Created by vantoan on 31,July,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/
@HiltViewModel
class MainSdkViewModel @Inject constructor(
    private val authRepository: AuthRepository,
    private val socketRepository: SocketRepository,
    private val messageRepository: MessageRepository,
    private val groupRepository: GroupRepository,
    private val contactRepository: ContactRepository,
    private val stickerRepository: StickerRepository,
    private val downloadRepository: DownloadRepository,
    private val userRepository: UserRepository,
    private val jsonSerializer: JsonSerializer,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseSDKViewModel(socketRepository, groupRepository, navigationDispatcher) {
    private val isShowLog = true
    private var isSyncLocalContact = false
    var mainViewModelCall: Call? = null
    val groupScreenState: LiveData<ScreenStateObj> = MutableLiveData()
    val contactScreenState: LiveData<ScreenStateObj> = MutableLiveData()

    private val _listContact = MutableLiveData<EventLiveData<List<NeUser>>>()
    val listContact: LiveData<EventLiveData<List<NeUser>>> = _listContact

    private val _listGroup = MutableLiveData<EventLiveData<List<NeGroup>>>()
    val listGroup: LiveData<EventLiveData<List<NeGroup>>> = _listGroup

    private var _isConnectSocket = MutableLiveData<Boolean>().apply { value = false }
    val isConnectSocket: LiveData<Boolean> = _isConnectSocket

    private var _userStatus = MutableLiveData<EventLiveData<NeUser>>()
    val userStatus: LiveData<EventLiveData<NeUser>> = _userStatus

    private val _eventUpdate = MutableLiveData<EventLiveData<Event>>()
    val eventUpdate: LiveData<EventLiveData<Event>> = _eventUpdate

    private val _messageReceive = MutableLiveData<EventLiveData<NeMessage>>()
    val messageReceive: LiveData<EventLiveData<NeMessage>> = _messageReceive

    val _messageUpdate = MutableLiveData<Message?>()
    var messageUpdate: LiveData<Message?> = _messageUpdate

    val _messageDelete = MutableLiveData<Message?>()
    var messageDelete: LiveData<Message?> = _messageDelete

    var _acceptSecretChat = MutableLiveData<NeGroup?>()
    val acceptSecretChat: LiveData<NeGroup?> = _acceptSecretChat

    val deleteSecretGroup = MutableLiveData<Long>()

    var _createDeepLink = MutableLiveData<String?>()
    val createDeepLink: LiveData<String?> = _createDeepLink

    private var _disableDeepLink = MutableLiveData<EventLiveData<Boolean>>()
    val disableDeepLink: LiveData<EventLiveData<Boolean>> = _disableDeepLink

    private var _joinGroupWithDeepLink = MutableLiveData<EventLiveData<NeGroup?>>()
    val joinGroupWithDeepLink: LiveData<EventLiveData<NeGroup?>> = _joinGroupWithDeepLink

    var cacheListGroup: List<NeGroup>? = emptyList()

    init {
        initGroupLink()
    }

    // SDK
    fun initSDK(neUserChat: NeUser?, callbackSuccess: () -> Unit) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            if (isShowLog) Logger.e("initSDK:config=" + getPreferences.sdkConfig + "user=" + getPreferences.user)
            updateProfile(getUser, neUserChat)
            initKeyForSecretChat()
            initSocket(neUserChat)
            callbackSuccess()
        }
    }

    private suspend fun updateProfile(neUser: NeUser?, neUserChat: NeUser?) = withContext(getPostExecutionThread.io) {
        if (neUser == null || neUser.username.isNull || neUser.avatar.isNull) return@withContext
        authRepository.updateProfile(name = neUser.username, avatar = neUser.avatar, email = neUser.email).collect {
            if (it is ApiResponseSuccess) {
                if (isShowLog) Logger.e("updateProfileSuccess")
            } else {
                if (isShowLog) Logger.e("updateProfileFail")
            }
            initKeyForSecretChat()
            initSocket(neUserChat)
        }
    }

    // Core
    fun connectSocket() = socketRepository.connectSocket()

    fun disConnectSocket() = socketRepository.disConnectSocket()

    // NetAlo
    private fun registerFcm(tokenFcm: String?) {
        getPreferences.fcmToken = tokenFcm
        socketRepository.registerFcm(token = tokenFcm)
    }

    fun initKeyForSecretChat() {
        RealmStatic.init(realmManager = getDbManager)
        ChatSecretUtils.loadKeyAndInit(getPreferences)
    }

    private suspend fun initSocket(neUserChat: NeUser?) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketRepository.connectSocket()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketState(neUserChat)
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketUpdateUserStatus()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketUpdateCall()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketUpdateStartCall()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketDeleteMessage()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketMessage()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketCreateGroup()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketUpdateGroup()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketDeleteGroup()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketGroup()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketUpdateMessage()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            executeStartSecretChatEvent()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            executeAcceptSecretChatEvent()
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            executeDeleteSecretChatEvent()
        }
    }

    private suspend fun socketState(neUserChat: NeUser?) {
        getIOSocket.flowState.collect { state ->
            when (state) {
                SocketState.START -> {
                    if (isShowLog) Logger.e("SocketState.START")
                    _isConnectSocket.post(true)
                }
                SocketState.CONNECTED -> {
                    if (isShowLog) Logger.e("SocketState.CONNECTED")
                    initFirebase()
                }
                SocketState.CONNECTING -> if (isShowLog) Logger.e("SocketState.CONNECTING")
                SocketState.CLOSED -> if (isShowLog) Logger.e("SocketState.CLOSED")
                SocketState.LOGIN_SUCCESS -> {
                    if (isShowLog) Logger.e("SocketState.LOGIN_SUCCESS")
                    _isConnectSocket.post(true)
                    neUserChat?.apply {
                        startOneByOneChat(this, true)
                    } ?: syncGroup(SyncType.LIST_SYNC_DB_SERVER)
                }
                SocketState.FAILURE -> {
                    if (isShowLog) Logger.e("SocketState.FAILURE")
                    _isConnectSocket.post(false)
                }
                SocketState.LOG_OUT -> {
                    if (isShowLog) Logger.e("SocketState.LOG_OUT")
                    socketRepository.clearData()
                    launchOnViewModelScope(getPostExecutionThread.main) {
                        navigationDispatcher.emit {
                            it.navigateIfSafe(R.id.loginActivity)
                        }
                    }
                }
            }
        }
    }

    private fun initFirebase() {
        try {
            if (FirebaseApp.getApps(Utils.getApp()).isNotEmpty()) {
                FirebaseMessaging.getInstance().token.addOnCompleteListener(
                    OnCompleteListener { task ->
                        if (!task.isSuccessful) {
                            Logger.e("Fetching FCM registration token failed", task.exception)
                            return@OnCompleteListener
                        }
                        Logger.e("Fetching FCM registration token ===" + task.result)
                        registerFcm(task.result)
                    }
                )
            }
        } catch (e: Exception) {
            e.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(e)
        }
    }

    private suspend fun socketUpdateUserStatus() {
        getIOSocket.flowUserStatus.collect { userStatus ->
            userStatus?.apply {
                // Logger.e("socketUpdateUserStatus==$userStatus")
                userRepository.getUser(this)?.let { neUser ->
                    _userStatus.post(EventLiveData(neUser))
                }
            }
        }
    }

    private suspend fun socketUpdateGroup() {
        getIOSocket.flowUpdateGroup.collect { updateGroup ->
            updateGroup?.apply {
                if (isShowLog) Logger.e("socketUpdateGroup=$this")

                this.blockedAll?.let {
                    if (it.isNotEmpty()) {
                        groupRepository.updateBlockUsers(it, updateGroup.groupId)?.apply {
                            updateGroup(groupRepository.convertDBToNeGroup(this))
                        }
                    }
                }

                this.unblockedAll?.let {
                    if (it.isNotEmpty()) {
                        groupRepository.updateUnblockUsers(it, updateGroup.groupId)?.apply {
                            updateGroup(groupRepository.convertDBToNeGroup(this))
                        }
                    }
                }
                if (background?.isNotEmpty() == true) {
                    getDbManager.updateBackgroundGroup(groupId, background!!)?.apply {
                        updateGroup(groupRepository.convertDBToNeGroup(this))
                    }
                }
                syncGroup()
            }
        }
    }

    private suspend fun socketUpdateCall() {
        getIOSocket.flowEvent.collect { call ->
            if (isShowLog) Logger.e("socketUpdateCall=$call")
            call?.let { event ->
                _eventUpdate.post(EventLiveData(event))
            }
        }
    }

    private suspend fun socketGroup() {
        getIOSocket.flowGroup.collect { group ->
            group?.apply {
                if (isShowLog) Logger.e("socketGroup=$this")
                if (name == "has joined") {
                    groupRepository.mapGroupToFcm(this)?.let {
                        WorkManager.getInstance(Utils.getApp()).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                            workDataOf(FirebaseDefine.KEY_DATA to it)
                        )
                        getIOSocket.flowGroup.value = null
                    }
                }
            }
        }
    }

    /**
     * ??? What is the logic here?
     */
    private suspend fun socketDeleteGroup() {
        getIOSocket.flowDeleteGroup.collect { deleteGroup ->
            deleteGroup?.apply {
                if (isShowLog) Logger.e("socketDeleteGroup=$this")
            }
        }
    }

    /**
     * ??? What is the logic here?
     */
    private suspend fun socketCreateGroup() {
        getIOSocket.flowCreateGroup.collect { createGroup ->
            createGroup?.apply {
                if (isShowLog) Logger.e("socketCreateGroup=$this")
            }
        }
    }

    private fun isShowLocalNotification(neMessage: NeMessage): Boolean {
        if (!getDbManager.checkIsMute(neMessage.groupId)) {
            val isCallSuccess = neMessage.attachment?.call?.acceptedUins?.contains(getUserId.toString())
            if (isCallSuccess == false) {
                return true
            } else if (isCallSuccess == true) {
                return false
            }
            if (!isAppForeground()) {
                return true
            } else if (neMessage.owner?.id != getUserId) {
                if (NetAloSDK.fragmentDisplay != Constants.GROUP_FRAGMENT && NetAloSDK.fragmentDisplay != Constants.BASE_CHAT_FRAGMENT) {
                    return true
                } else {
                    if (NetAloSDK.fragmentDisplay == Constants.GROUP_FRAGMENT) {
                        return false
                    } else if (NetAloSDK.fragmentDisplay == Constants.BASE_CHAT_FRAGMENT) {
                        return NetAloSDK.activeGroup != neMessage.groupId
                    }
                }
            }
        }
        return false
    }

    private fun isShowSecretChatLocalNotification(dbGroup: DbGroupEntity?): Boolean {
        if (!isAppForeground()) {
            return true
        } else {
            if (dbGroup?.groupOwnerId != getUserId) {
                if (NetAloSDK.fragmentDisplay != Constants.GROUP_FRAGMENT && NetAloSDK.fragmentDisplay != Constants.BASE_CHAT_FRAGMENT) {
                    return true
                }
            }
        }
        return false
    }

    /**
     * This logic is very confused, need to recheck the flow & refactoring it
     */
    private suspend fun socketMessage() {
        getIOSocket.flowMessage.collect { message ->
            message?.apply {
                val groupId = this.groupId?.toLongOrNull() ?: return@collect
                val dbMessageEntity = messageRepository.mapMessageSKToDb(message = this)
                val neMessage = messageRepository.mapMessageDBToUi(dbMessageEntity = dbMessageEntity)
                if (isShowLog) Logger.e("socketMessage:neMessage=$neMessage")
                // Handle message with status received or seen
                this.status = when {
                    neMessage.owner?.id == getPreferences.getUserId -> {
                        MessageStatusType.MESSAGE_SEEN
                    }
                    NetAloSDK.activeGroup == this.groupId -> {
                        MessageStatusType.MESSAGE_SEEN
                    }
                    else -> {
                        MessageStatusType.MESSAGE_RECEIVED
                    }
                }
                val dbGroupEntity = groupRepository.getGroupDbById(groupId.toString())
                if (isShowLog) Logger.e("neGroup=$dbGroupEntity")
                var isInCreaseCount = true
                if (dbGroupEntity == null) {
                    updateGroupToDB(Group(groupId = groupId))?.let { neGroup ->
                        if (neGroup.id == NetAloSDK.activeGroup || neGroup.owner?.id == getPreferences.getUserId) {
                            isInCreaseCount = false
                            updateSeenForLatestMessage(
                                groupId = neGroup.id,
                                messageId = dbMessageEntity.messageId,
                                status = if (this.status == MessageStatusType.MESSAGE_SEEN) {
                                    SocketMessageStatus.MESSAGE_STATUS_SEEN
                                } else {
                                    SocketMessageStatus.MESSAGE_STATUS_RECEIVED
                                }
                            )
                        }
                        messageRepository.insertLatestMessage(
                            dbMessageEntity = dbMessageEntity,
                            isIncreaseCount = isInCreaseCount
                        )?.run {
                            _messageReceive.post(EventLiveData(neMessage))
                            syncGroup()
                        }
                        // Show notification
                        if (isShowLocalNotification(neMessage)) {
                            WorkManager.getInstance(Utils.getApp()).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                                workDataOf(FirebaseDefine.KEY_DATA to messageRepository.mapMessageToFcm(neMessage))
                            )
                        }
                    }
                } else {
                    if (dbMessageEntity.messageGroupId == NetAloSDK.activeGroup || dbMessageEntity.messageOwnerId == getPreferences.getUserId) {
                        isInCreaseCount = false
                        updateSeenForLatestMessage(
                            groupId = dbMessageEntity.messageGroupId,
                            messageId = dbMessageEntity.messageId,
                            status = if (this.status == MessageStatusType.MESSAGE_SEEN) {
                                SocketMessageStatus.MESSAGE_STATUS_SEEN
                            } else {
                                SocketMessageStatus.MESSAGE_STATUS_RECEIVED
                            }
                        )
                    }
                    messageRepository.insertLatestMessage(
                        dbMessageEntity = dbMessageEntity,
                        isIncreaseCount = isInCreaseCount
                    )?.run {
                        _messageReceive.post(EventLiveData(neMessage))
                        syncGroup()
                    }
                    // Show local notification
                    if (isShowLocalNotification(neMessage)) {
                        WorkManager.getInstance(Utils.getApp()).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                            workDataOf(FirebaseDefine.KEY_DATA to messageRepository.mapMessageToFcm(neMessage))
                        )
                    }
                }
                /**
                 * reset flowMessage value to prevent observe old data
                 * Will replace it with EventLiveData
                 */
                getIOSocket.flowMessage.value = null
            }
        }
    }

    private suspend fun socketUpdateMessage() {
        getIOSocket.flowUpdateMessage.collect { messageUpdate ->
            messageUpdate?.let { message ->
                Logger.e("socketUpdateMessage==$message")
                _messageUpdate.postValue(message)
            }
            val messageId = messageUpdate?.messageId ?: return@collect

            /**
             * Need to recheck this flow, messageDb always return null
             */
            val messageDb = getDbManager.getMessageById(messageId = messageId)
            messageDb?.let {
                messageUpdate.receivedUins?.let { listReceived ->
                    val listReceivedUpdate = listReceived.filterNot { receiverId -> it.messageReceives.any { receiverId == it } }
                    if (listReceivedUpdate.isNotEmpty()) {
                        it.messageReceives.addAll(listReceivedUpdate)
                    }
                }

                messageUpdate.seenUins?.let { listSeen ->
                    val listSeenUpdate = listSeen.filterNot { seenId -> it.messageSeens.any { seenId == it } }
                    if (listSeenUpdate.isNotEmpty()) {
                        it.messageSeens.addAll(listSeenUpdate)
                    }
                }

                getDbManager.syncMessage(messageDb)
            }
        }
    }

    private suspend fun socketDeleteMessage() {
        getIOSocket.flowDeleteMessage.collect { messages ->
            messages?.apply {
                if (isShowLog) Logger.d("socketDeleteMessage = $this")
                if (status == MESSAGE_STATUS_DELETED.toLong() || deletedUins?.contains(getUserId) == true) {
                    launchOnViewModelScope(getPostExecutionThread.io) {
                        messageRepository.removeMessage(messageId)
                        groupRepository.deleteEncryptedFilesInMessage(messageId)
                        syncGroup()
                    }
                    _messageDelete.postValue(this)
                }
            }
        }
    }

    // Group
    private var isListGroupCalling = false
    private var syncTypeGroup: SyncType? = null
    fun syncGroup(syncType: SyncType? = null) {
        if (isListGroupCalling && syncTypeGroup == syncType) return
        isListGroupCalling = true
        syncTypeGroup = syncType ?: SyncType.LIST_SYNC_DB
        launchOnViewModelScope(getPostExecutionThread.io) {
            val syncData = syncType ?: if (isConnectSocket.value == true) SyncType.LIST_SYNC_DB_SERVER else SyncType.LIST_SYNC_DB
            socketRepository.getListGroup(GroupConfig(index = 0, groupType = syncData)).collect { state ->
                when (state) {
                    is State.SuccessLocal, is State.SuccessMerged -> {
                        val listGroup = when (state) {
                            is State.SuccessMerged -> {
                                cacheMessages(state.data)
                                cacheListGroup = state.data
                                state.data
                            }
                            is State.SuccessLocal -> {
                                if (cacheListMessages.isEmpty()) {
                                    cacheMessages(state.data)
                                }
                                state.data
                            }
                            else -> emptyList()
                        }
                        groupScreenState.postData()
                        _listGroup.post(EventLiveData(listGroup))
                        isListGroupCalling = false
                    }
                    is State.Loading -> if (_listGroup.value.isNull) groupScreenState.postDoing()
                    is State.Error -> {
                        isListGroupCalling = false
                        groupScreenState.postError(state.message)
                    }
                }
            }
        }
    }

    // Init data
    fun syncData(isSyncContact: Boolean? = true) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val millisSecond = measureTimeMillis {
                syncLocalFile(isLoadFull = true, {})
                syncContact(SyncType.LIST_SYNC_SERVER, isSyncContact)
            }
            Logger.e("Time:syncData=$millisSecond")
        }
    }

    // Contact
    fun syncContact(syncType: SyncType, isSyncContact: Boolean? = true, callbackSuccess: (List<NeUser>) -> Unit? = { }) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val contactItemRequest = ContactItemRequest(
                limit = ConfigDef.PAGE_SIZE_CONTACT_REGISTERED,
                phoneUser = getPreferences.user?.getPhone
            )
            contactRepository.getContactRegistered(contactItemRequest = contactItemRequest).collect { state ->
                when (state) {
                    is State.SuccessLocal -> {
                        // Logger.e("listContact:SuccessLocal==" + state.data.map { it.getDisplayName })
                        _listContact.post(EventLiveData(state.data))
                    }
                    is State.SuccessMerged -> {
                        if (getPreferences.isContactPermission == true && !isSyncLocalContact) {
                            getContactFromServer(Utils.getApp(), isSyncContact)
                        }
                        val listContact = state.data
                        Logger.e("listContact:SuccessMerged==" + listContact.map { it })
                        if (listContact.isNull || listContact.isEmpty()) {
                            contactScreenState.postEmpty()
                        } else {
                            contactScreenState.postData()
                            _listContact.post(EventLiveData(listContact))
                        }
                        callbackSuccess(listContact)
                    }
                    is State.Loading -> {
                    }
                    is State.Error -> {
                    }
                }
            }
        }
    }

    suspend fun getContactFromServer(context: Context, isSyncContact: Boolean? = true) {
        val contactItemRequest = ContactItemRequest(
            udid = "",
            compact = 0,
            offset = 0,
            limit = ConfigDef.PAGE_SIZE_CONTACT_ADDRESS_BOOK
        )
        contactRepository.getUserContacts(contactItemRequest).collect { state ->
            when (state) {
                is State.SuccessMerged -> {
                    if (isSyncContact == true) {
                        val contactResponses = state.data
                        val data = mutableListOf<ContactInfo>()
                        getContactFromDevice(context = context).forEach { localContactInfo ->
                            var isExits = false
                            val contactPhoneNumber = localContactInfo.contactPhone
                            for (contactResponse in contactResponses) {
                                if (isContactExist(contactResponse = contactResponse, dbContactEntity = localContactInfo)) {
                                    isExits = true
                                    break
                                }
                            }
                            if (!isExits && contactPhoneNumber != null) {
                                data.add(
                                    ContactInfo(
                                        phone = contactPhoneNumber,
                                        name = localContactInfo.getName
                                    )
                                )
                            }
                        }
                        sentContactToServer(udid = "", isForce = false, listPhoneContacts = data)
                    }
                    isSyncLocalContact = true
                }
                is State.Loading -> {
                    Log.d("ContactRepository", "State.Loading ")
                }
                is State.Error -> {
                    Log.d("ContactRepository", "State.Error = ")
                }
                else -> Unit
            }
        }
    }

    private fun isContactExist(contactResponse: ContactResponse, dbContactEntity: DbContactEntity?): Boolean {
        return (contactResponse.phone == dbContactEntity?.contactPhone && contactResponse.name == dbContactEntity?.serverName)
    }

    private suspend fun sentContactToServer(udid: String, isForce: Boolean, listPhoneContacts: List<ContactInfo>) {
        if (listPhoneContacts.isNotEmpty()) {
            val force = if (isForce) 1 else 0
            val listSize = listPhoneContacts.size
            val pageSize = ConfigDef.PAGE_SIZE_CONTACT_UPLOAD
            val page = if (listSize % pageSize == 0) listSize / pageSize else listSize / pageSize + 1
            val data = mutableListOf<List<ContactInfo>>()
            for (i in 0 until page) {
                data.add(
                    listPhoneContacts.subList(
                        i * pageSize,
                        if ((i + 1) * pageSize < listSize) (i + 1) * pageSize else listSize
                    )
                )
            }
            data.forEach { listContacts ->
                contactRepository.sendPhoneContacts(
                    udid = udid,
                    isForce = force,
                    contactInfo = listContacts
                ).collect()
            }
        }
    }

    private suspend fun getContactFromDevice(context: Context): List<DbContactEntity> = withContext(getPostExecutionThread.default) {
        val listLocalContacts = mutableListOf<LocalContactInfo>()
        var listLocalContactsDB: List<DbContactEntity> = emptyList()
        try {
            val contactsList = context.getContactName()
            val numbersList = context.getContactNumbers()
            var name: String? = null
            numbersList?.forEach { map ->
                map.value.forEach { number ->
                    var phoneNumber = getOnlyDigits(number)
                    if (phoneNumber.length < 10) {
                        phoneNumber = "0$phoneNumber"
                    }
                    // convert phone to +84
                    phoneNumber.convertPhoneNumber()?.let { _phoneNumber ->
                        if (contactsList?.any { contact ->
                            name = contact.contactName
                            contact.id == map.key
                        } == true
                        ) {
                            val localContactInfo = LocalContactInfo(
                                contactName = name,
                                contactPhone = _phoneNumber,
                                contactNameUnaccent = if (name?.isEmpty() == true) "" else name?.unAccentText()
                            )
                            listLocalContacts.add(localContactInfo)
                        }
                    }
                }
            }
            listLocalContactsDB = contactRepository.getListLocalContactsDB(listLocalContacts)
            contactRepository.saveLocalContact(listLocalContactsDB)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return@withContext listLocalContactsDB
    }

    // Load Picture
    fun syncLocalFile(isLoadFull: Boolean = false, callbackSuccess: (List<LocalFileModel>) -> Unit) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val localMediaFile = downloadRepository.getAllLocalFileForMedia(isLoadFull)
            getDbManager.saveListLocalFileToDb(localMediaFile)?.let {
                withContext(getPostExecutionThread.main) {
                    callbackSuccess(downloadRepository.localFileMapper(it))
                }
            }
        }
    }

    // Sticker
    fun loadStickers() {
        launchOnViewModelScope(getPostExecutionThread.io) {
            stickerRepository.initStickers(
                object : CallbackStickerResult<Boolean> {

                    override fun callBackDefaultStickerSuccess(result: Boolean) {
                        if (!result) stickerRepository.addDefaultStickers()
                    }

                    override fun callBackMoreStickerSuccess(result: Boolean) {
                        if (!result) stickerRepository.getMoreStickersFromServer(downloadRepository, callbackSuccess = {})
                    }
                }
            )
        }
    }

    fun handleFile(isEncryptedFile: Boolean, messageId: String?, neDocument: NeDocument, context: Context) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val fileUrl = neDocument.url ?: return@launchOnViewModelScope
            val downloadFilePath = if (isEncryptedFile) {
                "${FileUtils.secretFolderPath}/$messageId/${neDocument.name}"
            } else {
                "${FileUtils.downloadFolderPath}/${messageId}_${neDocument.name}"
            }
            if (com.netacom.base.chat.android_utils.FileUtils.isFileExists(downloadFilePath)) {
                withContext(getPostExecutionThread.main) {
                    FileUtils.openFile(
                        context = context,
                        file = File(downloadFilePath),
                        callback = {
                            screenStateError.postErrorOnce(R.string.not_open_file)
                        }
                    )
                }
            } else {
                downloadRepository.downloadFiles(arrayListOf(fileUrl.getUrlImage())).collect {
                    when (it) {
                        is ApiResponseSuccess -> {
                            val responseDownload = it.data
                            responseDownload?.let { responseBody ->
                                if (isEncryptedFile) {
                                    FileUtils.saveEncryptedFileToLocal(
                                        responseBody,
                                        "${FileUtils.secretFolderPath}/$messageId",
                                        neDocument.name ?: System.currentTimeMillis().toString()
                                    ).let { localPath ->
                                        withContext(getPostExecutionThread.main) {
                                            handleOpenFileBecauseActivityNotFound(context, localPath, neDocument, responseBody)
                                        }
                                    }
                                } else {
                                    FileUtils.saveFileToLocal(responseBody, "${messageId}_${neDocument.name ?: System.currentTimeMillis().toString()}")
                                        .let { localPath ->
                                            withContext(getPostExecutionThread.main) {
                                                handleOpenFileBecauseActivityNotFound(context, localPath, neDocument, responseBody)
                                            }
                                        }
                                }
                            }
                        }
                        is ApiResponseError -> {
                            withContext(getPostExecutionThread.main) {
                                Toast.makeText(context, "Cannot open this file!", Toast.LENGTH_SHORT).show()
                            }
                        }
                    }
                }
            }
        }
    }

    private fun handleOpenFileBecauseActivityNotFound(
        context: Context,
        localPath: String,
        neDocument: NeDocument,
        responseBody: ResponseBody
    ) {
        FileUtils.openFile(
            context,
            File(localPath),
            callback = {
                // handle for not found activity for open file type
                DialogUtil.showMessage(
                    context,
                    title = R.string.popup_confirm,
                    message = R.string.popup_confirm_download,
                    okFunc = {
                        launchOnViewModelScope(getPostExecutionThread.io) {
                            FileUtils.saveFileToLocal(responseBody, neDocument.name ?: System.currentTimeMillis().toString(), true)
                        }
                    }
                )
            }
        )
    }

    fun startSecretChat(neUser: NeUser) {
        val ownerUin = getUserId
        val receiverUin = neUser.id ?: let {
            return
        }
        launchOnViewModelScope(getPostExecutionThread.io) {
            val publicKey = ChatSecretUtils.getPublicKey()
            val pIdentityKey = publicKey.pIdentityKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val pBaseKey = publicKey.pBaseKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val pOneTimePreKey = publicKey.pOneTimePreKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val startSecretMessageRequest = StartSecretMessageRequest(
                ownerUin,
                receiverUin,
                publicKey.pDeviceId ?: "",
                pIdentityKey ?: "",
                pBaseKey ?: "",
                pOneTimePreKey ?: ""
            )
            Logger.e("startSecretMessageRequest=$startSecretMessageRequest")
            getIOSocket.startSecretChat(
                startSecretMessageRequest
            ).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        launchOnViewModelScope(getPostExecutionThread.io) {
                            groupRepository.saveGroupToDB(it)?.apply {
                                openChat(groupRepository.convertToNeGroup(it))
//                                syncGroup()
                            }
                        }
                    }
                    is ResultData.Failed -> {
                        screenStateError.postErrorOnce(result.toString())
                    }
                    else -> screenStateError.postErrorOnce(null)
                }
            }
        }
    }

    fun acceptSecretChat(groupId: Long?, fromNotification: Boolean, callbackResult: CallbackResult<NeGroup>? = null) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val publicKey = ChatSecretUtils.getPublicKey()
            val pIdentityKey = publicKey.pIdentityKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val pBaseKey = publicKey.pBaseKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val pOneTimePreKey = publicKey.pOneTimePreKey?.let {
                it.run {
                    val result = if (this.endsWith("\n")) this.subSequence(0, this.length - 1).toString() else this
                    result
                }
            }

            val acceptSecretMessageRequest = AcceptSecretMessageRequest(
                getUserId,
                groupId,
                publicKey.pDeviceId ?: "",
                pIdentityKey ?: "",
                pBaseKey ?: "",
                pOneTimePreKey ?: ""
            )
            Logger.e("acceptSecretMessageRequest=$acceptSecretMessageRequest")
            getIOSocket.acceptSecretChat(
                acceptSecretMessageRequest
            ).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        groupRepository.saveGroupToDB(it)?.apply {
                            if (fromNotification) {
                                callbackResult?.callBackSuccess(this)
                            }
                            syncGroup()
                        }
                    }
                    is ResultData.Failed -> {
                        screenStateError.postErrorOnce(result.toString())
                    }
                    else -> screenStateError.postErrorOnce(null)
                }
            }
        }
    }

    private suspend fun executeStartSecretChatEvent() {
        getIOSocket.flowStartSecretChat.collect {
            it?.let {
                acceptSecretChat(it.groupId, false)
                getIOSocket.flowStartSecretChat.value = null
            }
        }
    }

    private suspend fun executeAcceptSecretChatEvent() {
        getIOSocket.flowAcceptSecretChat.collect {
            it?.let {
                val groupId = it.groupId.toString()
                ChatSecretUtils.secretAcceptHashMap[groupId] = it
                groupRepository.getGroupDbById(groupId)?.let { dbGroupEntity ->
                    groupRepository.updateSecretGroup(dbGroupEntity, it).also { updateDbGroupEntity ->
                        initSecretChat(groupRepository.convertDBToNeGroup(updateDbGroupEntity), null)
                    }
                }
                getIOSocket.flowAcceptSecretChat.value = null
            }
        }
    }

    private suspend fun executeDeleteSecretChatEvent() {
        getIOSocket.flowDeleteSecretChat.collect {
            it?.let {
                launchOnViewModelScope(getPostExecutionThread.io) {
                    // Show local notification
                    val dbGroup = groupRepository.getGroupDbById(it.groupId.toString())
                    if (isShowSecretChatLocalNotification(dbGroup)) {
                        val fcmNotificationJson = jsonSerializer.asJson(
                            FcmNotification().apply {
                                type = NOTIFICATION_TYPE_DELETE_SECRET
                                group_id = it.groupId
                                group_type = GroupType.GROUP_TYPE_PRIVATE
                                mSenderUin = dbGroup?.groupOwnerId ?: 0L
                                image = dbGroup?.getDisplayAvatar ?: ""
                                isFcm = false
                                title = getDbManager.getPartnerInfoById(mSenderUin)?.getName ?: StringUtils.getString(com.netacom.lite.R.string.unknown_user)
                                name = title
                                content = SpanUtils().append(StringUtils.getString(com.netacom.lite.R.string.fcm_delete_secret_chat)).create().toString()
                            },
                            FcmNotification::class.java
                        )
                        WorkManager.getInstance(Utils.getApp()).enqueueOneTimeNetworkWorkRequest<MessageWorker>(
                            workDataOf(FirebaseDefine.KEY_DATA to fcmNotificationJson)
                        )
                    }
                    groupRepository.deleteEncryptedFilesInGroup(it.toString())
                }
                deleteSecretGroup.post(it.groupId)
                getIOSocket.flowDeleteSecretChat.value = null
            }
        }
    }

    fun initSecretChat(neGroup: NeGroup?, callbackResult: CallbackResult<NeGroup?>?) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            if (neGroup?.partnerPublicKey == null && ChatSecretUtils.secretAcceptHashMap.containsKey(neGroup?.id)) {
                ChatSecretUtils.secretAcceptHashMap[neGroup?.id]?.let {
                    neGroup?.partnerPublicKey = NePublicKey().apply {
                        this.pUin = it.uin
                        this.pBaseKey = it.base_key
                        this.pIdentityKey = it.identity_key
                        this.pDeviceId = it.device_id
                        this.pOneTimePreKey = it.one_time_pre_key
                    }
                }
            }
            if (neGroup?.partnerPublicKey != null) {
                ChatSecretUtils.init(neGroup, isAlice = neGroup.owner?.id == getUserId)
                _acceptSecretChat.post(neGroup)
                callbackResult?.callBackSuccess(neGroup)
            } else {
                _acceptSecretChat.post(null)
                callbackResult?.callBackSuccess(null)
            }
        }
    }

    // Call
    private suspend fun socketUpdateStartCall() {
        getIOSocket.flowStartCall.collect { resultData ->
            when (resultData) {
                is ResultData.Success -> {
                    if (isShowLog) Logger.e("socketUpdateStartCall=${resultData.data}")
                    resultData.data?.let { call ->
                        handleReceiveCall(call = call)
                    }
                }
                else -> Unit
            }
        }
    }

    private suspend fun handleReceiveCall(call: Call, _fromFcm: Boolean = false) = withContext(getPostExecutionThread.io) {
        val checkCallRequest = CheckCallRequest(
            groupId = call.groupId.toLongOrNull(),
            callId = call.callId.toLongOrNull()
        )
        Logger.e("callBackSuccess=$call")
        getIOSocket.checkCall(
            checkCallRequest = checkCallRequest,
            callbackResult =
            object : CallbackResult<Int> {
                override fun callBackSuccess(result: Int) {
                    if (result == CallDefine.CALL_STATUS_ONGOING) {
                        CallingUtils.fromFcm = _fromFcm
                        if (CallingUtils.isOnCall) {
                            val eventCallBusy = EventRequest(
                                groupId = call.groupId,
                                receiverUin = call.mCallerUin.toLongOrNull(),
                                type = EventType.EVENT_TYPE_CALL_BUSY.value,
                                senderUin = getPreferences.getUserId
                            )
                            Logger.e("callBackSuccess=eventCallBusy")
                            getIOSocket.sendEvent(eventCallBusy)
                        } else {
                            Logger.e("callBackSuccess=CallingUtils.isOnCall")
                            CallingUtils.isOnCall = true
                            mainViewModelCall = call
                            NetAloSDK.openNetAlo(context = Utils.getApp().applicationContext, call = mainViewModelCall)
                        }
                    }
                }

                override fun callBackError(error: String?) {
                    if (isShowLog) Logger.d("Call error!!!")
                }
            }
        )
    }

    fun logout(callbackSuccess: () -> Unit?) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            ChatSecretUtils.secretAcceptHashMap.clear()
            stickerRepository.isRunning = false
            authRepository.logout().collect {
                if (it is ApiResponseSuccess) {
                    Logger.e("logout:callbackSuccess")
                } else {
                    screenStateError.postErrorOnce(it.message)
                }
                socketRepository.unregisterFcm(
                    uin = getPreferences.getUserId,
                    token = getPreferences.token
                )
                socketRepository.clearData()
                socketRepository.disConnectSocket()
                getPreferences.userIgnoreOptionUpdate = false
                withContext(getPostExecutionThread.main) {
                    callbackSuccess()
                }
            }
        }
    }

    // DeepLink
    private fun initGroupLink() {
        launchOnViewModelScope(getPostExecutionThread.io) {
            getIOSocket.flowState.collect { state ->
                if (state == SocketState.LOGIN_SUCCESS) {
                    getPreferences.deepLinkToken.value?.let { deepLinkToken ->
                        joinGroupWithDeepLink(deepLinkToken)
                        getPreferences.setDeepLinkToken(null)
                    }
                }
            }
        }
    }

    fun createDeepLink(groupId: Long, callbackResult: CallbackResult<String?>? = null) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val createLinkGroupRequest = CreateDeepLinkRequest(groupId)
            getIOSocket.createDeepLink(createLinkGroupRequest).let { resultData ->
                when (resultData) {
                    is ResultData.Success -> {
                        withContext(getPostExecutionThread.main) {
                            callbackResult?.callBackSuccess(resultData.data)
                        }
                        if (callbackResult == null) _createDeepLink.post(resultData.data)
                    }
                    else -> if (callbackResult == null) _createDeepLink.post("")
                }
            }
        }
    }

    fun getDeepLink(groupId: Long, callbackResult: CallbackResult<String?>? = null) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val listLinkGroupRequest = ListLinkGroupRequest(group_id = groupId, pindex = 0, psize = 1)
            getIOSocket.getDeepLink(listLinkGroupRequest).let { resultData ->
                when (resultData) {
                    is ResultData.Success -> {
                        withContext(getPostExecutionThread.main) {
                            callbackResult?.callBackSuccess(resultData.data)
                        }
                        if (callbackResult == null) _createDeepLink.post(resultData.data)
                    }
                    else -> if (callbackResult == null) _createDeepLink.post("")
                }
            }
        }
    }

    fun disableDeepLink(token: String) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val disableDeepLinkRequest = DisableDeepLinkRequest(token = token)
            getIOSocket.disableDeepLink(disableDeepLinkRequest).let { resultData ->
                when (resultData) {
                    is ResultData.Success -> {
                        _disableDeepLink.post(EventLiveData(true))
                    }
                    else -> _disableDeepLink.post(EventLiveData(false))
                }
            }
        }
    }

    fun joinGroupWithDeepLink(token: String) {
        Logger.e("___preferencesHelperImpl.deepLinkToken 33333")
        launchOnViewModelScope(getPostExecutionThread.io) {
            val joinGroupWithLinkRequest = JoinGroupWithLinkRequest(token = token)
            getIOSocket.joinGroupWithDeepLink(joinGroupWithLinkRequest).let { resultData ->
                when (resultData) {
                    is ResultData.Success -> {
                        resultData.data?.let {
                            groupRepository.saveGroupToDB(it)?.apply {
                                Logger.e("getDeepLinkTokenFlow==$this")
                                if (checkGroupExitsInDb(this)) {
                                    _joinGroupWithDeepLink.post(EventLiveData(this))
                                } else {
                                    _joinGroupWithDeepLink.post(EventLiveData(null))
                                }
                            } ?: _joinGroupWithDeepLink.post(EventLiveData(null))
                        } ?: _joinGroupWithDeepLink.post(EventLiveData(null))
                    }
                    else -> _joinGroupWithDeepLink.post(EventLiveData(null))
                }
            }
        }
    }

    var isProcessing = false
    var cacheListMessages: MutableList<NeMessage> = emptyList<NeMessage>().toMutableList()
    private fun cacheMessages(listGroup: List<NeGroup>) {
        if (!isProcessing) {
            cacheListMessages.clear()
            launchOnViewModelScope(getPostExecutionThread.io) {
                isProcessing = true
                if (listGroup.size >= 15) {
                    listGroup.subList(0, 9)
                } else {
                    listGroup
                }.forEach { neGroup ->
                    neGroup.id?.let { groupId ->
                        cacheListMessages.addAll(groupRepository.getMessageByCreatedAt(groupId = groupId).filterNot { it.status == MessageStatusType.MESSAGE_DELETED })
                    }
                }
                isProcessing = false
            }
        }
    }
}
