/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.call

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.navigation.NavController
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.model.ScreenStateObj
import com.netacom.base.chat.util.DateTimeUtils
import com.netacom.base.chat.util.isNotNull
import com.netacom.base.chat.util.isNull
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKViewModel
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.full.model.CallInfoModel
import com.netacom.full.ui.main.chat.BaseChatFragmentDirections
import com.netacom.lite.entity.ui.call.NeCallLog
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.network.model.request.CallLogDeleteRequest
import com.netacom.lite.network.model.request.CallLogItemByUserRequest
import com.netacom.lite.repository.CallLogRepository
import com.netacom.lite.repository.CallingRepository
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.repository.base.State
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.suspendCancellableCoroutine
import java.util.Calendar
import java.util.Calendar.DAY_OF_YEAR
import java.util.Calendar.YEAR
import java.util.Calendar.getInstance
import javax.inject.Inject
import kotlin.collections.ArrayList

/**
Created by vietnguyen on 5/Oct/2020
Company: Netacom.
Email: vietnt@netacom.vn
 **/
@ExperimentalCoroutinesApi
@HiltViewModel
class CallLogViewModel @Inject constructor(
    socketRepository: SocketRepository,
    private val callLogRepository: CallLogRepository,
    private val groupRepository: GroupRepository,
    private val callingRepository: CallingRepository,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseSDKViewModel(socketRepository, groupRepository, navigationDispatcher) {
    val callLogScreenState: LiveData<ScreenStateObj> = MutableLiveData()
    private var groupNum: Long = -1
    private var listFullLogs: List<NeCallLog> = arrayListOf()

    private val _groupCallLogs = MutableLiveData<EventLiveData<List<NeCallLog>>>()
    val groupCallLogs: LiveData<EventLiveData<List<NeCallLog>>> = _groupCallLogs

    private val _callLogsByUser = MutableLiveData<EventLiveData<List<NeCallLog>>>()
    val callLogsByUser: LiveData<EventLiveData<List<NeCallLog>>> = _callLogsByUser

    var cacheCallLogs: Pair<Int, List<NeCallLog>?> = Pair(-1, null)

    fun getCallLogs(isMissCall: Boolean = false) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            callLogRepository.getCallLogs().collect { state ->
                groupNum = -1
                when (state) {
                    is State.SuccessLocal -> {
                        val listCallLogs = mutableListOf<NeCallLog>()
                        groupCallLogs(updateNeCallLogs(state.data)).map {
                            if (isMissCall) {
                                if (it.isMissCall && it.callerId?.toLongOrNull() != getUserId) {
                                    listCallLogs.add(it)
                                } else {
                                }
                            } else {
                                listCallLogs.add(it)
                            }
                        }
                        if (listCallLogs.isNull) {
                            callLogScreenState.postEmpty()
                        } else {
                            _groupCallLogs.post(EventLiveData(listCallLogs))
                            callLogScreenState.postData()
                        }
                    }
                    is State.SuccessMerged -> {
                        val listCallLogs = mutableListOf<NeCallLog>()
                        listFullLogs = updateNeCallLogs(state.data)
                        groupCallLogs(listFullLogs).map {
                            if (isMissCall) {
                                if (it.isMissCall && it.callerId?.toLongOrNull() != getUserId) {
                                    listCallLogs.add(it)
                                } else {
                                }
                            } else {
                                listCallLogs.add(it)
                            }
                        }
                        if (listCallLogs.isNull) {
                            callLogScreenState.postEmpty()
                        } else {
                            _groupCallLogs.post(EventLiveData(listCallLogs))
                            callLogScreenState.postData()
                        }
                    }
                    is State.Loading -> callLogScreenState.postDoing()
                    is State.Error -> callLogScreenState.postError(state.message)
                }
            }
        }
    }

    fun getCallLogsByUser(relateUid: String, millisCallLog: Long) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val callLogItemRequest = CallLogItemByUserRequest(
                relateUid = relateUid
            )
            callLogRepository.getCallLogsByUser(callLogItemRequest).collect { state ->
                when (state) {
                    is State.SuccessLocal,
                    is State.SuccessMerged -> {
                        val listCallLogs = when (state) {
                            is State.SuccessMerged -> state.data
                            is State.SuccessLocal -> state.data
                            else -> emptyList()
                        }
                        if (listCallLogs.isNotNull) {
                            _callLogsByUser.post(
                                EventLiveData(
                                    getCallLogOfUserFollowDate(
                                        updateNeCallLogs(listCallLogs),
                                        millisCallLog
                                    )
                                )
                            )
                            callLogScreenState.postData()
                        } else {
                            callLogScreenState.postEmpty()
                        }
                    }
                    is State.Loading -> callLogScreenState.postDoing()
                    is State.Error -> callLogScreenState.postError(state.message)
                }
            }
        }
    }

    fun deletedCallLogs(entry: NeCallLog, isMissCall: Boolean) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val idList = listFullLogs.filter {
                DateTimeUtils.parseCalenderFromString(it.createAt ?: 0L)?.get(DAY_OF_YEAR + YEAR) == DateTimeUtils.parseCalenderFromString(entry.createAt ?: 0L)
                    ?.get(DAY_OF_YEAR + YEAR) && it.groupNum == entry.groupNum && it.getPartnerName == entry.getPartnerName
            }.map { it.callId ?: "" }
            var ids = ""
            idList.indices.forEach {
                ids += if (it == idList.size - 1) {
                    idList[it]
                } else {
                    "${idList[it]},"
                }
            }
            if (ids.isNotEmpty()) {
                val callLogDeleteRequest = CallLogDeleteRequest(callIds = ids)
                callLogRepository.deleteCallLog(callLogDeleteRequest).collect { state ->
                    when (state) {
                        is State.SuccessLocal, is State.SuccessMerged -> {
                            val listCallDb = when (state) {
                                is State.SuccessMerged -> state.data
                                is State.SuccessLocal -> state.data
                                else -> emptyList()
                            }
                            val listCallLogs = mutableListOf<NeCallLog>()
                            groupNum = -1
                            listFullLogs = updateNeCallLogs(listCallDb)
                            groupCallLogs(listFullLogs).map {
                                if (isMissCall) {
                                    if (it.isMissCall && it.callerId?.toLongOrNull() != getUserId) {
                                        listCallLogs.add(it)
                                    } else {
                                    }
                                } else {
                                    listCallLogs.add(it)
                                }
                            }
                            _groupCallLogs.post(EventLiveData(listCallLogs))
                            if (listCallLogs.isNotEmpty()) {
                                callLogScreenState.postData()
                            } else {
                                callLogScreenState.postEmpty()
                            }
                        }
                        is State.Loading -> callLogScreenState.postDoing()
                        is State.Error -> callLogScreenState.postError(state.message)
                    }
                }
            } else {
                callLogScreenState.postError(StringUtils.getString(R.string.error_server_msg))
            }
        }
    }

    fun gotoNewCallDialog() {
        navigationDispatcher.emit {
            it.navigateIfSafe(CallLogCameraFragmentDirections.newCall())
        }
    }

    fun makeCall(_neGroup: NeGroup, isVideoEnable: Boolean) {
        navigationDispatcher.emit {
            it.navigateIfSafe(NewCallDialogDirections.makeCall(_neGroup, isVideoEnable))
        }
    }

    fun gotoChatInfoDialog(entry: NeCallLog, callbackError: () -> Unit) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            if (entry.groupId != null && entry.groupId?.isNotBlank() == true) {
                entry.groupId?.let {
                    groupRepository.getGroupDbById(it)?.let { group ->
                        groupRepository.convertDBToNeGroup(group)?.apply {
                            var partnerId: Long? = null
                            entry.callerId?.let { id ->
                                if (id.toLongOrNull() != getUserId) {
                                    partnerId = id.toLongOrNull()
                                } else {
                                    entry.calleeIds?.let { callIds ->
                                        callIds.filterNot { key -> key.toLongOrNull() == getUserId }
                                            .firstOrNull()?.let { findId ->
                                                partnerId = findId.toLongOrNull()
                                            }
                                    }
                                }
                            }

                            if (partnerId != null) {
                                partnerId?.let { pId ->
                                    callingRepository.getPartnerInfoById(pId)
                                        ?.let { it1 -> members = listOf(it1) }
                                    navigationDispatcher.emit { nav ->
                                        nav.navigateIfSafe(
                                            BaseChatFragmentDirections.openInfo(
                                                CallInfoModel(
                                                    this,
                                                    entry.isMissCall,
                                                    true,
                                                    entry.createAt ?: 0L
                                                )
                                            )
                                        )
                                    }
                                }
                            } else {
                                callbackError()
                            }
                        }
                    }
                }
            } else {
                callbackError()
            }
        }
    }

    private fun updateNeCallLogs(callLogs: List<NeCallLog>): List<NeCallLog> {
        val partnerIdSet = ArrayList<Long>()
        callLogs.forEach {
            it.calleeIds?.forEach { id ->
                partnerIdSet.add(id.toLongOrNull() ?: 0L)
            }
        }
        val userList = callingRepository.getListPartnerInfoById(partnerIdSet.toTypedArray())
        val myId = getUserId

        val result = mutableListOf<NeCallLog>()
        callLogs.forEach { callLog ->
            val partnerId = callLog.run {
                var id = 0L
                this.calleeIds?.let {
                    it.forEach lit@{ it1 ->
                        if (it1.toLongOrNull() != myId) {
                            id = it1.toLongOrNull() ?: 0L
                            return@lit
                        }
                    }
                }
                id
            }
            var partnerName: String? = null
            var partnerAvatar: String? = null
            userList?.firstOrNull { partnerId == it.id }?.let { user ->
                partnerName = user.getDisplayName
                partnerAvatar = user.avatar
            }
            val isMissCall = callLog.acceptIds?.size != callLog.calleeIds?.size
            result.add(
                NeCallLog(
                    callId = callLog.callId,
                    groupId = callLog.groupId,
                    mediaType = callLog.mediaType,
                    callerId = callLog.callerId,
                    calleeIds = callLog.calleeIds,
                    acceptIds = callLog.acceptIds,
                    startAt = callLog.startAt,
                    stopAt = callLog.stopAt,
                    acceptAt = callLog.acceptAt,
                    createAt = callLog.createAt,
                    connectedAt = callLog.connectedAt,
                    partnerName = partnerName,
                    partnerAvatar = partnerAvatar,
                    isMissCall = isMissCall
                )
            )
        }

        return result
    }

    // group list by date
    private suspend fun groupCallLogs(allCallLogs: List<NeCallLog>): MutableList<NeCallLog> = suspendCancellableCoroutine {
        val groupCallLogs = mutableListOf<NeCallLog>()
        val groupByDate = allCallLogs.groupBy { DateTimeUtils.parseCalenderFromString(it.createAt ?: 0L)?.get(DAY_OF_YEAR + YEAR) }
        if (groupByDate.isNotEmpty()) {
            groupByDate.keys.forEach { key ->
                groupByDate[key]?.let { callLogs ->
                    groupCallLogs.addAll(
                        groupByNameAndCallType(
                            callLogs
                        )
                    )
                }
            }
        }
        it.resume(groupCallLogs) {
            it.printStackTrace()
        }
    }

    // group list by name and call type
    private fun groupByNameAndCallType(listForGroup: List<NeCallLog>): List<NeCallLog> {
        val groupCallLogs = mutableListOf<NeCallLog>()
        var preCallLog: NeCallLog? = null
        listForGroup.forEach {
            if (it.partnerName == preCallLog?.partnerName) {
                if (it.isMissCall != preCallLog?.isMissCall) {
                    groupNum += 1
                    groupCallLogs.add(it)
                    preCallLog = it
                }
            } else {
                groupNum += 1
                groupCallLogs.add(it)
                preCallLog = it
            }
            it.groupNum = groupNum
        }

        return groupCallLogs
    }

    private fun getCallLogOfUserFollowDate(callLogs: List<NeCallLog>, millis: Long): List<NeCallLog> {
        val calendar: Calendar = getInstance()
        calendar.timeInMillis = millis
        return callLogs.filter { DateTimeUtils.parseCalenderFromString(it.createAt ?: 0L)?.get(DAY_OF_YEAR + YEAR) == calendar.get(DAY_OF_YEAR + YEAR) }
    }
}
