package com.netacom.full.ui.main.call.adapter

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.recyclerview.widget.DiffUtil
import com.netacom.base.chat.adapter.BaseListAdapter
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.full.R
import com.netacom.full.databinding.ItemRcvCallLogsDetailBinding
import com.netacom.full.databinding.RowCallLogBinding
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.lite.entity.ui.call.NeCallLog
import com.netacom.lite.entity.ui.call.NeCallLogItem
import com.netacom.lite.entity.ui.user.NeUser

class CallLogAdapter(
    private var user: NeUser?,
    private val languageApp: String,
    private val isCallLogDetail: Boolean = false,
    private val themeHelperImpl: ThemeHelperImpl,
    private val infoClick: ((NeCallLog) -> Unit),
    private val removeClick: ((NeCallLog) -> Unit),
    itemClick: ((NeCallLog, View, Int) -> Unit)?
) : BaseListAdapter<NeCallLog, ViewDataBinding>(itemClick = itemClick, diffCallback = CallLogDiff()) {

    override fun onBindViewHolder(holder: BaseViewHolder<ViewDataBinding>, position: Int) {
        val item = getItem(position)
        if (isCallLogDetail) {
            bindCallLogsDetail(holder, logCall = item, neLogCallItem = NeCallLogItem(item, user))
        } else {
            bindCallLogs(holder, logCall = item, position = position, neLogCallItem = NeCallLogItem(item, user))
        }
        holder.binding.executePendingBindings()
    }

    override fun getItemId(position: Int): Long {
        return position.toLong()
    }

    override fun getItemViewType(position: Int): Int {
        return position
    }

    private fun bindCallLogsDetail(
        holder: BaseViewHolder<ViewDataBinding>,
        logCall: NeCallLog?,
        neLogCallItem: NeCallLogItem
    ) {
        with(holder.binding as ItemRcvCallLogsDetailBinding) {
            item = logCall
            callLogItem = neLogCallItem
            isCaller = user?.id.toString() == callLogItem?.callLog?.callerId
        }
    }

    private fun bindCallLogs(
        holder: BaseViewHolder<ViewDataBinding>,
        logCall: NeCallLog?,
        neLogCallItem: NeCallLogItem,
        position: Int
    ) {
        if (logCall == null) return
        with(holder.binding as RowCallLogBinding) {
            item = logCall
            callLogItem = neLogCallItem
            language = languageApp
            callInfo.clickDebounce {
                infoClick.invoke(logCall)
            }
            actionRemove.clickDebounce {
                if (swipeLayout.isOpen()) {
                    removeClick.invoke(logCall)
                }
            }
            layoutCallLogs.clickDebounce {
                itemClick?.invoke(logCall, holder.itemView, position)
            }
            themeHelperImpl.setThemeColorForViews(callInfo)
        }
    }

    override fun onCreateViewHolder(
        parent: ViewGroup,
        viewType: Int
    ): BaseViewHolder<ViewDataBinding> {
        return if (isCallLogDetail) {
            BaseViewHolder(
                DataBindingUtil.inflate(
                    LayoutInflater.from(parent.context),
                    R.layout.item_rcv_call_logs_detail,
                    parent,
                    false
                ) as ItemRcvCallLogsDetailBinding
            )
        } else {
            BaseViewHolder(
                DataBindingUtil.inflate(
                    LayoutInflater.from(parent.context),
                    R.layout.row_call_log,
                    parent,
                    false
                ) as RowCallLogBinding
            )
        }
    }

    private class CallLogDiff : DiffUtil.ItemCallback<NeCallLog>() {
        override fun areItemsTheSame(oldItem: NeCallLog, newItem: NeCallLog): Boolean {
            return oldItem.callId == newItem.callId
        }

        override fun areContentsTheSame(oldItem: NeCallLog, newItem: NeCallLog): Boolean {
            return oldItem.getPartnerAvatar == newItem.getPartnerAvatar && oldItem.getPartnerName == newItem.getPartnerName
        }
    }
}
