/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */
package com.netacom.full.ui.main.calling

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.navigation.NavController
import com.netacom.base.chat.app.isShowLog
import com.netacom.base.chat.base.BaseViewModel
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.lite.define.CallDefine.MEDIA_TYPE_AUDIO
import com.netacom.lite.define.CallDefine.MEDIA_TYPE_VIDEO
import com.netacom.lite.entity.socket.AnswerCall
import com.netacom.lite.entity.socket.Call
import com.netacom.lite.entity.socket.EndCall
import com.netacom.lite.entity.socket.Event
import com.netacom.lite.entity.socket.EventType
import com.netacom.lite.entity.socket.IceSdpType
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.repository.CallingRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.socket.request.AnswerCallRequest
import com.netacom.lite.socket.request.EventCallRequest
import com.netacom.lite.socket.request.EventRequest
import com.netacom.lite.socket.request.IceCandidateRequest
import com.netacom.lite.socket.request.IceSdpRequest
import com.netacom.lite.socket.request.RequestEvent
import com.netacom.lite.socket.request.StopCallRequest
import com.netacom.lite.socket.response.IceCandidateResponse
import com.netacom.lite.socket.response.IceSdpResponse
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import com.netacom.lite.webRTC.common.PeerConnectionClient
import com.netacom.lite.webRTC.common.callback.SwitchCameraCallback
import com.netacom.lite.webRTC.service.income.IncomeService
import com.netacom.lite.webRTC.service.outcome.OutcomeService
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import org.webrtc.IceCandidate
import org.webrtc.SessionDescription
import org.webrtc.StatsReport
import org.webrtc.SurfaceViewRenderer
import javax.inject.Inject

/**
 * Created by Tam Nguyen on 9/9/20.
 */
@HiltViewModel
class CallingViewModel @Inject constructor(
    private val socketRepository: SocketRepository,
    private val callingRepository: CallingRepository,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseViewModel() {

    companion object {
        const val TAG = "_CallingViewModel"
    }

    init {
        observeEvent()
    }

    private enum class CallStatus {
        INCOMING,
        CONNECTING,
        ENDING
    }

    private var _eventAnswerCall = MutableLiveData<EventLiveData<Boolean>>()
    val eventAnswerCall: LiveData<EventLiveData<Boolean>> = _eventAnswerCall

    private var _peerConnected = MutableLiveData<EventLiveData<Boolean>>()
    val peerConnected: LiveData<EventLiveData<Boolean>> = _peerConnected

    private var _peerDisconnected = MutableLiveData<EventLiveData<Boolean>>()
    val peerDisconnected: LiveData<EventLiveData<Boolean>> = _peerDisconnected

    private var _dataReceiveStop = MutableLiveData<EventLiveData<Boolean>>()
    val dataReceiveStop: LiveData<EventLiveData<Boolean>> = _dataReceiveStop

    private var _eventRequestCamera = MutableLiveData<EventLiveData<Boolean>>()
    val eventRequestCamera: LiveData<EventLiveData<Boolean>> = _eventRequestCamera

    private var _eventActiveCamera = MutableLiveData<EventLiveData<Boolean>>()
    val eventActiveCamera: LiveData<EventLiveData<Boolean>> = _eventActiveCamera

    private var _eventCallBusy = MutableLiveData<EventLiveData<Boolean>>()
    val eventCallBusy: LiveData<EventLiveData<Boolean>> = _eventCallBusy

    private var _eventReceivedEvent = MutableLiveData<EventLiveData<Boolean>>()
    val eventReceivedEvent: LiveData<EventLiveData<Boolean>> = _eventReceivedEvent

    private var call: Call? = null
    private var partnerUin: Long? = 0L
    private var status = CallStatus.INCOMING
    private var answerCall: AnswerCall? = null
    var isOutComeCalling = true

    private fun observeEvent() {
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            socketRepository.getIOSocket.flowAnswerCall.collect { value: AnswerCall? ->
                value?.let {
                    Logger.e("flowAnswerCall==$it")
                    handleAnswerCall(it)
                    socketRepository.getIOSocket.flowAnswerCall.value = null
                }
            }
        }
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            socketRepository.getIOSocket.flowIceSdp.collect { value: IceSdpResponse? ->
                value?.let {
                    Logger.e("handleIceSdp==$it")
                    handleIceSdp(it)
                    socketRepository.getIOSocket.flowIceSdp.value = null
                }
            }
        }
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            socketRepository.getIOSocket.flowIceCandidate.collect { value: IceCandidateResponse? ->
                value?.let {
                    Logger.e("handleIceCandidate==$it")
                    handleIceCandidate(it)
                    socketRepository.getIOSocket.flowIceCandidate.value = null
                }
            }
        }
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            socketRepository.getIOSocket.flowStopCall.collect { value: EndCall? ->
                value?.let {
                    Logger.e("handleStopCall==$it")
                    handleStopCall(it)
                    socketRepository.getIOSocket.flowStopCall.value = null
                }
            }
        }
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            socketRepository.getIOSocket.flowEvent.collect { value: Event? ->
                value?.let {
                    handleEvent(it)
                    /**
                     * Quick fix to avoid the call end suddenly.
                     * Will move all socket to the MainSdkViewModel & refactor this class.
                     */
                    socketRepository.getIOSocket.flowEvent.value = null
                }
            }
        }
    }

    private fun handleAnswerCall(_answerCall: AnswerCall) {
        if (call?.callId != _answerCall.callId) {
            Logger.e("Wrong call id, expected ${call?.callId} got=${_answerCall.callId}")
            return
        }

        this.answerCall = _answerCall
        _eventAnswerCall.post(EventLiveData(true))
    }

    private fun handleIceSdp(iceSdpResponse: IceSdpResponse) {
        if (isOutComeCalling) {
            if (iceSdpResponse.type == IceSdpType.SDP_TYPE_ANSWER.value) {
                val desc = SessionDescription(
                    SessionDescription.Type.fromCanonicalForm("answer"),
                    iceSdpResponse.sdp
                )
                OutcomeService.getInstance().setRemoteDescription(desc)
            }
        } else {
            if (iceSdpResponse.type == IceSdpType.SDP_TYPE_OFFER.value) {
                val desc = SessionDescription(
                    SessionDescription.Type.fromCanonicalForm("offer"),
                    iceSdpResponse.sdp
                )
                IncomeService.getInstance().setRemoteDescription(desc)
                IncomeService.getInstance().createAnswer()
            }
        }
    }

    private fun handleIceCandidate(iceCandidateResponse: IceCandidateResponse) {
        val webRtcCandidate = IceCandidate(
            iceCandidateResponse.sdpMid,
            iceCandidateResponse.sdpMlineIndex,
            iceCandidateResponse.candidate
        )
        if (isOutComeCalling) {
            OutcomeService.getInstance().addRemoteIceCandidate(webRtcCandidate)
        } else {
            IncomeService.getInstance().addRemoteIceCandidate(webRtcCandidate)
        }
    }

    private fun handleStopCall(stopCall: EndCall?) {
        if (stopCall?.uin?.toLongOrNull() == partnerUin && CallingUtils.isOnCall) {
            CallingUtils.stopCountdownTimeOut()
            _dataReceiveStop.post(EventLiveData(true))
        }
    }

    private fun handleEvent(event: Event) {
        Logger.e("handleEvent==$event")
        when (event.type) {
            EventType.EVENT_TYPE_CALL_REQUEST_VIDEO.value -> {
                if (call?.mMediaType == 1) {
                    call?.mMediaType = 2
                    _eventRequestCamera.post(EventLiveData(true))
                }
            }
            EventType.EVENT_TYPE_CALL_BUSY.value -> {
                CallingUtils.stopCountdownTimeOut()
                _eventCallBusy.post(EventLiveData(true))
            }
            EventType.EVENT_TYPE_CALL_RECEIVED_EVENT.value -> {
                _eventReceivedEvent.post(EventLiveData(true))
            }
            EventType.EVENT_TYPE_CALL_CHECKING_AVAILABLE.value -> {
                sendEventCallType(eventType = EventType.EVENT_TYPE_CALL_RESET.value)
            }
            EventType.EVENT_TYPE_CALL_INACTIVE_VIDEO.value -> {
                _eventActiveCamera.post(EventLiveData(false))
            }
            EventType.EVENT_TYPE_CALL_ACTIVE_VIDEO.value -> {
                _eventActiveCamera.post(EventLiveData(true))
            }
            EventType.EVENT_TYPE_CALL_RESET.value -> {
                /**
                 * Need to discuss with iOS
                 */
            }
            EventType.EVENT_TYPE_CALL_CONNECTING.value -> {
                if (status == CallStatus.INCOMING) {
                    status = CallStatus.ENDING
                    _dataReceiveStop.post(EventLiveData(true))
                } else {
                    /**
                     * Need to discuss with iOS
                     */
                }
            }
            else -> {
                // todo nothing
            }
        }
    }

    private fun sendEventCallType(eventType: Int) {
        partnerUin?.let { _partnerUin ->
            val eventRequest = EventRequest(
                groupId = call?.groupId,
                receiverUin = _partnerUin,
                type = eventType,
                senderUin = socketRepository.getUserId
            )
            socketRepository.getIOSocket.sendEvent(eventRequest)
        }
    }

    /**
     * From outcome
     * */
    fun makeCall(
        neGroup: NeGroup?,
        videoEnable: Boolean,
        callbackError: () -> Unit = {}
    ) {
        val groupId = neGroup?.id?.toLongOrNull() ?: let {
            callbackError()
            return
        }
        val memberIds = mutableListOf<Long>()
        neGroup.members?.forEach {
            it.id?.let { memberId ->
                if (memberId != socketRepository.getUserId) {
                    partnerUin = memberId
                }
                memberIds.add(memberId)
            }
        }
        OutcomeService.getInstance().initServer()
        socketRepository.getIOSocket.createCall(
            ownerId = socketRepository.getUserId,
            groupId = groupId,
            listPartnerId = memberIds,
            mediaType = if (videoEnable) MEDIA_TYPE_VIDEO else MEDIA_TYPE_AUDIO,
            callSuccess = {
                CallingUtils.isOnCall = true
                call = it
            },
            callFail = {
                callbackError()
            }
        )
        CallingUtils.startCountdownTimeOut(
            timeOut = Constants.RTC_OUTCOME_TIMEOUT,
            endTimeCallback = { _dataReceiveStop.post(EventLiveData(true)) }
        )
    }

    fun endCallOut() {
        launchOnViewModelScope(socketRepository.getPostExecutionThread.main) {
            CallingUtils.stopCountdownTimeOut()
            OutcomeService.getInstance().stopCall()
            val callId = call?.callId ?: return@launchOnViewModelScope
            val groupId = call?.groupId ?: return@launchOnViewModelScope

            val stopCallRequest = StopCallRequest(
                groupId = groupId,
                callId = callId,
                uin = socketRepository.getUserId
            )
            socketRepository.getIOSocket.stopCall(stopCallRequest)
            CallingUtils.isOnCall = false
            delay(1500)
            navigationDispatcher.emit {
                it.popBackStack()
            }
        }
    }

    fun initView(
        pipRenderer: SurfaceViewRenderer,
        fullscreenRenderer: SurfaceViewRenderer,
        isSwappedFeeds: Boolean,
        isVideoEnabled: Boolean
    ) {
        if (!isOutComeCalling) {
            status = CallStatus.CONNECTING
        }

        /**
         * Please don't modify this, we will merged OutComeService & IncomeService.
         */
        val callingService = if (isOutComeCalling) {
            OutcomeService.getInstance()
        } else {
            IncomeService.getInstance()
        }

        callingService.initView(
            pipRenderer = pipRenderer,
            fullscreenRenderer = fullscreenRenderer,
            isSwappedFeeds = isSwappedFeeds,
            isVideoEnabled = isVideoEnabled,
            event = object : PeerConnectionClient.PeerConnectionEvents {
                val groupId = call?.groupId
                val callId = call?.callId

                override fun onLocalDescription(sdp: SessionDescription?) {
                    if (isShowLog) Logger.t(TAG)?.d(
                        "initViewonLocalDescription groupId = $groupId" + " --- " +
                            callId + " --- " + partnerUin
                    )
                    if (isShowLog) Logger.d(
                        "initViewonLocalDescription isOutComeCalling = $isOutComeCalling"
                    )

                    partnerUin?.let { _partnerUin ->
                        val iceSdp = IceSdpRequest(
                            senderUin = socketRepository.getUserId,
                            groupId = groupId,
                            callId = callId,
                            receiverUin = _partnerUin,
                            type = if (isOutComeCalling) IceSdpType.SDP_TYPE_OFFER.value else IceSdpType.SDP_TYPE_ANSWER.value,
                            sdp = sdp?.description ?: ""
                        )
                        socketRepository.getIOSocket.sendIceSdp(iceSdp)
                    }
                }

                override fun onIceCandidate(candidate: IceCandidate?) {
                    candidate?.let {
                        if (isShowLog) Logger.t(TAG)?.d(
                            "initViewonIceCandidate groupId = $groupId" + " --- " +
                                callId + " --- " + partnerUin
                        )
                        if (isShowLog) Logger.d("initView onIceCandidate")
                        partnerUin?.let { _partnerUin ->
                            val iceCandidate = IceCandidateRequest(
                                groupId = groupId,
                                callId = callId,
                                receiverUin = _partnerUin,
                                candidate = it.sdp,
                                sdpMid = it.sdpMid,
                                sdpMlineIndex = it.sdpMLineIndex,
                                senderUin = socketRepository.getUserId
                            )
                            socketRepository.getIOSocket.sendIceCandidate(iceCandidate)
                        }
                    }
                }

                override fun onPeerConnectionStatsReady(reports: Array<StatsReport>?) {
                    if (isShowLog) Logger.d("initView onPeerConnectionStatsReady")
                }

                override fun onConnected() {
                    if (isShowLog) Logger.d("initView onConnected")
                    if (isSwappedFeeds) {
                        if (isOutComeCalling) {
                            OutcomeService.getInstance().setSwappedFeeds(false)
                        } else {
                            IncomeService.getInstance().setSwappedFeeds(false)
                        }
                    }
                    _peerConnected.post(EventLiveData(true))
                    CallingUtils.stopCountdownTimeOut()

                    if (isOutComeCalling) {
                        partnerUin?.let { _partnerUin ->
                            val requestConnecting = EventRequest(
                                groupId = groupId,
                                receiverUin = _partnerUin,
                                type = EventType.EVENT_TYPE_CALL_CONNECTING.value,
                                senderUin = socketRepository.getUserId
                            )
                            socketRepository.getIOSocket.sendEvent(requestConnecting)

                            val requestCallEvent = EventCallRequest(
                                groupId = groupId,
                                callId = callId,
                                event = 2
                            )
                            socketRepository.getIOSocket.sendEventCall(requestCallEvent)
                        }
                    }
                }

                override fun onDisconnected() {
                    if (isShowLog) Logger.d("initView onDisconnected")
                    _peerDisconnected.post(EventLiveData(true))
                }
            }
        )

        if (!isOutComeCalling && socketRepository.getIOSocket.isConnect()) {
            val answerCallRequest = AnswerCallRequest(
                groupId = call?.groupId,
                callId = call?.callId,
                calleeUin = socketRepository.getUserId,
                answer = 1
            )
            socketRepository.getIOSocket.sendAnswerCall(answerCallRequest)
            CallingUtils.startCountdownTimeOut(
                timeOut = Constants.RTC_INCOME_TIMEOUT,
                endTimeCallback = {
                    _dataReceiveStop.post(EventLiveData(true))
                }
            )
        }

        if (isOutComeCalling) {
            OutcomeService.getInstance().createOffer(isRetry = false)
        }
    }

    fun setEnableVideo(isEnable: Boolean) {
        if (call?.mMediaType == 1) {
            call?.mMediaType = 2
            val requestEnableVideo = RequestEvent(
                groupId = call?.groupId,
                type = EventType.EVENT_TYPE_CALL_REQUEST_VIDEO.value,
                senderUin = socketRepository.getUserId
            )
            socketRepository.getIOSocket.requestEvent(requestEnableVideo)
            if (isOutComeCalling) {
                OutcomeService.getInstance().enableVideoRemote(true)
            } else {
                IncomeService.getInstance().enableVideoRemote(true)
            }
        } else {
            val partnerId = if (isOutComeCalling) {
                partnerUin
            } else {
                call?.mCallerUin?.toLongOrNull()
            }
            partnerId?.let {
                val eventRequest = EventRequest(
                    groupId = call?.groupId,
                    type = if (isEnable) EventType.EVENT_TYPE_CALL_ACTIVE_VIDEO.value else EventType.EVENT_TYPE_CALL_INACTIVE_VIDEO.value,
                    senderUin = socketRepository.getUserId,
                    receiverUin = it
                )
                socketRepository.getIOSocket.sendEvent(eventRequest)
            }
        }

        if (isOutComeCalling) {
            OutcomeService.getInstance().enableVideoLocal(isEnable)
        } else {
            IncomeService.getInstance().enableVideoLocal(isEnable)
        }
    }

    fun setEnableSpeaker(isEnable: Boolean) {
        if (isOutComeCalling) {
            OutcomeService.getInstance().enableSpeaker(isEnable)
        } else {
            IncomeService.getInstance().enableSpeaker(isEnable)
        }
    }

    fun setEnableAudio(isEnable: Boolean) {
        if (isOutComeCalling) {
            OutcomeService.getInstance().enableAudio(isEnable)
        } else {
            IncomeService.getInstance().enableAudio(isEnable)
        }
    }

    fun switchCamera(switchCameraCallback: SwitchCameraCallback) {
        if (isOutComeCalling) {
            OutcomeService.getInstance().switchCamera(switchCameraCallback)
        } else {
            IncomeService.getInstance().switchCamera(switchCameraCallback)
        }
    }

    /**
     * From income
     * */
    fun setCallMode(call: Call) {
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            this@CallingViewModel.call = call
            this@CallingViewModel.partnerUin = call.mCallerUin.toLongOrNull()
            IncomeService.newInstance().initServer()
            val callReceived = EventRequest(
                groupId = call.groupId,
                receiverUin = call.mCallerUin.toLongOrNull(),
                type = EventType.EVENT_TYPE_CALL_RECEIVED_EVENT.value,
                senderUin = socketRepository.getUserId
            )

            val callChecking = EventRequest(
                groupId = call.groupId,
                receiverUin = call.mCallerUin.toLongOrNull(),
                type = EventType.EVENT_TYPE_CALL_CHECKING_AVAILABLE.value,
                senderUin = socketRepository.getUserId
            )
            socketRepository.getIOSocket.sendEvent(callReceived)
            socketRepository.getIOSocket.sendEvent(callChecking)
        }
    }

    fun getMemberById(partnerId: Long? = 0L, callback: CallbackResult<NeUser>) {
        launchOnViewModelScope(socketRepository.getPostExecutionThread.io) {
            callingRepository.getPartnerInfoById(partnerId)?.let {
                callback.callBackSuccess(it)
            } ?: callback.callBackError()
        }
    }

    fun endCallIn(isNotification: Boolean, callbackResult: () -> Unit) {
        launchOnViewModelScope(socketRepository.getPostExecutionThread.main) {
            CallingUtils.stopCountdownTimeOut()
            IncomeService.getInstance().stopCall()
            if (socketRepository.isConnect() && status != CallStatus.ENDING) {
                val stopCallRequest = StopCallRequest(
                    groupId = call?.groupId,
                    callId = call?.callId,
                    uin = socketRepository.getUserId
                )
                socketRepository.getIOSocket.stopCall(stopCallRequest)
            }
            CallingUtils.isOnCall = false
            delay(1500)
            navigationDispatcher.emit { nav ->
                isNotification.let { _isNotification ->
                    if (_isNotification) {
                        callbackResult()
                    } else {
                        nav.popBackStack()
                    }
                }
            }
        }
    }
}
