package com.netacom.full.ui.main.calling

import android.annotation.SuppressLint
import android.app.KeyguardManager
import android.content.Context.KEYGUARD_SERVICE
import android.content.Context.POWER_SERVICE
import android.os.Build
import android.os.Bundle
import android.os.PowerManager
import android.view.WindowManager
import androidx.activity.addCallback
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.lifecycle.lifecycleScope
import androidx.navigation.fragment.navArgs
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDataFragment
import com.netacom.full.databinding.FragmentCallBinding
import com.netacom.full.extensions.fadeIn
import com.netacom.full.extensions.fadeOut
import com.netacom.full.ui.sdk.NetAloActivity
import com.netacom.full.utils.DialogUtil
import com.netacom.full.widget.toolbar.DToolbarListener
import com.netacom.lite.define.CallDefine
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.webRTC.common.callback.SwitchCameraCallback
import com.netacom.lite.webRTC.service.income.IncomeService
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.webrtc.RendererCommon

/**
Created by vantoan on 02,NOV,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

@AndroidEntryPoint
class IncomeFragment : BaseSDKDataFragment<FragmentCallBinding, CallingViewModel>(
    R.layout.fragment_call,
    CallingViewModel::class.java
) {
    private var powerManager: PowerManager? = null
    private var wakeLock: PowerManager.WakeLock? = null
    private var isVideoEnable: Boolean = true
    private var isStartCalling = false
    private var isEnCall = false
    private val args: IncomeFragmentArgs by navArgs()

    override fun setViewModel(): Int = BR.viewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        CallingUtils.isOnCall = true
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.O_MR1) {
            requireActivity().setShowWhenLocked(true)
            requireActivity().setTurnScreenOn(true)
            val keyguardManager =
                requireActivity().getSystemService(KEYGUARD_SERVICE) as KeyguardManager?
            keyguardManager?.requestDismissKeyguard(requireActivity(), null)
        } else {
            @Suppress("DEPRECATION")
            requireActivity().window.addFlags(
                WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON or
                    WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED or
                    WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON or
                    WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD
            )
        }
    }

    @SuppressLint("InvalidWakeLockTag")
    override fun initViews() {
        powerManager = requireActivity().getSystemService(POWER_SERVICE) as PowerManager?
        wakeLock = powerManager?.newWakeLock(PowerManager.PROXIMITY_SCREEN_OFF_WAKE_LOCK, "mixin")
        viewModel.isOutComeCalling = false

        requireActivity().onBackPressedDispatcher.addCallback(this) {
            endCall()
        }
    }

    override fun initData() {
        args.call?.let {
            isVideoEnable = it.mMediaType == CallDefine.MEDIA_TYPE_VIDEO
            viewModel.setCallMode(it)
            viewModel.getMemberById(
                partnerId = it.mCallerUin.toLongOrNull(),
                callback = object : CallbackResult<NeUser> {
                    override fun callBackSuccess(result: NeUser) {
                        binding.remoteUsername = result.getDisplayName
                        binding.remoteAvatar = result.getDisplayAvatar
                    }

                    override fun callBackError(error: String?) {
                    }
                }
            )
        }

        if (isVideoEnable) binding.callToolbar.setVideoInMode()
        else binding.callToolbar.setAudioInMode()
        binding.pipRenderer.tag = "mirrorEnable"
        binding.pipRenderer.setMirror(true)
        binding.pipRenderer.setZOrderMediaOverlay(true)
        binding.pipRenderer.setEnableHardwareScaler(true)

        binding.fullscreenRenderer.setMirror(true)
        binding.fullscreenRenderer.setEnableHardwareScaler(false)
        binding.fullscreenRenderer.setScalingType(RendererCommon.ScalingType.SCALE_ASPECT_FILL)

        checkPermissionCamera(
            {
                RingtonePlayer.ringCallIncome(requireContext())
                RingtonePlayer.ringVibrator(requireContext())
            },
            {
                endCall()
            }
        )
    }

    override fun syncEvent() {
        viewModel.peerConnected.observeOnce {
            if (isVideoEnable) {
                binding.layoutVideo.isGone = true
                binding.switchCamera.fadeIn()
                binding.pipRenderer.fadeIn()
                binding.layoutUser.fadeOut(true)
                binding.callToolbar.showCamera(isShow = true, isCheck = true)
                binding.callToolbar.showSpeaker(isShow = true, isCheck = true)
            } else {
                binding.callToolbar.showCamera(isShow = true, isCheck = false)
                binding.callToolbar.showSpeaker(isShow = true, isCheck = false)
            }
            binding.callToolbar.showEndCall(isShow = true)
            binding.callToolbar.showAcceptCancel(isShow = false)
            binding.callToolbar.showMic(isShow = true, isCheck = true)

            viewModel.setEnableSpeaker(binding.callToolbar.isSpeakerOn())
            viewModel.setEnableAudio(binding.callToolbar.isAudioOn())
            CallingUtils.startCountdownStatus(binding.txtStatus, lifecycleScope)
            CallingUtils.startCountDownForToggle(binding.layoutButton, binding.switchCamera)

            CallingUtils.swipeCamera(
                view = binding.pipRenderer,
                widthScreen = binding.layoutCall.width.toFloat(),
                heightScreen = binding.layoutCall.height.toFloat()
            )
        }

        viewModel.eventRequestCamera.observeOnce {
            isVideoEnable = true
            binding.pipRenderer.fadeOut(true)
            IncomeService.getInstance().enableVideoRemote(true)
            DialogUtil.showMessage(
                context = requireContext(),
                message = R.string.request_open_camera,
                okLabel = R.string.popup_confirm_logout_ok,
                cancelLabel = R.string.popup_confirm_logout_no,
                okFunc = {
                    binding.callToolbar.showCamera(isShow = true, isCheck = true)
                    IncomeService.getInstance().enableVideoLocal(true)
                    binding.switchCamera.fadeIn()
                    binding.pipRenderer.fadeIn()
                    binding.layoutUser.fadeOut(true)
                    viewModel.setEnableVideo(isEnable = true)
                },
                cancelFunc = {
                },
                cancelable = false
            )
        }

        viewModel.eventActiveCamera.observeOnce {
            if (it) {
                IncomeService.getInstance().enableVideoRemote(true)
                binding.layoutVideo.isGone = true
                binding.layoutUser.fadeOut(true)
            } else {
                IncomeService.getInstance().enableVideoRemote(false)
                binding.layoutVideo.isVisible = true
                if (!binding.callToolbar.isCameraOn()) {
                    binding.layoutUser.fadeIn()
                }
            }
        }

        viewModel.dataReceiveStop.observeOnce {
            endCall()
        }

        viewModel.peerDisconnected.observeOnce {
            endCall()
        }

        binding.callToolbar.setDToolbarClickListener(
            object : DToolbarListener {
                override fun onAccept() {
                    isStartCalling = true
                    binding.txtStatus.text = resources.getString(R.string.connecting)
                    RingtonePlayer.stopCallIncome()
                    RingtonePlayer.stopVibrator()
                    viewModel.initView(
                        pipRenderer = binding.pipRenderer,
                        fullscreenRenderer = binding.fullscreenRenderer,
                        isSwappedFeeds = false,
                        isVideoEnabled = isVideoEnable
                    )
                }

                override fun onToggleCamera(isChecked: Boolean) {
                    lifecycleScope.launch {
                        if (!isVideoEnable) {
                            isVideoEnable = true
                        }
                        if (isChecked) {
                            binding.layoutUser.fadeOut(true)
                            delay(500)
                            binding.switchCamera.fadeIn()
                            binding.pipRenderer.fadeIn()
                        } else {
                            binding.pipRenderer.fadeOut(isGone = true)
                            if (binding.layoutVideo.isVisible) {
                                delay(500)
                                binding.layoutUser.fadeIn()
                            }
                        }
                        viewModel.setEnableVideo(isChecked)
                    }
                }

                override fun onToggleSpeaker(isChecked: Boolean) {
                    viewModel.setEnableSpeaker(isChecked)
                }

                override fun onToggleMicro(isChecked: Boolean) {
                    viewModel.setEnableAudio(isChecked)
                }

                override fun onHangUp() {
                    endCall()
                }
            }
        )

        binding.switchCamera.clickDebounce(
            listenerBlock = {
                viewModel.switchCamera(
                    switchCameraCallback = object : SwitchCameraCallback {
                        override fun switchDone(b: Boolean) {
                            MainScope().launch {
                                if (b) {
                                    when (binding.pipRenderer.tag) {
                                        "mirrorEnable" -> {
                                            binding.pipRenderer.setMirror(false)
                                            binding.pipRenderer.tag = "mirrorDisable"
                                        }
                                        "mirrorDisable" -> {
                                            binding.pipRenderer.setMirror(true)
                                            binding.pipRenderer.tag = "mirrorEnable"
                                        }
                                    }
                                }
                            }
                        }

                        override fun switchError(s: String?) {
                            showSnackBar(s)
                        }
                    }
                )
            }
        )

        binding.layoutCall.clickDebounce {
            if (binding.callToolbar.isVisible) {
                CallingUtils.hideToggleAnimation(binding.layoutButton, binding.switchCamera)
            } else {
                CallingUtils.showToggleAnimation(binding.layoutButton, binding.switchCamera, binding.pipRenderer.isVisible)
            }
        }
    }

    private fun endCall() {
        if (!isEnCall) {
            isEnCall = true
            binding.txtStatus.tag = resources.getString(R.string.end_call)
            RingtonePlayer.stopCallIncome()
            RingtonePlayer.stopVibrator()
            RingtonePlayer.ringEndCall(
                context = requireContext(),
                speakerEnable = binding.callToolbar.isSpeakerOn()
            )
            CallingUtils.stopCountdownStatus()
            binding.txtStatus.text = resources.getString(R.string.end_call)
            binding.pipRenderer.isGone = true
            binding.layoutUser.fadeIn()
            binding.callToolbar.fadeOut(true)
            viewModel.endCallIn(args.isNotification) {
                (activity as NetAloActivity).apply {
                    disConnectSocket()
                    finishAffinity()
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()
        CallingUtils.registerProximitySensor(
            context = requireContext(),
            nearDeviceCallback = {
                if (wakeLock?.isHeld == false) {
                    wakeLock?.acquire(10 * 60 * 1000L)
                }
            },
            farDeviceCallback = {
                if (wakeLock?.isHeld == true) {
                    wakeLock?.release()
                }
            }
        )
    }

    override fun onPause() {
        if (wakeLock?.isHeld == true) {
            wakeLock?.release()
        }
        super.onPause()
        CallingUtils.unregisterProximitySensor(context = requireContext())
    }

    override fun onStop() {
        super.onStop()
        if (powerManager?.isInteractive == false) {
            return
        }
    }

    override fun setupTheme() {
    }

    /**
     * This function support for Android devices have "Pop Up Camera" only.
     * Override this function to detect the camera pop-down.
     *
     * == TODO ==
     */

/*override fun cameraStatus(isPopDownCamera: Boolean) {
        if (isPopDownCamera && isStartCalling) {
            isStartCalling = false
            outCalling()
        }
    }*/
}
