/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */
package com.netacom.full.ui.main.calling

import android.annotation.SuppressLint
import android.content.Context
import android.os.PowerManager
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import androidx.navigation.fragment.navArgs
import com.netacom.base.chat.android_utils.NetworkUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDataFragment
import com.netacom.full.databinding.FragmentCallBinding
import com.netacom.full.extensions.fadeIn
import com.netacom.full.extensions.fadeOut
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.utils.DialogUtil
import com.netacom.full.widget.toolbar.DToolbarListener
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.webRTC.common.callback.SwitchCameraCallback
import com.netacom.lite.webRTC.service.outcome.OutcomeService
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.webrtc.RendererCommon

/**
 * Created by Tam Nguyen on 9/9/20.
 */

@AndroidEntryPoint
class OutComeFragment : BaseSDKDataFragment<FragmentCallBinding, CallingViewModel>(
    R.layout.fragment_call,
    CallingViewModel::class.java
) {

    private val args: OutComeFragmentArgs by navArgs()
    private var powerManager: PowerManager? = null
    private var wakeLock: PowerManager.WakeLock? = null
    private var neGroup: NeGroup? = null
    private var isVideoEnable: Boolean = true
    private var isEndCall = false

    override fun setViewModel(): Int = BR.viewModel

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()
    private var oldNetworkType: String? = null

    @SuppressLint("InvalidWakeLockTag")
    override fun initViews() {
        neGroup = args.neGroup
        isVideoEnable = args.isVideoEnable
        powerManager = requireActivity().getSystemService(Context.POWER_SERVICE) as PowerManager?
        wakeLock = powerManager?.newWakeLock(PowerManager.PROXIMITY_SCREEN_OFF_WAKE_LOCK, "mixin")
        viewModel.isOutComeCalling = true
        if (isVideoEnable) binding.callToolbar.setVideoOutMode()
        else binding.callToolbar.setAudioOutMode()

        binding.pipRenderer.tag = "mirrorEnable"
        binding.pipRenderer.setMirror(true)
        binding.pipRenderer.setZOrderMediaOverlay(false)
        binding.pipRenderer.setEnableHardwareScaler(true)

        binding.fullscreenRenderer.setMirror(false)
        binding.fullscreenRenderer.setEnableHardwareScaler(false)
        binding.fullscreenRenderer.setScalingType(RendererCommon.ScalingType.SCALE_ASPECT_FILL)
        oldNetworkType = NetworkUtils.getNetworkType().name
    }

    override fun initData() {
        neGroup?.members?.firstOrNull { _neUser ->
            _neUser.id != preferencesHelperImpl.getUserId
        }?.let {
            binding.remoteUsername = it.getDisplayName
            binding.remoteAvatar = it.getDisplayAvatar
        }
    }

    override fun syncEvent() {
        mainSdkViewModel.isConnectSocket.observe { isConnected ->
            preferencesHelperImpl.networkChange?.let { newNetworkType ->
                if (newNetworkType != oldNetworkType && isConnected) {
                    OutcomeService.getInstance().createOffer(isRetry = true)
                }
            }
        }

        viewModel.eventReceivedEvent.observeOnce {
            binding.txtStatus.text = resources.getString(R.string.ringing)
        }

        viewModel.eventAnswerCall.observeOnce {
            binding.txtStatus.text = resources.getString(R.string.connecting)
            viewModel.initView(
                pipRenderer = binding.pipRenderer,
                fullscreenRenderer = binding.fullscreenRenderer,
                isSwappedFeeds = false,
                isVideoEnabled = isVideoEnable
            )
            RingtonePlayer.stopCallOutcome()
        }

        viewModel.peerConnected.observeOnce {
            if (isVideoEnable) {
                binding.layoutVideo.isGone = true
                binding.switchCamera.fadeIn()
                binding.pipRenderer.fadeIn()
                binding.layoutUser.fadeOut(true)
            }
            binding.callToolbar.showCamera(isShow = true, isCheck = isVideoEnable)
            viewModel.setEnableSpeaker(binding.callToolbar.isSpeakerOn())
            viewModel.setEnableAudio(binding.callToolbar.isAudioOn())
            CallingUtils.startCountdownStatus(binding.txtStatus, lifecycleScope)
            CallingUtils.startCountDownForToggle(binding.layoutButton, binding.switchCamera)
            CallingUtils.swipeCamera(
                view = binding.pipRenderer,
                widthScreen = binding.layoutCall.width.toFloat(),
                heightScreen = binding.layoutCall.height.toFloat()
            )
        }

        viewModel.dataReceiveStop.observeOnce {
            endCall()
        }

        viewModel.peerDisconnected.observeOnce {
            endCall()
        }

        viewModel.eventCallBusy.observeOnce {
            endCall(status = resources.getString(R.string.user_busy))
        }

        viewModel.eventRequestCamera.observeOnce {
            isVideoEnable = true
            binding.pipRenderer.fadeOut(true)
            OutcomeService.getInstance().enableVideoRemote(true)
            DialogUtil.showMessage(
                context = requireContext(),
                message = R.string.request_open_camera,
                okLabel = R.string.popup_confirm_logout_ok,
                cancelLabel = R.string.popup_confirm_logout_no,
                okFunc = {
                    binding.callToolbar.showCamera(isShow = true, isCheck = true)
                    OutcomeService.getInstance().enableVideoLocal(true)
                    binding.switchCamera.fadeIn()
                    binding.pipRenderer.fadeIn()
                    binding.layoutUser.fadeOut(true)
                    viewModel.setEnableVideo(isEnable = true)
                },
                cancelFunc = {
                },
                cancelable = false
            )
        }

        viewModel.eventActiveCamera.observeOnce {
            if (it) {
                OutcomeService.getInstance().enableVideoRemote(true)
                binding.layoutVideo.isGone = true
                binding.layoutUser.fadeOut(true)
            } else {
                OutcomeService.getInstance().enableVideoRemote(false)
                binding.layoutVideo.isVisible = true
                if (!binding.callToolbar.isCameraOn()) {
                    binding.layoutUser.fadeIn()
                }
            }
        }

        binding.callToolbar.setDToolbarClickListener(
            object : DToolbarListener {
                override fun onToggleCamera(isChecked: Boolean) {
                    lifecycleScope.launch {
                        if (!isVideoEnable) {
                            isVideoEnable = true
                        }
                        if (isChecked) {
                            binding.layoutUser.fadeOut(true)
                            delay(500)
                            binding.switchCamera.fadeIn()
                            binding.pipRenderer.fadeIn()
                        } else {
                            binding.pipRenderer.fadeOut(isGone = true)
                            if (binding.layoutVideo.isVisible) {
                                delay(500)
                                binding.layoutUser.fadeIn()
                            }
                        }
                        viewModel.setEnableVideo(isChecked)
                    }
                }

                override fun onToggleSpeaker(isChecked: Boolean) {
                    viewModel.setEnableSpeaker(isChecked)
                    RingtonePlayer.switchSpeaker(
                        context = requireContext(),
                        speakerEnable = isChecked
                    )
                }

                override fun onToggleMicro(isChecked: Boolean) {
                    viewModel.setEnableAudio(isChecked)
                }

                override fun onHangUp() {
                    endCall()
                }
            }
        )

        binding.switchCamera.clickDebounce(
            listenerBlock = {
                viewModel.switchCamera(
                    switchCameraCallback = object : SwitchCameraCallback {
                        override fun switchDone(b: Boolean) {
                            MainScope().launch {
                                if (b) {
                                    when (binding.pipRenderer.tag) {
                                        "mirrorEnable" -> {
                                            binding.pipRenderer.setMirror(false)
                                            binding.pipRenderer.tag = "mirrorDisable"
                                        }
                                        "mirrorDisable" -> {
                                            binding.pipRenderer.setMirror(true)
                                            binding.pipRenderer.tag = "mirrorEnable"
                                        }
                                    }
                                }
                            }
                        }

                        override fun switchError(s: String?) {
                            showSnackBar(s)
                        }
                    }
                )
            }
        )

        binding.layoutCall.clickDebounce {
            if (binding.callToolbar.isVisible) {
                CallingUtils.hideToggleAnimation(binding.layoutButton, binding.switchCamera)
            } else {
                CallingUtils.showToggleAnimation(binding.layoutButton, binding.switchCamera, binding.pipRenderer.isVisible)
            }
        }

        checkPermissionCall(
            {
                RingtonePlayer.switchSpeaker(
                    context = requireContext(),
                    speakerEnable = binding.callToolbar.isSpeakerOn()
                )
                RingtonePlayer.ringCallOutcome(context = requireContext())
                makeCall()
            },
            {
                endCall()
            }
        )
    }

    private fun makeCall() {
        viewModel.makeCall(
            neGroup = neGroup,
            videoEnable = isVideoEnable,
            callbackError = {
                endCall()
            }
        )
    }

    private fun endCall(status: String? = null) {
        if (!isEndCall) {
            isEndCall = true
            binding.txtStatus.tag = resources.getString(R.string.end_call)
            RingtonePlayer.stopCallOutcome()
            RingtonePlayer.ringEndCall(
                context = requireContext(),
                speakerEnable = binding.callToolbar.isSpeakerOn()
            )
            CallingUtils.stopCountdownStatus()
            binding.txtStatus.text = status ?: resources.getString(R.string.end_call)
            binding.pipRenderer.isGone = true
            binding.layoutUser.fadeIn()
            binding.callToolbar.fadeOut(true)
            viewModel.endCallOut()
        }
    }

    override fun onResume() {
        super.onResume()
        CallingUtils.registerProximitySensor(
            context = requireContext(),
            nearDeviceCallback = {
                if (wakeLock?.isHeld == false) {
                    wakeLock?.acquire(10 * 60 * 1000L)
                }
            },
            farDeviceCallback = {
                if (wakeLock?.isHeld == true) {
                    wakeLock?.release()
                }
            }
        )
    }

    override fun onPause() {
        if (wakeLock?.isHeld == true) {
            wakeLock?.release()
        }
        super.onPause()
        CallingUtils.unregisterProximitySensor(context = requireContext())
    }

    override fun onStop() {
        super.onStop()
        if (powerManager?.isInteractive == false) {
            return
        }
    }

    override fun setupTheme() {
    }
}
