package com.netacom.full.ui.main.chat

import android.os.Bundle
import android.view.View
import androidx.core.view.isGone
import androidx.fragment.app.activityViewModels
import androidx.navigation.fragment.navArgs
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKBottomDialog
import com.netacom.full.databinding.PopupMessageMoreBinding
import com.netacom.lite.define.GroupType
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.util.CallbackResult
import dagger.hilt.android.AndroidEntryPoint

@AndroidEntryPoint
class MessageFunctionDialog : BaseSDKBottomDialog<PopupMessageMoreBinding, ChatViewModel>(
    R.layout.popup_message_more,
    ChatViewModel::class.java
) {
    private val args: MessageFunctionDialogArgs by navArgs()

    private val chatViewModel: ChatViewModel by activityViewModels()

    private var neMessage: NeMessage? = null

    private var isSenderOrOwner = true

    override fun setLayoutHeight(): Int = DialogDef.LAYOUT_WRAP_CONTENT

    override fun setViewModel(): Int = BR.viewModel

    override fun initViews() {
        binding.appId = viewModel.appId()
        binding.messageType = neMessage?.type ?: -1
        if (viewModel.isSecretGroup(neMessage?.groupId)) {
            binding.messageType = -1
        }
        binding.isSenderOrOwner = isSenderOrOwner
        if (args.neGroup.type == GroupType.GROUP_TYPE_CHANNEL) {
            binding.actionShare.isGone = true
            if (!viewModel.isLeaderInGroup(args.neGroup)) {
                binding.actionReply.isGone = true
                binding.actionDelete.isGone = true
                binding.isSenderOrOwner = false
            }
        }

        binding.actionReply.clickDebounce {
            neMessage?.let { _neMessage ->
                viewModel.replyMessage(_neMessage)
                dismiss()
            } ?: showSnackBar(R.string.cant_reply)
        }

        binding.actionCopy.clickDebounce {
            neMessage?.let { _neMessage ->
                viewModel.forwardMessage(_neMessage)
                dismiss()
            } ?: showSnackBar(R.string.cant_reply)
        }

        binding.actionForward.clickDebounce {
            viewModel.actionForward()
            dismiss()
        }

        binding.actionShare.clickDebounce {
            viewModel.actionShare(
                neMessage,
                object : CallbackResult<Boolean> {
                    override fun callBackSuccess(result: Boolean) {
                        dismiss()
                    }

                    override fun callBackError(error: String?) {
                        showSnackBar(R.string.cant_share_message)
                    }
                }
            )
        }

        binding.actionDelete.clickDebounce {
            binding.layoutMessages.visibility = View.GONE
            binding.layoutDelete.visibility = View.VISIBLE
        }

        binding.tvDeleteAll.clickDebounce {
            chatViewModel.deleteMessage(
                neMessage?.groupId?.toLongOrNull(),
                neMessage?.id,
                isDeleteAll = true
            )
            dismiss()
        }

        binding.tvDeleteOnlyMe.clickDebounce {
            chatViewModel.deleteMessage(
                neMessage?.groupId?.toLongOrNull(),
                neMessage?.id,
                isDeleteAll = false
            )
            dismiss()
        }

        binding.tvDeleteCancel.clickDebounce {
            dismiss()
        }

        binding.actionDownload.clickDebounce {
            neMessage?.let { message ->
                viewModel.downloadMediaMessage(message)
                dismiss()
            }
        }
    }

    override fun initData() {
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(
                ivReplay,
                ivDelete,
                ivForward,
                ivCopy,
                ivShare,
                ivDownload
            )
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        neMessage = args.neMessage
        isSenderOrOwner = args.isSenderOrOwner
    }
}
