package com.netacom.full.ui.main.chat.adapter

import android.view.View
import android.view.ViewGroup
import androidx.databinding.ViewDataBinding
import androidx.recyclerview.widget.DiffUtil
import com.netacom.base.chat.adapter.BaseMultiTypeListSimpleAdapter
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.full.R
import com.netacom.full.databinding.ItemForwardAllContactBinding
import com.netacom.full.databinding.ItemForwardRecentlyContactBinding
import com.netacom.full.databinding.ItemRcvUserGroupHeaderBinding
import com.netacom.lite.entity.ui.message.NeForwardMessage

class ForwardMessageAdapter(private var currentUserId: Long = 0L, itemClick: ((NeForwardMessage, View, Int) -> Unit)?) : BaseMultiTypeListSimpleAdapter<NeForwardMessage, ViewDataBinding>(itemClick, diff = ForwardMessageDiff()) {

    companion object {
        const val HEADER = 0
        const val ALL = 1
        const val RECENTLY = 2
    }

    override fun bindMultiType(position: Int): Int {
        return when (getItem(position)?.type) {
            HEADER -> HEADER
            ALL -> ALL
            RECENTLY -> RECENTLY
            else -> RECENTLY
        }
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder<ViewDataBinding> {
        return when (viewType) {
            HEADER -> BaseViewHolder(inflateView(parent, R.layout.item_rcv_user_group_header) as ItemRcvUserGroupHeaderBinding)
            ALL -> BaseViewHolder(inflateView(parent, R.layout.item_forward_all_contact) as ItemForwardAllContactBinding)
            RECENTLY -> BaseViewHolder(inflateView(parent, R.layout.item_forward_recently_contact) as ItemForwardRecentlyContactBinding)
            else -> BaseViewHolder(inflateView(parent, R.layout.item_forward_recently_contact) as ItemForwardRecentlyContactBinding)
        }
    }

    override fun onBindViewHolder(holder: BaseViewHolder<ViewDataBinding>, position: Int) {
        getItem(position)?.apply {
            with(holder) {
                when (getItemViewType(position)) {
                    HEADER -> bindForwardHeader(holder = this, header = header)
                    ALL -> bindForwardAll(holder = this, item = this@apply)
                    RECENTLY -> bindForwardRecently(holder = this, item = this@apply)
                }
            }
        }
    }

    private fun bindForwardHeader(holder: BaseViewHolder<ViewDataBinding>, header: String?) {
        with(holder.binding as ItemRcvUserGroupHeaderBinding) {
            header?.let {
                this.header = header
            }
        }
    }

    private fun bindForwardAll(holder: BaseViewHolder<ViewDataBinding>, item: NeForwardMessage?) {
        with(holder.binding as ItemForwardAllContactBinding) {
            item?.neUser?.let { _neUser ->
                this.item = _neUser
                itemClick?.let {
                    viewMain.clickDebounce {
                        itemClick.invoke(item, viewMain, 0)
                    }
                }
            }
        }
    }

    private fun bindForwardRecently(holder: BaseViewHolder<ViewDataBinding>, item: NeForwardMessage?) {
        with(holder.binding as ItemForwardRecentlyContactBinding) {
            item?.let { neForwardMessage ->
                this.neGroup = neForwardMessage.neGroup
                val neUser = neGroup?.members?.find { member ->
                    member.id != currentUserId
                }
                neUser?.let { user ->
                    this.neUser = user
                } ?: neForwardMessage.neGroup?.owner
                itemClick?.let {
                    viewMain.clickDebounce {
                        itemClick.invoke(item, viewMain, 0)
                    }
                }
            }
        }
    }

    private class ForwardMessageDiff : DiffUtil.ItemCallback<NeForwardMessage>() {
        override fun areItemsTheSame(oldItem: NeForwardMessage, newItem: NeForwardMessage): Boolean {
            return oldItem.type == newItem.type
        }

        override fun areContentsTheSame(oldItem: NeForwardMessage, newItem: NeForwardMessage): Boolean {
            return when (oldItem.type) {
                HEADER -> {
                    oldItem.header == newItem.header
                }
                ALL -> {
                    oldItem.neUser?.id == newItem.neUser?.id
                }
                RECENTLY -> {
                    oldItem.neGroup?.id == newItem.neGroup?.id
                }
                else -> {
                    false
                }
            }
        }
    }
}
