package com.netacom.full.ui.main.chat.viewholder

import androidx.core.view.isGone
import androidx.core.view.isInvisible
import androidx.core.view.isVisible
import androidx.databinding.ViewDataBinding
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.android_utils.FileUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.network.ApiResponseError
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.full.R
import com.netacom.full.databinding.RowMessageAudioReceiverBinding
import com.netacom.full.define.ChatMessageType
import com.netacom.full.define.MessagePosition
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.full.utils.AudioSupport
import com.netacom.lite.define.GroupType.GROUP_TYPE_PRIVATE
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.repository.DownloadRepository
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.Constants.EMPTY
import com.netacom.lite.util.getUrlImage
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class MessageAudioReceiverViewHolder(
    val _binding: RowMessageAudioReceiverBinding,
    private val themeHelperImpl: ThemeHelperImpl,
    private val groupType: Int
) : BaseViewHolder<ViewDataBinding>(_binding) {

    fun bind(
        message: NeMessage?,
        userToken: String,
        messagePosition: MessagePosition,
        isSingle: Boolean,
        downloadRepository: DownloadRepository,
        avatarCallback: () -> Unit
    ) {
        message?.let { item ->
            _binding.messagePosition = messagePosition
            _binding.item = item
            _binding.receiverColor = themeHelperImpl.mainColor

            when (AudioSupport.messageId) {
                item.id?.toLongOrNull() -> {
                    AudioSupport.setViewPlayer(view = _binding.root)
                    _binding.imgPlay.setImageResource(R.drawable.ic_record_pause)
                }
                else -> {
                    if (AudioSupport.audioView?.toString() == _binding.root.toString()) {
                        AudioSupport.audioView = null
                    }
                    _binding.imgPlay.setImageResource(R.drawable.ic_record_play)
                    _binding.seekbar.progress = 0
                    item.attachment?.audio?.apply {
                        _binding.txtProgress.text = AppUtils.setAudioTextViewTime(time = this.duration ?: 0)
                    }
                }
            }

            _binding.ivAvatarReceiver.clickDebounce {
                avatarCallback()
            }

            setupPlayAction(item, userToken, groupType == GROUP_TYPE_PRIVATE, downloadRepository)
            setupMessagePosition(messagePosition, isSingle)

            themeHelperImpl.setThemeColorForMessage(_binding.ctlContainer, ChatMessageType.PARTNER_MESSAGE_AUDIO, groupType)
            if (groupType == GROUP_TYPE_PRIVATE) {
                themeHelperImpl.setThemeColorForSecretChat(_binding.imgPlay, _binding.seekbar)
            } else {
                themeHelperImpl.setThemeColorForViews(_binding.imgPlay, _binding.seekbar)
            }
        }
    }

    private fun setupPlayAction(
        item: NeMessage,
        userToken: String,
        isEncrypt: Boolean,
        downloadRepository: DownloadRepository
    ) {
        _binding.imgPlay.clickDebounce {
            if (isEncrypt) {
                val localAudioSecretPath = "${com.netacom.lite.util.FileUtils.secretFolderPath}/${item.id}/0"
                if (FileUtils.isFileExists(localAudioSecretPath)) {
                    actionAudio(item, userToken, localAudioSecretPath)
                } else {
                    _binding.audioLoading.isVisible = true
                    CoroutineScope(Dispatchers.IO).launch {
                        downloadRepository.downloadFiles(arrayListOf(item.attachment?.audio?.url.getUrlImage())).collect {
                            when (it) {
                                is ApiResponseSuccess -> {
                                    val responseDownload = it.data
                                    responseDownload?.let { responseBody ->
                                        com.netacom.lite.util.FileUtils.saveEncryptedFileToLocal(
                                            body = responseBody,
                                            path = "${com.netacom.lite.util.FileUtils.secretFolderPath}/${item.id}",
                                            "0"
                                        ).let { path ->
                                            withContext(Dispatchers.Main) {
                                                _binding.audioLoading.isVisible = false
                                                actionAudio(item, userToken, path)
                                            }
                                        }
                                    }
                                }
                                is ApiResponseError -> {
                                }
                            }
                        }
                    }
                }
            } else {
                actionAudio(item, userToken, item.attachment?.audio?.url ?: EMPTY)
            }
        }
    }

    private fun actionAudio(item: NeMessage, userToken: String, url: String) {
        if (AudioSupport.messageId == item.id?.toLongOrNull() &&
            (AudioSupport.status == AudioSupport.AudioStatus.STATUS_PLAYING || AudioSupport.status == AudioSupport.AudioStatus.STATUS_LOADING)
        ) {
            /**
             * This audio play -> pause
             * */
            AudioSupport.stopAudio()
        } else if (AudioSupport.messageId != item.id?.toLongOrNull() &&
            (AudioSupport.status == AudioSupport.AudioStatus.STATUS_PLAYING || AudioSupport.status == AudioSupport.AudioStatus.STATUS_LOADING)
        ) {
            /**
             * Other audio play -> pause
             * This audio pause -> play
             * */
            AudioSupport.stopAudio()

            AudioSupport.setViewPlayer(view = _binding.root)
            AudioSupport.playAudio(
                context = _binding.root.context,
                messageId = item.id?.toLongOrNull(),
                audioUrl = url,
                userToken = userToken
            )
        } else {
            /**
             * Play this audio
             * */
            AudioSupport.setViewPlayer(view = _binding.root)
            AudioSupport.playAudio(
                context = _binding.root.context,
                messageId = item.id?.toLongOrNull(),
                audioUrl = url,
                userToken = userToken
            )
        }
    }

    private fun setupMessagePosition(messagePosition: MessagePosition, isOneToOne: Boolean) {
        when (messagePosition) {
            MessagePosition.SINGLE -> {
                _binding.isShowName = !isOneToOne
                _binding.ivAvatarReceiver.isVisible = true
                _binding.vSeparator.isVisible = true
                _binding.tvReceiverTime.isVisible = true
            }
            MessagePosition.TOP -> {
                _binding.isShowName = !isOneToOne
                _binding.ivAvatarReceiver.isInvisible = true
                _binding.vSeparator.isVisible = true
                _binding.tvReceiverTime.isGone = true
            }
            MessagePosition.MID -> {
                _binding.isShowName = false
                _binding.ivAvatarReceiver.isInvisible = true
                _binding.vSeparator.isGone = true
                _binding.tvReceiverTime.isGone = true
            }
            MessagePosition.BOTTOM -> {
                _binding.isShowName = false
                _binding.ivAvatarReceiver.isVisible = true
                _binding.vSeparator.isGone = true
                _binding.tvReceiverTime.isVisible = true
            }
        }
    }
}
