/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.contact

import android.content.ContentProviderOperation
import android.os.Bundle
import android.provider.ContactsContract
import android.view.View
import android.widget.Toast
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.core.widget.doAfterTextChanged
import androidx.fragment.app.activityViewModels
import androidx.navigation.NavController
import androidx.navigation.fragment.findNavController
import androidx.navigation.fragment.navArgs
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.toast
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKBottomDialog
import com.netacom.full.databinding.ViewAddContactDialogBinding
import com.netacom.full.extensions.checkPhone
import com.netacom.full.extensions.convertPhoneNumber
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.chat.BaseChatFragment
import com.netacom.full.ui.main.chat.ChatViewModel
import com.netacom.full.ui.main.group.ChatInfoFragment
import com.netacom.full.utils.DialogUtil
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.database.contact.DbContactEntity
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.network.model.response.RegisterContact
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants.EMPTY
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.ObsoleteCoroutinesApi

/**
 * Created by Tam Nguyen on 9/03/20. 
 */

@AndroidEntryPoint
class AddContactDialog : BaseSDKBottomDialog<ViewAddContactDialogBinding, ContactViewModel>(
    R.layout.view_add_contact_dialog,
    ContactViewModel::class.java
) {

    companion object {
        const val TAG = "_AddContactDialog"
    }

    private val args: AddContactDialogArgs by navArgs()

    override fun setViewModel(): Int = BR.viewModel

    override fun setLayoutHeight(): Int = DialogDef.LAYOUT_FULL

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()
    private val chatViewModel: ChatViewModel by activityViewModels()

    private var userID: Long? = 0L
    private var neUser: NeUser? = null
    private var isChat: Boolean = false
    private var className: String? = null
    private var neContactData: NeUser? = null
    private lateinit var navController: NavController

    override fun initViews() {
        neUser?.let { _neUser ->
            binding.item = _neUser
        }
        binding.txtName.doAfterTextChanged { editTable ->
            editTable ?: return@doAfterTextChanged
            if (neUser == null) {
                binding.tvName.isVisible = true
                binding.tvName.text = AppUtils.nameRepresent(editTable.toString())
            }
            binding.imgClear.visibility = if (editTable.isNotEmpty()) View.VISIBLE else View.GONE
            if ((neUser?.getPhone?.isNotEmpty() == true)) {
                binding.tvDone.isEnabled = editTable.isNotEmpty()
                neUser?.getPhone.orEmpty().checkPhone()
            } else {
                binding.tvDone.isEnabled =
                    editTable.isNotEmpty() && binding.tvPhone.text.toString().checkPhone()
            }
        }

        binding.tvPhone.doAfterTextChanged { _tvPhone ->
            _tvPhone ?: return@doAfterTextChanged
            binding.imgClearPhone.visibility = if (_tvPhone.isNotEmpty()) View.VISIBLE else View.GONE
            binding.tvDone.isEnabled =
                binding.txtName.text.toString().isNotEmpty() && _tvPhone.toString().checkPhone()
        }

        binding.tvPhone.setText(neUser?.getPhone.orEmpty())
        if (neUser?.getPhone?.isNotEmpty() == true) {
            binding.imgClearPhone.isVisible = true
        } else {
            binding.imgClearPhone.isGone = true
        }
        binding.txtName.setText(neUser?.getDisplayName)

        binding.icClose.clickDebounce {
            dismiss()
        }

        binding.imgClear.clickDebounce {
            binding.txtName.setText(EMPTY)
        }

        binding.imgClearPhone.clickDebounce {
            binding.tvPhone.setText(EMPTY)
        }

        binding.tvDone.clickDebounce {
            val phoneNumber = binding.tvPhone.text.toString()
            val name = binding.txtName.text.toString()
            if (name.isBlank() || !phoneNumber.checkPhone()) {
                showSnackBar(R.string.error_text_phone_number)
                binding.txtName.requestFocus()
            } else {
                when {
                    viewModel.checkIsMyPhone(phoneNumber) -> {
                        DialogUtil.showMessage(
                            context = requireContext(),
                            message = R.string.text_message_it_you,
                            cancelLabel = R.string.popup_confirm,
                            okButtonIsGone = true
                        )
                    }
                    checkContactByPhoneDb(phoneNumber) -> {
                        neUser?.let { user ->
                            viewModel.startOneByOneChat(user)
                            dismiss()
                        }
                    }
                    else -> {
                        checkContactByPhoneServer(name = name, phoneNumber = phoneNumber)
                    }
                }
            }
        }
    }

    override fun initData() {
    }

    override fun setupTheme() {
        themeHelperImpl.setThemeColorForViews(binding.tvPhone, binding.txtName)
        themeHelperImpl.setThemeStateListForButtonAction(binding.tvDone)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        neUser = args.neUser
        isChat = args.isChat
        className = args.className
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        navController = findNavController()
    }

    private fun checkContactByPhoneServer(name: String, phoneNumber: String) {
        viewModel.checkContactByPhoneNumber(
            phoneNumber = phoneNumber,
            object : CallbackResult<RegisterContact> {
                override fun callBackSuccess(result: RegisterContact) {
                    userID = result.id
                    checkPermissionContact(
                        {
                            addContact(name = name, phoneNumber = phoneNumber)
                        },
                        {
                            dismiss()
                        }
                    )
                }

                override fun callBackError(error: String?) {
                    inviteFriend(name = name, phoneNumber = phoneNumber)
                }
            }
        )
    }

    private fun addNewContact(name: String, phoneNumber: String, callbackSuccess: () -> Unit) {
        val operationList = ArrayList<ContentProviderOperation>()
        operationList.add(
            ContentProviderOperation.newInsert(ContactsContract.RawContacts.CONTENT_URI)
                .withValue(ContactsContract.RawContacts.ACCOUNT_TYPE, null)
                .withValue(ContactsContract.RawContacts.ACCOUNT_NAME, null)
                .build()
        )

        operationList.add(
            ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI)
                .withValueBackReference(ContactsContract.Data.RAW_CONTACT_ID, 0)
                .withValue(
                    ContactsContract.Data.MIMETYPE,
                    ContactsContract.CommonDataKinds.StructuredName.CONTENT_ITEM_TYPE
                )
                .withValue(ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME, name)
                .withValue(ContactsContract.CommonDataKinds.StructuredName.FAMILY_NAME, EMPTY)
                .build()
        )

        operationList.add(
            ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI)
                .withValueBackReference(ContactsContract.Data.RAW_CONTACT_ID, 0)
                .withValue(
                    ContactsContract.Data.MIMETYPE,
                    ContactsContract.CommonDataKinds.Phone.CONTENT_ITEM_TYPE
                )
                .withValue(ContactsContract.CommonDataKinds.Phone.NUMBER, phoneNumber)
                .withValue(
                    ContactsContract.CommonDataKinds.Phone.TYPE,
                    ContactsContract.CommonDataKinds.Phone.TYPE_HOME
                )
                .build()
        )

        try {
            requireActivity().contentResolver.applyBatch(ContactsContract.AUTHORITY, operationList)
            callbackSuccess()

            phoneNumber.convertPhoneNumber()?.let { _phoneNumber ->
                if (_phoneNumber.length >= 10) {
                    val dbContactEntity = DbContactEntity().apply {
                        this.contactPhone = _phoneNumber
                        this.localName = name
                    }
                    viewModel.saveContactToDb(listOf(dbContactEntity))
                }
            }
            dismiss()
        } catch (e: Exception) {
            FirebaseCrashlytics.getInstance().recordException(e)
            e.printStackTrace()
        }
    }

    @OptIn(ObsoleteCoroutinesApi::class)
    private fun addContact(name: String, phoneNumber: String) {
        viewModel.addContact(
            name = name,
            phone = phoneNumber,
            object : CallbackResult<RegisterContact> {
                override fun callBackSuccess(result: RegisterContact) {
                    addNewContact(
                        name = name,
                        phoneNumber = phoneNumber,
                        callbackSuccess = {
                            val neUser = NeUser(id = result.id, username = result.fullName, phone = result.phone, email = result.email, avatar = result.profileUrl)
                            showSnackBar(R.string.common_success)
                            Logger.e("isChat==$isChat  className==$className   tag==${BaseChatFragment.TAG}  neUser==$neUser")
                            when {
                                className == BaseChatFragment.TAG -> {
                                    mainSdkViewModel.syncContact(SyncType.LIST_SYNC_DB)
                                    chatViewModel._addNewContactSuccess.postValue(EventLiveData(neUser))
                                    dismiss()
                                }
                                className == ChatInfoFragment.TAG -> {
                                    requireActivity().runOnUiThread {
                                        mainSdkViewModel.syncContact(SyncType.LIST_SYNC_DB)
                                        mainSdkViewModel.onBack()
                                    }
                                }
                                isChat -> {
                                    viewModel.startOneByOneChat(neUser)
                                }
                                else -> {
                                    mainSdkViewModel.syncContact(SyncType.LIST_SYNC_DB)
                                    viewModel.onBack()
                                    chatViewModel._addNewContactSuccess.postValue(EventLiveData(neUser))
                                }
                            }
                        }
                    )
                }

                override fun callBackError(error: String?) {
                    requireActivity().runOnUiThread {
                        requireContext().toast(getString(R.string.cant_add_contact), Toast.LENGTH_LONG)
                    }
                    dismiss()
                }
            }
        )
    }

    private fun showGoToChat(userId: Long) {
        DialogUtil.showMessage(
            context = requireContext(),
            message = R.string.popup_check_contact_chat,
            okLabel = R.string.contact_chat_now,
            cancelLabel = R.string.common_later,
            okFunc = {
            },
            cancelFunc = {
            }
        )
    }

    private fun inviteFriend(name: String, phoneNumber: String) {
        DialogUtil.showMessage(
            context = requireContext(),
            message = R.string.popup_confirm_user_use_app,
            okLabel = R.string.popup_confirm,
            cancelLabel = R.string.common_deny,
            okFunc = {
                addNewContact(
                    name,
                    phoneNumber,
                    callbackSuccess = {
                        Logger.d("Save contact to local success")
                        phoneNumber.convertPhoneNumber()?.let {
                            viewModel.inviteFriend(it)
                        }
                    }
                )
            }
        )
    }

    private fun checkContactByPhoneDb(phone: String): Boolean {
        var checkContact = false
        viewModel.getContactByPhone(
            phone = (phone.convertPhoneNumber().orEmpty()),
            object : CallbackResult<NeUser?> {
                override fun callBackSuccess(result: NeUser?) {
                    if (result == null) checkContact = false
                    else {
                        userID = result.id ?: 0L
                        neContactData = result
                        checkContact = true
                    }
                }

                override fun callBackError(error: String?) {
                }
            }
        )
        return checkContact
    }
}
