/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.contact

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.navigation.NavController
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.network.ApiResponseError
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.network.ResultData
import com.netacom.base.chat.util.unAccentText
import com.netacom.full.basechat.BaseSDKViewModel
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.full.ui.main.contact.adapter.RecentSearchAdapter
import com.netacom.full.ui.main.contact.model.SubNeContact
import com.netacom.full.utils.SearchUtil.getFilterKeysFromFilter
import com.netacom.full.utils.convertPhoneNumber
import com.netacom.full.utils.standardizedPhoneNumber
import com.netacom.lite.entity.database.contact.DbContactEntity
import com.netacom.lite.entity.database.contact.DbSearchContactEntity
import com.netacom.lite.entity.database.group.DbGroupEntity
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.network.model.response.RegisterContact
import com.netacom.lite.repository.ContactRepository
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.supervisorScope
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import java.util.Locale
import javax.inject.Inject

@ExperimentalCoroutinesApi
@HiltViewModel
class ContactViewModel @Inject constructor(
    socketRepository: SocketRepository,
    private val groupRepository: GroupRepository,
    private val contactRepository: ContactRepository,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseSDKViewModel(socketRepository, groupRepository, navigationDispatcher) {

    companion object {
        const val TAG = "_ContactViewModel"
    }

    private val _recentContacts = MutableLiveData<EventLiveData<List<NeUser>>>()
    val recentContacts: LiveData<EventLiveData<List<NeUser>>> = _recentContacts

    private val _recentContactsSearch = MutableLiveData<EventLiveData<List<RecentSearchAdapter.SubNeItem>>>()
    val recentContactsSearch: MutableLiveData<EventLiveData<List<RecentSearchAdapter.SubNeItem>>> = _recentContactsSearch

    private val _searchContacts = MutableLiveData<EventLiveData<List<RecentSearchAdapter.SubNeItem>>>()
    val searchContacts: MutableLiveData<EventLiveData<List<RecentSearchAdapter.SubNeItem>>> = _searchContacts

    private val _listContactNotRegister = MutableLiveData<EventLiveData<ResultData<List<NeUser>>>>()
    val listContactNotRegister: LiveData<EventLiveData<ResultData<List<NeUser>>>> = _listContactNotRegister

    private val _inviteFriend = MutableLiveData<EventLiveData<String>>()
    val inviteFriend: LiveData<EventLiveData<String>> = _inviteFriend

    var cacheContactUI: List<SubNeContact>? = null

    /**
     * cache Registered Contact & Group to support Search Contact screen.
     */
    private var cacheContact: List<DbContactEntity>? = null
    private var cacheGroup: List<DbGroupEntity>? = null

    /**
     * call this function to check group is exists from socket
     * success: response exists group
     * fail: return null or empty group
     * */

    fun getListContactNotRegister() {
        launchOnViewModelScope(getPostExecutionThread.io) {
            _listContactNotRegister.post(
                EventLiveData(
                    ResultData.Success(contactRepository.getListContactNotRegister())
                )
            )
        }
    }

    /**
     * Based on the phone number to get the Contact Info from Local Storage
     */
    fun getContactByPhone(phone: String, callbackResult: CallbackResult<NeUser?>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.getContactByPhone(phone = phone)?.apply {
                callbackResult.callBackSuccess(this)
            } ?: callbackResult.callBackError()
        }
    }

    fun addContact(name: String, phone: String, callbackResult: CallbackResult<RegisterContact>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.addContact(name = name, phone = phone)
                .collect { state ->
                    when (state) {
                        is ApiResponseSuccess -> {
                            state.data?.let { registerContact ->
                                callbackResult.callBackSuccess(registerContact)
                            }
                        }
                        is ApiResponseError -> {
                            callbackResult.callBackError()
                        }
                    }
                }
        }
    }

    fun inviteFriend(phoneNumber: String) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            inviteFriend.post(EventLiveData(phoneNumber))
        }
    }

    fun getRecentlyContact(limit: Long) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            supervisorScope {
                val listNeUser = contactRepository.getListUserFromRecentContact(limit)
                _recentContacts.post(EventLiveData(listNeUser))
            }
        }
    }

    fun getRecentContactsSearch() {
        launchOnViewModelScope(getPostExecutionThread.io) {
            supervisorScope {
                val listNeUser = contactRepository.getListUserFromSearchContacts()
                _recentContactsSearch.post(
                    listNeUser?.map { data ->
                        RecentSearchAdapter.SubNeItem(
                            _type = RecentSearchAdapter.TYPE_RESULT_CONTACT_SEARCH,
                            _neUser = NeUser().apply {
                                id = data.id
                                username = data.username
                                email = data.email
                                phone = data.getPhone
                                avatar = data.avatar
                            }
                        )
                    }.let {
                        EventLiveData(it)
                    }
                )
            }
        }
    }

    fun saveSearchContact(_dbSearchContactEntity: DbSearchContactEntity) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.saveSearchContact(_dbSearchContactEntity)
        }
    }

    fun deleteContactSearch(callbackResult: CallbackResult<Boolean>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.deleteContactSearch()?.let {
                callbackResult.callBackSuccess(it)
            } ?: callbackResult.callBackError()
        }
    }

    private suspend fun getCacheData() = withContext(getPostExecutionThread.io) {
        if (cacheContact == null) {
            cacheContact = contactRepository.getLocalContacts()
        }

        if (cacheGroup == null) {
            cacheGroup = contactRepository.getGroupDb()
        }
    }

    /**
     * WAITING!!!!  Need to discuss with the iOS team to find out the final solution
     * Refer: ContactSearchFragment
     */
    fun searchLocalContact(filter: String) {
        launchOnViewModelScope(getPostExecutionThread.default) {
            getCacheData()
            val result = mutableListOf<RecentSearchAdapter.SubNeItem>()
            // search with name in netalo contact
            supervisorScope {
                val searchByNames = cacheContact?.run {
                    searchContactFromName(filter, this)
                } ?: listOf()
                result.addAll(getSearchContactResult(searchByNames))
                // if listSearchNetaloContact size == 0 then search in phone contact if filter has format sub phone number
                if (result.isEmpty() && AppUtils.isMiniVnPhoneNumber(filter)) {
                    result.addAll(
                        cacheContact?.run {
                            val searchByPhones = searchContactFromNumber(
                                filter,
                                this.filter { it.isRegistered == true }
                            )
                            getSearchContactResult(searchByPhones)
                        } ?: listOf()
                    )
                }
                // if cannot filter in netalo contact then check filter in personal contact and invite them
                if (result.isEmpty() && AppUtils.isVnPhoneNumber(filter)) {
                    result.addAll(
                        cacheContact?.run {
                            val searchByPersonalContact =
                                searchContactFromNumber(
                                    filter,
                                    this.filter { it.isRegistered == false }
                                )
                            getSearchPhoneInPersonalContactResult(searchByPersonalContact)
                        } ?: listOf()
                    )
                }
                result.addAll(
                    cacheGroup?.run {
                        getSearchGroupResult(filter, this, searchByNames)
                    } ?: listOf()
                )
                // if filter is phone, check is it netalo user, if it is netalo user then save it
                if (result.isEmpty() && AppUtils.isVnPhoneNumber(filter)) {
                    contactRepository.checkContactByPhoneNumber(filter).collect {
                        if (it is ApiResponseSuccess) {
                            result.add(
                                RecentSearchAdapter.SubNeItem(
                                    _type = RecentSearchAdapter.TYPE_ADD_CONTACT_REGISTER,
                                    _neUser = NeUser(
                                        id = it.data?.id,
                                        username = it.data?.fullName,
                                        phone = it.data?.phone,
                                        avatar = it.data?.profileUrl
                                    )
                                )
                            )
                        } else result.add(
                            RecentSearchAdapter.SubNeItem(
                                _type = RecentSearchAdapter.TYPE_ADD_CONTACT_LOCAL,
                                _neUser = NeUser(
                                    phone = filter.convertPhoneNumber()
                                )
                            )
                        )
                    }
                }
                _searchContacts.postValue(EventLiveData(result))
            }
        }
    }

    private suspend fun getSearchGroupResult(
        filter: String,
        listGroup: List<DbGroupEntity>,
        listContact: List<DbContactEntity>
    ): List<RecentSearchAdapter.SubNeItem> =
        suspendCancellableCoroutine { continuation ->
            val result = mutableListOf<RecentSearchAdapter.SubNeItem>()
            val groupSearchResults = mutableListOf<DbGroupEntity>()
            val filterKeys = getFilterKeysFromFilter(filter.toLowerCase(Locale.getDefault()).unAccentText())
            val groupUseForSearchs = listGroup.toMutableList()
            // search from group name
            filterKeys.forEach { key ->
                val searchGroupByKey = groupUseForSearchs.filter { group ->
                    group.getDisplayName.unAccentText().toLowerCase(Locale.getDefault()).contains(key)
                }
                groupSearchResults.addAll(searchGroupByKey)
                groupUseForSearchs.removeAll(searchGroupByKey)
            }
            // search group has netalo contact from search contact result
            val groupSearchByPartners = groupUseForSearchs.filter {
                it.groupMember.any { partner ->
                    listContact.any { dbContact ->
                        dbContact.contactId?.equals(partner) == true
                    }
                }
            }
            groupSearchResults.addAll(groupSearchByPartners)
            if (!groupSearchResults.isNullOrEmpty()) {
                result.add(
                    RecentSearchAdapter.SubNeItem(
                        _type = RecentSearchAdapter.TYPE_HEADER_SEARCH,
                        _neHeader = RecentSearchAdapter.TYPE_HEADER_GROUP
                    )
                )
                result.addAll(
                    groupSearchResults.map {
                        RecentSearchAdapter.SubNeItem(
                            _type = RecentSearchAdapter.TYPE_RESULT_GROUP_SEARCH,
                            _neGroup = groupRepository.convertDBToNeGroup(it)
                        )
                    }
                )
            }
            continuation.resume(result) {
                it.printStackTrace()
            }
        }

    private suspend fun getSearchContactResult(
        listContact: List<DbContactEntity>
    ): List<RecentSearchAdapter.SubNeItem> = suspendCancellableCoroutine { continuation ->
        val result = mutableListOf<RecentSearchAdapter.SubNeItem>()
        if (!listContact.isNullOrEmpty()) {
            result.add(
                RecentSearchAdapter.SubNeItem(
                    _type = RecentSearchAdapter.TYPE_HEADER_SEARCH,
                    _neHeader = RecentSearchAdapter.TYPE_HEADER_CONTACT
                )
            )
            result.addAll(
                listContact.map {
                    RecentSearchAdapter.SubNeItem(
                        _type = RecentSearchAdapter.TYPE_RESULT_CONTACT_SEARCH,
                        _neUser = contactRepository.getContactDbUiMapper.mapFromEntity(it)
                    )
                }
            )
        }
        continuation.resume(result) {
            it.printStackTrace()
        }
    }

    private suspend fun getSearchPhoneInPersonalContactResult(listContact: List<DbContactEntity>): List<RecentSearchAdapter.SubNeItem> =
        suspendCancellableCoroutine { continuation ->
            val result = mutableListOf<RecentSearchAdapter.SubNeItem>()
            if (!listContact.isNullOrEmpty()) {
                result.addAll(
                    listContact.map {
                        RecentSearchAdapter.SubNeItem(
                            _type = if (it.isPartner == true) RecentSearchAdapter.TYPE_ADD_CONTACT_REGISTER else RecentSearchAdapter.TYPE_INVITE_FRIEND,
                            _neUser = contactRepository.getContactDbUiMapper.mapFromEntity(it)
                        )
                    }
                )
            }
            continuation.resume(result) {
                it.printStackTrace()
            }
        }

    private suspend fun searchContactFromNumber(
        filter: String,
        listContact: List<DbContactEntity>
    ): List<DbContactEntity> = suspendCancellableCoroutine { continuation ->
        val searchContactResults: List<DbContactEntity>
        val filterPhoneSearch =
            if (filter.startsWith("0")) filter.subSequence(1, filter.length).toString()
            // else if (filter.startsWith("+84")) filter.subSequence(3, filter.length).toString()
            else filter
        searchContactResults =
            listContact.filter { it.contactPhone?.contains(filterPhoneSearch) == true }
        continuation.resume(searchContactResults) {
            it.printStackTrace()
        }
    }

    /**
     * search contact with filter
     */
    private suspend fun searchContactFromName(
        filter: String,
        listRegisteredContact: List<DbContactEntity>
    ): List<DbContactEntity> = suspendCancellableCoroutine { continuation ->
        val filterLower = filter.toLowerCase(Locale.getDefault())
        val searchContactResults = mutableListOf<DbContactEntity>()
        val filterKeys = getFilterKeysFromFilter(filterLower)
        val netaloContacts =
            listRegisteredContact.filter { it.isRegistered == true }.toMutableList()
        // search with multi keys generate from filter
        filterKeys.forEach { filterKey ->
            val searchForKeyResult = netaloContacts.filter {
                (
                    it.serverName?.toLowerCase(Locale.getDefault())?.contains(filterKey) == true ||
                        it.localName?.toLowerCase(Locale.getDefault())?.contains(filterKey) == true ||
                        it.localName?.unAccentText()?.toLowerCase(Locale.getDefault())?.contains(filterKey) == true ||
                        it.serverName?.unAccentText()?.toLowerCase(Locale.getDefault())?.contains(filterKey) == true
                    ) ||
                    (it.contactPhone?.contains(filterKey) == true)
            }
            // add search result
            searchContactResults.addAll(searchForKeyResult)
            // remove search result avoid duplicate result
            netaloContacts.removeAll(searchForKeyResult)
        }
        continuation.resume(searchContactResults) {
            it.printStackTrace()
        }
    }

    fun checkContactByPhoneNumber(
        phoneNumber: String,
        callbackResult: CallbackResult<RegisterContact>
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.checkContactByPhoneNumber(phoneNumber = phoneNumber)
                .collect { state ->
                    withContext(getPostExecutionThread.main) {
                        when (state) {
                            is ApiResponseSuccess -> {
                                state.data?.let { registerContact ->
                                    callbackResult.callBackSuccess(registerContact)
                                } ?: callbackResult.callBackError()
                            }
                            is ApiResponseError -> callbackResult.callBackError()
                        }
                    }
                }
        }
    }

    fun saveContactToDb(listContact: List<DbContactEntity>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.saveLocalContact(listContact)
        }
    }

    fun openAddContact(className: String? = null) {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                ContactFragmentDirections.addContactPhone(
                    neUser = null,
                    isChat = true,
                    className = className
                )
            )
        }
    }

    fun openSearch() {
        navigationDispatcher.emit {
            it.navigateIfSafe(ContactFragmentDirections.actionContactFragmentToContactSearch())
        }
    }

    fun openInvite() {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                ContactFragmentDirections.actionContactFragmentToInviteFriendContact()
            )
        }
    }

    fun openInviteFriend(name: String, phoneNumber: String) {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                InviteFriendFragmentDirections.actionInviteFriendContactToInviteFriendInfo(
                    name = name,
                    phone = phoneNumber
                )
            )
        }
    }

    fun actionSearch(neContact: NeUser, className: String? = null) {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                ContactSearchFragmentDirections.addContactPhone(
                    neUser = neContact,
                    isChat = false,
                    className = className
                )
            )
        }
    }

    fun back() {
        navigationDispatcher.emit { nav ->
            nav.popBackStack()
        }
    }

    fun checkIsMyPhone(phoneNumber: String): Boolean {
        val standardizedPhoneNumber = phoneNumber.standardizedPhoneNumber()
        return getPreferences.user?.getPhone?.let { myPhoneNumber ->
            standardizedPhoneNumber == myPhoneNumber.standardizedPhoneNumber()
        } ?: false
    }

    fun filterContactHasBlocked(
        listContact: List<NeUser>,
        sortSelected: String,
        callbackResult: (List<SubNeContact>) -> Unit
    ) {
        launchOnViewModelScope(getPostExecutionThread.main) {
            listContact.filter { user ->
                groupRepository.getGroupOneToOneByUserId(user.id ?: 0)?.blockers?.isNotEmpty() ?: false
            }.forEach {
                it.isBlocked = true
            }
            val list = ContactUtils.sortContacts(sortBy = sortSelected, listContacts = listContact)
            callbackResult(list)
        }
    }
}
