/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.contact.adapter

import android.view.View
import android.view.ViewGroup
import androidx.core.view.isGone
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.LifecycleCoroutineScope
import androidx.recyclerview.widget.DiffUtil
import com.netacom.base.chat.adapter.BaseMultiTypeListSimpleAdapter
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.unAccentText
import com.netacom.full.R
import com.netacom.full.databinding.ItemContactBinding
import com.netacom.full.databinding.ItemRcvMemberBinding
import com.netacom.full.databinding.ItemRcvUserGroupBinding
import com.netacom.full.databinding.ItemRcvUserGroupHeaderBinding
import com.netacom.full.databinding.ItemRcvUserMessageBinding
import com.netacom.full.ui.main.contact.ContactUtils
import com.netacom.full.ui.main.contact.model.SubNeContact
import com.netacom.lite.entity.ui.user.NeUser
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

open class ContactAdapter(
    itemClick: ((SubNeContact, View, Int) -> Unit)?,
    private val voiceCall: ((NeUser) -> Unit),
    private val videoCall: ((NeUser) -> Unit),
    private val contactType: String,
    val emptyCallBack: ((Boolean) -> Unit)?
) : BaseMultiTypeListSimpleAdapter<SubNeContact, ViewDataBinding>(itemClick, diff = ContactDiff()) {

    companion object {
        private const val CONTACT_HEADER = 0
        private const val CONTACT_BODY = 1
        const val CONTACT_TYPE_MAIN = "MAIN"
        const val CONTACT_TYPE_NEW_MESSAGE = "NEW_MESSAAGE"
        const val CONTACT_TYPE_NEW_GROUP = "NEW_GROUP"
        const val CONTACT_TYPE_MEMBERS = "MEMBERS"
    }

    private val neContacts = mutableListOf<NeUser>()

    override fun bindMultiType(position: Int): Int {
        return when {
            itemCount == 0 -> 0
            getItem(position)?.isHeader() == true -> CONTACT_HEADER
            else -> CONTACT_BODY
        }
    }

    fun swapData(listData: List<SubNeContact>) {
        this.neContacts.clear()
        val users = listData.mapNotNull {
            it._itemNeContact?._neContact
        }
        this.neContacts.addAll(users)
        display(listData)
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder<ViewDataBinding> {
        return when (viewType) {
            CONTACT_HEADER -> BaseViewHolder(inflateView(parent, R.layout.item_rcv_user_group_header) as ItemRcvUserGroupHeaderBinding)
            CONTACT_BODY ->
                when (contactType) {
                    CONTACT_TYPE_MAIN -> BaseViewHolder(inflateView(parent, R.layout.item_contact) as ItemContactBinding)
                    CONTACT_TYPE_NEW_MESSAGE -> BaseViewHolder(inflateView(parent, R.layout.item_rcv_user_message) as ItemRcvUserMessageBinding)
                    CONTACT_TYPE_NEW_GROUP -> BaseViewHolder(inflateView(parent, R.layout.item_rcv_user_group) as ItemRcvUserGroupBinding)
                    else -> BaseViewHolder(inflateView(parent, R.layout.item_contact) as ItemContactBinding)
                }
            else -> BaseViewHolder(inflateView(parent, R.layout.item_contact) as ItemContactBinding)
        }
    }

    override fun onBindViewHolder(holder: BaseViewHolder<ViewDataBinding>, position: Int) {
        getItem(position)?.let { item ->
            with(holder) {
                when (getItemViewType(position)) {
                    CONTACT_HEADER -> bindContactHeader(this, subNeContact = item)
                    CONTACT_BODY ->
                        when (contactType) {
                            CONTACT_TYPE_MAIN -> bindContactMain(
                                this,
                                neContact = item._itemNeContact?._neContact,
                                isDivider = (layoutPosition != (itemCount - 1)),
                                position = position
                            )
                            CONTACT_TYPE_NEW_MESSAGE -> bindContactNewMessage(
                                this,
                                neContact = item._itemNeContact?._neContact,
                                isDivider = (layoutPosition != (itemCount - 1)),
                                position = position
                            )
                            CONTACT_TYPE_NEW_GROUP -> bindContactNewGroup(
                                this,
                                neContact = item._itemNeContact?._neContact,
                                isDivider = (layoutPosition != (itemCount - 1)),
                                position = position
                            )
                            CONTACT_TYPE_MEMBERS -> bindMembers(
                                this,
                                neContact = item._itemNeContact?._neContact,
                                isDivider = (layoutPosition != (itemCount - 1)),
                                position = position
                            )
                        }
                }

                binding.executePendingBindings()
            }
        }
    }

    private fun bindContactHeader(holder: BaseViewHolder<ViewDataBinding>, subNeContact: SubNeContact) {
        with(holder.binding as ItemRcvUserGroupHeaderBinding) {
            this.header = subNeContact._neHeader ?: "#"
        }
    }

    private fun bindContactMain(
        holder: BaseViewHolder<ViewDataBinding>,
        neContact: NeUser?,
        isDivider: Boolean,
        position: Int
    ) {
        getItem(position)?.let { item ->
            with(holder.binding as ItemContactBinding) {
                imgAvatar.setImageDrawable(null)
                this.item = neContact
                this.hasDivider = isDivider
                this.imgVoiceCall.isGone = neContact?.isBlocked == true
                this.imgVideoCall.isGone = neContact?.isBlocked == true
                this.imgVoiceCall.clickDebounce {
                    neContact?.let {
                        voiceCall.invoke(it)
                    }
                }
                this.imgVideoCall.clickDebounce {
                    neContact?.let {
                        videoCall.invoke(it)
                    }
                }

                holder.itemView.clickDebounce {
                    itemClick?.invoke(item, holder.itemView, position)
                }
            }
        }
    }

    private fun bindContactNewMessage(
        holder: BaseViewHolder<ViewDataBinding>,
        neContact: NeUser?,
        isDivider: Boolean,
        position: Int
    ) {
        getItem(position)?.let { item ->
            with(holder.binding as ItemRcvUserMessageBinding) {
                imgAvatar.setImageDrawable(null)
                this.item = neContact
                this.hasDivider = isDivider
                holder.itemView.clickDebounce {
                    itemClick?.invoke(item, holder.itemView, position)
                }
            }
        }
    }

    open fun bindContactNewGroup(
        holder: BaseViewHolder<ViewDataBinding>,
        neContact: NeUser?,
        isDivider: Boolean,
        position: Int
    ) {
    }

    private fun bindMembers(
        holder: BaseViewHolder<ViewDataBinding>,
        neContact: NeUser?,
        isDivider: Boolean,
        position: Int
    ) {
        getItem(position)?.let { item ->
            with(holder.binding as ItemRcvMemberBinding) {
                this.item = neContact
                this.hasDivider = isDivider
                holder.itemView.clickDebounce {
                    itemClick?.invoke(item, holder.itemView, position)
                }
            }
        }
    }

    private class ContactDiff : DiffUtil.ItemCallback<SubNeContact>() {
        override fun areItemsTheSame(oldItem: SubNeContact, newItem: SubNeContact): Boolean {
            return if (oldItem.isHeader() && newItem.isHeader()) {
                true
            } else !oldItem.isHeader() && !newItem.isHeader()
        }

        override fun areContentsTheSame(oldItem: SubNeContact, newItem: SubNeContact): Boolean {
            return if (oldItem.isHeader() && newItem.isHeader() && oldItem._neHeader == newItem._neHeader) {
                true
            } else (
                !oldItem.isHeader() && !newItem.isHeader() && oldItem._itemNeContact?._neContact?.id == newItem._itemNeContact?._neContact?.id &&
                    oldItem._itemNeContact?._neContact?.phone == newItem._itemNeContact?._neContact?.phone &&
                    oldItem._itemNeContact?._neContact?.isOnline == newItem._itemNeContact?._neContact?.isOnline &&
                    oldItem._itemNeContact?._neContact?.getDisplayName == newItem._itemNeContact?._neContact?.getDisplayName &&
                    oldItem._itemNeContact?._neContact?.getDisplayAvatar == newItem._itemNeContact?._neContact?.getDisplayAvatar
                )
        }
    }

    fun filter(filterBy: String, lifecycleScope: LifecycleCoroutineScope) {
        lifecycleScope.launch(Dispatchers.IO) {
            val termContact = neContacts.filter {
                var result: Boolean
                result = it.getDisplayName.unAccentText().contains(
                    filterBy,
                    ignoreCase = true
                ) ?: false
                if (!result) {
                    result = it.phone?.contains(filterBy, ignoreCase = true) == true
                }
                Logger.d("_InviteFriendAdapter result = $result")
                result
            }

            val newList = ContactUtils.sortContacts(listContacts = termContact)
            withContext(Dispatchers.Main) {
                emptyCallBack?.let { callback ->
                    callback(newList.isEmpty())
                }
                display(newList)
            }
        }
    }

    @Synchronized
    fun updateStatus(userStatus: NeUser) {
        getData.firstOrNull {
            it._itemNeContact?._neContact?.id == userStatus.id
        }?.apply {
            getItemUser()?._neContact?.isOnline = userStatus.isOnline ?: false
            val index = getData.indexOf(this)
            notifyItemChanged(index, this)
        }
    }
}
