/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.contact.adapter

import android.graphics.Color
import android.text.Spannable
import android.text.SpannableString
import android.text.style.ForegroundColorSpan
import android.view.View
import android.view.ViewGroup
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.LifecycleCoroutineScope
import androidx.recyclerview.widget.DiffUtil
import com.netacom.base.chat.adapter.BaseMultiTypeListSimpleAdapter
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.full.R
import com.netacom.full.databinding.ItemAddContactLocalSearchBinding
import com.netacom.full.databinding.ItemAddContactRegisterSearchBinding
import com.netacom.full.databinding.ItemInviteSearchBinding
import com.netacom.full.databinding.ItemMoreSearchBinding
import com.netacom.full.databinding.ItemRcvUserGroupHeaderBinding
import com.netacom.full.databinding.ItemResultSearchBinding
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.full.utils.SearchUtil
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.util.Constants.EMPTY
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.Locale

/**
 * Created by Tam Nguyen on 9/4/20.
 */

class RecentSearchAdapter(
    itemClick: ((SubNeItem, View, Int) -> Unit)?,
    private val contactClick: ((NeUser?, Boolean) -> Unit),
    private val groupClick: ((NeGroup?) -> Unit),
    private val inviteFriendClick: ((NeUser) -> Unit),
    private val saveRegisterContactClick: ((NeUser) -> Unit),
    private val saveLocalContact: ((NeUser) -> Unit),
    private val themeHelperImpl: ThemeHelperImpl
) : BaseMultiTypeListSimpleAdapter<RecentSearchAdapter.SubNeItem, ViewDataBinding>(itemClick, diff = DiffCallBack()) {

    companion object {
        const val TYPE_RESULT_CONTACT_SEARCH = 1
        const val TYPE_RESULT_GROUP_SEARCH = 2
        const val TYPE_HEADER_SEARCH = 3
        const val TYPE_INVITE_FRIEND = 4
        const val TYPE_ADD_CONTACT_REGISTER = 5
        const val TYPE_ADD_CONTACT_LOCAL = 6
        const val TYPE_MORE_SEARCH_CONTACT = 7

        const val TYPE_HEADER_CONTACT = 100
        const val TYPE_HEADER_GROUP = 101

        private var FILTER_SEARCH: String = ""
        private var COLOR_TEXT_HIGHLIGHT: Int = Color.RED
        private var COLOR_TEXT_NORMAL: Int = Color.BLACK

        private fun spanText(_filter: String, _content: String?): Spannable {
            _content ?: return SpannableString("")
            val span = SpannableString(_content)
            val content = _content.toLowerCase(Locale.getDefault())
            val filterKeys = SearchUtil.getFilterKeysFromFilter(_filter)
            filterKeys.forEach {
                if (it.isNotEmpty() && content.contains(it.toLowerCase(Locale.getDefault()))) {
                    val startSpan = content.indexOf(string = it, ignoreCase = true)
                    span.setSpan(
                        ForegroundColorSpan(COLOR_TEXT_HIGHLIGHT),
                        startSpan,
                        startSpan + it.length,
                        Spannable.SPAN_EXCLUSIVE_INCLUSIVE
                    )
                }
            }

            return span
        }
    }

    // private val subNeItem = mutableListOf<SubNeItem>()
    private val totalItem = mutableListOf<SubNeItem>()

    override fun bindMultiType(position: Int): Int {
        return getItem(position)?._type ?: 0
    }

    fun setData(
        listSubNeItem: List<SubNeItem>,
        filter: String = EMPTY,
        lifecycleScope: LifecycleCoroutineScope
    ) {
        lifecycleScope.launch(Dispatchers.IO) {
            FILTER_SEARCH = filter
            totalItem.clear()
            totalItem.addAll(listSubNeItem)
            val listNew = mutableListOf<SubNeItem>()
            if (listSubNeItem.isNotEmpty() && listSubNeItem[0]._type == TYPE_HEADER_SEARCH && listSubNeItem[0]._neHeader == TYPE_HEADER_CONTACT) {
                var count = 0
                listSubNeItem.forEach {
                    when {
                        it._type != TYPE_RESULT_CONTACT_SEARCH -> {
                            listNew.add(it)
                        }
                        count < 5 -> {
                            listNew.add(it)
                            count++
                        }
                        count == 5 -> {
                            listNew.add(
                                SubNeItem(
                                    _type = TYPE_MORE_SEARCH_CONTACT
                                )
                            )
                            count++
                        }
                    }
                }
            } else {
                listNew.addAll(listSubNeItem)
            }
            display(listNew)
        }
    }

    override fun onCreateViewHolder(
        parent: ViewGroup,
        viewType: Int
    ): BaseViewHolder<ViewDataBinding> {
        return when (viewType) {
            TYPE_RESULT_CONTACT_SEARCH -> BaseViewHolder(
                inflateView(parent, R.layout.item_result_search) as ItemResultSearchBinding
            )
            TYPE_RESULT_GROUP_SEARCH -> BaseViewHolder(
                inflateView(parent, R.layout.item_result_search) as ItemResultSearchBinding
            )
            TYPE_HEADER_SEARCH -> BaseViewHolder(
                inflateView(
                    parent,
                    R.layout.item_rcv_user_group_header
                ) as ItemRcvUserGroupHeaderBinding
            )
            TYPE_INVITE_FRIEND -> BaseViewHolder(
                inflateView(parent, R.layout.item_invite_search) as ItemInviteSearchBinding
            )
            TYPE_ADD_CONTACT_REGISTER -> BaseViewHolder(
                inflateView(
                    parent,
                    R.layout.item_add_contact_register_search
                ) as ItemAddContactRegisterSearchBinding
            )
            TYPE_ADD_CONTACT_LOCAL -> BaseViewHolder(
                inflateView(
                    parent,
                    R.layout.item_add_contact_local_search
                ) as ItemAddContactLocalSearchBinding
            )
            TYPE_MORE_SEARCH_CONTACT -> BaseViewHolder(
                inflateView(parent, R.layout.item_more_search) as ItemMoreSearchBinding
            )
            else -> BaseViewHolder(
                inflateView(parent, R.layout.item_result_search) as ItemResultSearchBinding
            )
        }
    }

    override fun onBindViewHolder(holder: BaseViewHolder<ViewDataBinding>, position: Int) {
        getItem(position)?.let { item ->
            with(holder) {
                when (getItemViewType(position)) {
                    TYPE_RESULT_CONTACT_SEARCH -> bindContactSearch(this, item._neUser)
                    TYPE_RESULT_GROUP_SEARCH -> bindGroupSearch(this, item._neGroup)
                    TYPE_HEADER_SEARCH -> bindHeaderSearch(this, item._neHeader)
                    TYPE_INVITE_FRIEND -> bindInviteFriend(this, item._neUser)
                    TYPE_ADD_CONTACT_REGISTER -> bindAddContactRegistered(this, item._neUser)
                    TYPE_ADD_CONTACT_LOCAL -> bindAddContactLocal(this, item._neUser)
                    TYPE_MORE_SEARCH_CONTACT -> bindSearchMore(this)
                }
                binding.executePendingBindings()
            }
        }
    }

    private fun bindContactSearch(holder: BaseViewHolder<ViewDataBinding>, neContact: NeUser?) {
        with(holder.binding as ItemResultSearchBinding) {
            neContact?.let { _neContact ->
                this.avatarUrl = _neContact.getDisplayAvatar
                this.displayName = _neContact.getDisplayName
                this.tvDisplayName.text = spanText(
                    _filter = FILTER_SEARCH,
                    _content = _neContact.getDisplayName
                )
                this.llMain.clickDebounce {
                    contactClick.invoke(_neContact, true)
                }
            }
        }
    }

    private fun bindGroupSearch(holder: BaseViewHolder<ViewDataBinding>, neGroup: NeGroup?) {
        with(holder.binding as ItemResultSearchBinding) {
            neGroup?.let { _neGroup ->
                this.avatarUrl = _neGroup.getDisplayAvatar
                this.displayName = _neGroup.getDisplayName
                this.tvDisplayName.text = spanText(
                    _filter = FILTER_SEARCH,
                    _content = _neGroup.getDisplayName
                )
                this.llMain.clickDebounce {
                    groupClick.invoke(_neGroup)
                }
            }
        }
    }

    private fun bindHeaderSearch(holder: BaseViewHolder<ViewDataBinding>, neHeader: Int?) {
        with(holder.binding as ItemRcvUserGroupHeaderBinding) {
            neHeader?.let {
                val headerBinding = if (it == TYPE_HEADER_CONTACT) {
                    holder.itemView.context.resources.getString(R.string.search_title_contact)
                } else {
                    holder.itemView.context.resources.getString(R.string.search_title_conversation)
                }

                this.header = headerBinding
            }
        }
    }

    private fun bindInviteFriend(holder: BaseViewHolder<ViewDataBinding>, neContact: NeUser?) {
        with(holder.binding as ItemInviteSearchBinding) {
            neContact?.let { _neContact ->
                this.displayName = _neContact.getDisplayName
                this.tvDisplayName.text = spanText(
                    _filter = FILTER_SEARCH,
                    _content = _neContact.getDisplayName
                )
                this.tvInvite.clickDebounce {
                    inviteFriendClick.invoke(_neContact)
                }

                themeHelperImpl.setButtonBackgroundAndText(tvInvite)
                themeHelperImpl.setThemeColorForSubBackground(tvInvite)
            }
        }
    }

    private fun bindAddContactRegistered(
        holder: BaseViewHolder<ViewDataBinding>,
        neContact: NeUser?
    ) {
        with(holder.binding as ItemAddContactRegisterSearchBinding) {
            neContact?.let { _neContact ->
                this.avatarUrl =
                    if (_neContact.avatar.isNullOrEmpty()) "" else _neContact.getDisplayAvatar
                this.displayName = _neContact.getDisplayName
                this.llMain.clickDebounce {
                    contactClick.invoke(_neContact, false)
                }
                this.tvSave.clickDebounce {
                    saveRegisterContactClick.invoke(_neContact)
                }

                themeHelperImpl.setButtonBackgroundAndText(tvSave)
                themeHelperImpl.setThemeColorForSubBackground(tvSave)
            }
        }
    }

    private fun bindAddContactLocal(holder: BaseViewHolder<ViewDataBinding>, neContact: NeUser?) {
        with(holder.binding as ItemAddContactLocalSearchBinding) {
            neContact?.let { _neContact ->
                this.displayName = "${
                holder.itemView.context.resources.getString(
                    R.string.contact_search_save
                )
                } ${_neContact.getDisplayName}"
                this.llMain.clickDebounce {
                    saveLocalContact.invoke(_neContact)
                }
            }

            themeHelperImpl.setThemeColorForViews(ivAddContact, tvName)
        }
    }

    private fun bindSearchMore(holder: BaseViewHolder<ViewDataBinding>) {
        with(holder.binding as ItemMoreSearchBinding) {
            this.tvViewMore.clickDebounce {
                display(totalItem)
            }
        }
    }

    override fun getItemCount(): Int {
        return super.getItemCount()
    }

    class SubNeItem(
        val _neUser: NeUser? = null,
        val _neGroup: NeGroup? = null,
        val _neHeader: Int? = null,
        val _type: Int
    )

    private class DiffCallBack : DiffUtil.ItemCallback<SubNeItem>() {
        override fun areItemsTheSame(oldItem: SubNeItem, newItem: SubNeItem): Boolean {
            return oldItem._neHeader == newItem._neHeader
        }

        override fun areContentsTheSame(oldItem: SubNeItem, newItem: SubNeItem): Boolean {
            return oldItem.equals(newItem)
        }
    }
}
