package com.netacom.full.ui.main.group

import android.text.Editable
import android.text.TextWatcher
import android.view.animation.Animation
import android.view.animation.Transformation
import android.widget.LinearLayout
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.fragment.app.viewModels
import androidx.lifecycle.lifecycleScope
import androidx.navigation.fragment.navArgs
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKBottomDialog
import com.netacom.full.databinding.ViewAddMemberPopupBinding
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.contact.ContactViewModel
import com.netacom.full.ui.main.group.adapter.AddMemberAdapter
import com.netacom.full.ui.main.group.adapter.ChooseOwnerAdapter
import com.netacom.full.ui.main.group.adapter.CreateGroupAdapter
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.network.model.response.RegisterContact
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@AndroidEntryPoint
class AddMemberDialog : BaseSDKBottomDialog<ViewAddMemberPopupBinding, GroupViewModel>(
    R.layout.view_add_member_popup,
    GroupViewModel::class.java
) {
    private lateinit var contactAdapter: CreateGroupAdapter

    private lateinit var memberAdapter: AddMemberAdapter

    private lateinit var neGroup: NeGroup

    private var userOwner: NeUser? = null

    override fun setLayoutHeight(): Int = DialogDef.LAYOUT_FULL

    override fun setViewModel(): Int = BR.viewModel

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()

    private val contactViewModel: ContactViewModel by viewModels()

    private var uidAvatar: String? = ""

    val args: AddMemberDialogArgs by navArgs()

    private var isShowMemberChecked = false

    override fun initViews() {
        initGroupAdapter()
        initAddMemberAdapter()
        setupDoneClickBtn()
        filterSearch()
    }

    private fun initGroupAdapter() {
        neGroup = args.neGroup
        if (args.isChangeOwner) {
            contactAdapter = ChooseOwnerAdapter(
                ({ _, _, _ -> }),
                { neContact: NeUser, _: Int, isChecked: Boolean ->
                    userOwner = if (isChecked) {
                        neContact
                    } else {
                        null
                    }
                },
                themeHelperImpl,
            )
            if (neGroup.type == GroupType.GROUP_TYPE_CHANNEL) {
                binding.tvTitle.text = StringUtils.getString(R.string.choose_admin)
            } else {
                binding.tvTitle.text = StringUtils.getString(R.string.choose_owner)
            }
            binding.icClose.isGone = true
            binding.tvChangeOwner.isVisible = true
        } else {
            contactAdapter = CreateGroupAdapter(
                (
                    { _, _, _ ->
                    }
                    ),
                { neContact: NeUser, _: Int, isCheck: Boolean ->
                    if (isCheck) {
                        addMember(neContact)
                    } else {
                        removeMember(neContact)
                    }
                    checkMembers(contactAdapter.getListDataChecked())
                },
                themeHelperImpl,
                emptyCallBack = { isEmpty ->
                    if (isEmpty) {
                        binding.layoutEmpty.ctlEmptyContainer.isVisible = true
                        binding.layoutEmpty.btnEmpty.clickDebounce {
                            viewModel.showAddContactDialog(null, null)
                        }
                    }
                }
            )
            if (neGroup.type == GroupType.GROUP_TYPE_CHANNEL) {
                binding.tvTitle.text = StringUtils.getString(R.string.text_add_followers)
            }
        }
        binding.recyclerView.adapter = contactAdapter

        if (args.isChangeOwner) {
            neGroup.members?.let {

                val listContactNotYou = it.filterNot {
                    it.id == mainSdkViewModel.getUserId
                }
                contactAdapter.setData(
                    neContacts = listContactNotYou,
                    lifecycleScope = lifecycleScope
                )
            }
        } else {
            mainSdkViewModel.listContact.observeOnce { listContact ->
                /* listContact.toMutableList().removeAll(it)*/
                val listContactNotInGroup = neGroup.members?.let {
                    listContact.filter { itemContact ->
                        it.none { itemGroup ->
                            itemGroup.id == itemContact.id
                        }
                    }.filter {
                        it.isRegistered ?: false
                    }.map {
                        it
                    }.toList()
                }
                listContactNotInGroup?.let {
                    contactAdapter.setData(
                        neContacts = it,
                        lifecycleScope = lifecycleScope
                    )
                }
            }

            mainSdkViewModel.syncContact(SyncType.LIST_SYNC_SERVER)
        }
    }

    private fun initAddMemberAdapter() {
        memberAdapter = AddMemberAdapter(
            { item, _, _ -> },
            { item, _, _ ->
                removeMember(item)
                contactAdapter.unCheckItem(item)
                checkMembers(contactAdapter.getListDataChecked())
            },
            themeHelperImpl
        )
        binding.rcvNewMember.apply {
            layoutManager = LinearLayoutManager(context, RecyclerView.HORIZONTAL, false)
            adapter = memberAdapter
        }
    }

    private fun setupDoneClickBtn() {
        binding.tvDone.clickDebounce {
            args.neGroup.apply {
                viewModel.addMemberToGroup(this, memberAdapter.getAll())
            }
        }
    }

    override fun initData() {
        binding.icClose.clickDebounce {
            dismiss()
        }

        binding.tvChangeOwner.clickDebounce {
            userOwner?.let {
                viewModel.changeOwner(neGroup, it.id.toString(), willLeaveGroup = args.willLeaveGroup)
            } ?: dismiss()
        }
    }

    override fun setupTheme() {
        themeHelperImpl.setThemeColorForViews(binding.tvDone, binding.tvChangeOwner, binding.layoutEmpty.btnEmpty, binding.layoutEmpty.tvEmpty)
    }

    private fun addMember(neUser: NeUser) {
        memberAdapter.addMember(neUser)
        binding.rcvNewMember.smoothScrollToPosition(memberAdapter.itemCount)
    }

    private fun removeMember(neUser: NeUser) {
        memberAdapter.removeMember(neUser)
    }

    private fun checkMembers(members: List<NeUser?>) {
        if (members.isNotEmpty() && !isShowMemberChecked) {
            showMemberCheckedAnimation()
        } else if (members.isEmpty()) {
            hideMemberCheckedAnimation()
        }
    }

    private fun showMemberCheckedAnimation() {
        isShowMemberChecked = true
        val viewHeight = AppUtils.dpToPx(context = requireContext(), dp = 88f)
        val newMargin = -AppUtils.dpToPx(context = requireContext(), dp = 24f)

        (binding.flBottom.layoutParams as? LinearLayout.LayoutParams)?.let {
            val a: Animation = object : Animation() {
                override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
                    it.bottomMargin = (viewHeight * (interpolatedTime - 1)).toInt() + newMargin
                    binding.flBottom.layoutParams = it
                }
            }
            a.duration = 300 // in ms
            binding.flBottom.startAnimation(a)
        }
    }

    private fun hideMemberCheckedAnimation() {
        isShowMemberChecked = false

        val viewHeight = AppUtils.dpToPx(context = requireContext(), dp = 88f)
        val newMargin = -AppUtils.dpToPx(context = requireContext(), dp = 24f)

        (binding.flBottom.layoutParams as? LinearLayout.LayoutParams)?.let {
            val a: Animation = object : Animation() {
                override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
                    it.bottomMargin = -(viewHeight * interpolatedTime).toInt() + newMargin
                    binding.flBottom.layoutParams = it
                }
            }
            a.duration = 300 // in ms
            binding.flBottom.startAnimation(a)
        }
    }

    private fun filterSearch() {
        binding.editSearch.addTextChangedListener(
            object : TextWatcher {
                private var searchFor = Constants.EMPTY
                override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
                }

                override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                    val searchText = s.toString().trim()
                    if (searchText == searchFor) {
                        return
                    }

                    searchFor = searchText

                    lifecycleScope.launch {
                        delay(500) // debounce timeOut
                        if (searchText != searchFor) {
                            return@launch
                        }

                        if (isValidPhone(searchFor)) {
                            searchContactByPhone(searchFor)
                        } else {
                            contactAdapter.filter(
                                filterBy = searchFor,
                                lifecycleScope = lifecycleScope
                            )
                        }
                    }
                }

                override fun afterTextChanged(p0: Editable?) {
                }
            }
        )
    }

    private fun searchContactByPhone(phoneNumber: String) {
        contactViewModel.checkContactByPhoneNumber(
            phoneNumber = phoneNumber,
            object : CallbackResult<RegisterContact> {
                override fun callBackSuccess(result: RegisterContact) {
                    val user = NeUser().apply {
                        id = result.id
                        username = result.fullName
                        phone = result.phone
                        email = result.email
                    }
                    contactAdapter.setData(
                        neContacts = listOf(user),
                        lifecycleScope = lifecycleScope
                    )
                }

                override fun callBackError(error: String?) {
                    contactAdapter.setData(
                        neContacts = listOf(),
                        lifecycleScope = lifecycleScope
                    )
                }
            }
        )
    }

    private fun isValidPhone(number: String): Boolean {
        return number.length >= 10 && android.text.TextUtils.isDigitsOnly(number)
    }
}
