package com.netacom.full.ui.main.group

import androidx.activity.addCallback
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.fragment.app.viewModels
import androidx.navigation.fragment.navArgs
import com.netacom.base.chat.android_utils.LanguageUtils
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.DateTimeUtils
import com.netacom.base.chat.util.isNull
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseCoreCallCameraFragment
import com.netacom.full.basechat.BaseSDKViewModel
import com.netacom.full.databinding.FragmentChatInfoBinding
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.call.CallLogViewModel
import com.netacom.full.ui.main.call.adapter.CallLogAdapter
import com.netacom.full.ui.main.chat.adapter.MemberChatInfoAdapter
import com.netacom.full.ui.main.contact.ContactUtils
import com.netacom.full.ui.main.group.adapter.GalleryAdapter
import com.netacom.full.utils.DialogUtil
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_FILE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_IMAGE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_VIDEO
import com.netacom.lite.entity.ui.NeChatInfo
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint

@AndroidEntryPoint
class ChatInfoFragment : BaseCoreCallCameraFragment<FragmentChatInfoBinding, GroupViewModel>(
    R.layout.fragment_chat_info,
    GroupViewModel::class.java
) {
    companion object {
        const val SELECTED_IMAGE_POSITION = 0
        const val SELECTED_VIDEO_POSITION = 1
        const val SELECTED_FILE_POSITION = 2
        val TAG = ChatInfoFragment::class.java.simpleName
    }

    override fun setViewModel(): Int = BR.viewModel

    private val args: ChatInfoFragmentArgs by navArgs()

    private val callLogViewModel: CallLogViewModel by viewModels()

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()

    private lateinit var neGroup: NeGroup

    private var isMissCall = false

    private var isFromCallLog = false

    private var millisCallLog: Long = 0

    private lateinit var galleryAdapter: GalleryAdapter

    private var isFirstTime = true

    private var galleryFragment: GalleryFragment? = null

    override fun initViews() {
        with(binding) {
            args.callInfo?.let {
                neGroup = it.neGroup
                initConfig(this)
                setupGalleryFragment()
                checkNeGroup(neGroup)
                checkNotification(neGroup)

                isFromCallLog = it.isFromCallLog
                if (isFromCallLog) {
                    isMissCall = it.isMissCall
                    millisCallLog = it.date
                    setupCallLogInfo()
                }
            }

            toolbar.setOnLeftClickListener {
                viewModel.handleBackFm(neGroup)
            }

            llNotify.clickDebounce {
                viewModel.groupActionNotification(neGroup = neGroup)
            }
        }
    }

    override fun initData() {
        viewModel.updateGroup.observeOnce { _neGroup ->
            // only change group when match id
            if (neGroup.id == _neGroup.id) {
                neGroup = _neGroup
                checkNeGroup(_neGroup)
                checkNotification(_neGroup)
            }
        }

        binding.toolbar.setOnRightClickListener {
            binding.item?.let {
                viewModel.goToEditGroupOrPartnerInfo(it, neGroup)
            }
        }

        binding.imageViewAvatar.clickDebounce {
            binding.item?.let {
                viewModel.displayImage(it.getDisplayAvatar)
            }
        }

        galleryAdapter = GalleryAdapter(
            { item, _, _ ->
                when (item.mediaType) {
                    MESSAGE_TYPE_IMAGE, MESSAGE_TYPE_VIDEO -> {
                        viewModel.mediaData.removeObservers(viewLifecycleOwner)
                        viewModel.goToMediaView(item, neGroup)
                    }

                    MESSAGE_TYPE_FILE -> {
                        item.attachments?.file?.let {
                            mainSdkViewModel.handleFile(neGroup.isSecretChat(), item.messId, it, requireContext())
                        }
                    }
                }
            },
            MESSAGE_TYPE_IMAGE,
            themeHelperImpl,
            neGroup.isSecretChat()
        )

        mainSdkViewModel.deleteSecretGroup.observe {
            if (it.toString() == neGroup.id) {
                binding.imgVoiceCall.isGone = true
                binding.imgVideoCall.isGone = true
            }
        }
        binding.llFindMessage.clickDebounce {
            showSnackBar(R.string.common_function_is_building)
        }

        binding.shareContact.clickDebounce {
            if (neGroup.type == GroupType.GROUP_TYPE_GROUP) {
                neGroup.id?.toLongOrNull()?.let { groupId ->
                    if (neGroup.owner?.id == preferencesHelperImpl.getUserId) {
                        mainSdkViewModel.createDeepLink(groupId)
                    } else {
                        mainSdkViewModel.getDeepLink(groupId)
                    }
                }
            } else {
                showSnackBar(R.string.common_function_is_building)
            }
        }

        requireActivity().onBackPressedDispatcher.addCallback(this) {
            viewModel.handleBackFm(neGroup)
        }

        binding.llChangeBackground.clickDebounce {
            viewModel.handleBackFm(neGroup, isChangeBackground = true)
        }

        mainSdkViewModel.createDeepLink.observe { deepLink ->
            deepLink?.let { _deepLink ->
                if (_deepLink.isNotEmpty()) {
                    if (isFirstTime) {
                        isFirstTime = false
                    } else {
                        AppUtils.copyTextToClipboard(context = requireContext(), content = deepLink)
                    }
                } else if (_deepLink.isEmpty()) {
                    noticeNoDeepLink()
                }
            }
        }
    }

    private fun noticeNoDeepLink() {
        if (preferencesHelperImpl.getUserId == neGroup.owner?.id) {
            showSnackBar(R.string.message_cant_create_deeplink)
        } else {
            showSnackBar(R.string.message_contact_group_owner_for_deeplink)
        }
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(
                ivStartChat, imgVoiceCall, imgVideoCall, ivSearchMessage,
                ivCreateGroupOrAddMember, ivShare, ivSecretChat, ivNotify, textViewTitle, toolbar.getButtonLeft()!!,
                txtTimeAgo, tvAddContactPhoneBook, ivGroupQrCode, ivChangeBackground
            )
        }
    }

    override fun syncEvent() {
        callEvent()
    }

    private fun initConfig(binding: FragmentChatInfoBinding) {
        if (viewModel.checkHideInfo) {
            binding.llFindMessage.isGone = true
            binding.createGroupOrAddMember.isGone = true
            binding.shareContact.isGone = true
            binding.llSecretMessage.isGone = true
        }
    }

    private fun checkNeGroup(neGroup: NeGroup) {
        when (neGroup.type) {
            GroupType.GROUP_TYPE_GROUP, GroupType.GROUP_TYPE_CHANNEL -> {
                setupGroupInfo(neGroup)
            }
            GroupType.GROUP_TYPE_PUBLIC, GroupType.GROUP_TYPE_PRIVATE -> {
                neGroup.members?.firstOrNull { it ->
                    it.id != viewModel.getUserId
                }?.let { user ->
                    Logger.e("checkHidePhone==" + viewModel.checkHidePhone)
                    if (viewModel.checkHidePhone) {
                        user.phone = "**********"
                    }
                    setupUserInfo(user)
                }
            }
            else -> {
            }
        }
    }

    private fun setupGroupInfo(neGroup: NeGroup) {
        if (neGroup.owner?.id != mainSdkViewModel.getUserId) {
            binding.toolbar.hideActionRight()
        }
        with(binding) {
            ivCreateGroupOrAddMember.setImageResource(R.drawable.ic_add_member)
            layoutMakeFriend.isGone = true
            when (neGroup.type) {
                GroupType.GROUP_TYPE_GROUP -> {
                    tvCreateGroupOrAddMember.text = getString(R.string.add_member)
                    tvShareContactOrGroup.text = getString(R.string.message_link_sharing_group)
                    tvLeaveOrBlock.text = getString(R.string.leave_group)
                    tvMemberCount.text = String.format(
                        getString(R.string.text_member),
                        neGroup.members?.size
                    )
                    llChangeBackground.isVisible = viewModel.isLeaderInGroup(neGroup)
                    binding.llGroupQrCode.clickDebounce {
                        neGroup.id?.toLongOrNull()?.let { groupId ->
                            if (neGroup.owner?.id == preferencesHelperImpl.getUserId) {
                                mainSdkViewModel.createDeepLink(
                                    groupId,
                                    object : CallbackResult<String?> {
                                        override fun callBackSuccess(result: String?) {
                                            if (result.isNull) {
                                                noticeNoDeepLink()
                                            } else {
                                                viewModel.openQrCodeDialog(neGroup, result)
                                            }
                                        }

                                        override fun callBackError(error: String?) {
                                        }
                                    }
                                )
                            } else {
                                mainSdkViewModel.getDeepLink(
                                    groupId,
                                    object : CallbackResult<String?> {
                                        override fun callBackSuccess(result: String?) {
                                            if (result.isNull) {
                                                noticeNoDeepLink()
                                            } else {
                                                viewModel.openQrCodeDialog(neGroup, result)
                                            }
                                        }

                                        override fun callBackError(error: String?) {
                                        }
                                    }
                                )
                            }
                        }
                    }
                }
                GroupType.GROUP_TYPE_CHANNEL -> {
                    tvGroupDescription.isVisible = true
                    tvGroupDescription.text = neGroup.description
                    llFindMessage.isGone = true
                    tvMemberCount.text = String.format(
                        getString(R.string.text_followers),
                        neGroup.members?.size
                    )
                    if (viewModel.isLeaderInGroup(neGroup)) {
                        tvCreateGroupOrAddMember.text = getString(R.string.text_add_followers)
                        tvShareContactOrGroup.text = getString(R.string.text_message_link_sharing_channel)
                        llBlockOrLeave.isGone = true
                    } else {
                        createGroupOrAddMember.isGone = true
                        shareContact.isGone = true
                        tvLeaveOrBlock.text = getString(R.string.text_unfollow)
                    }
                    llChangeBackground.isVisible = viewModel.isLeaderInGroup(neGroup)
                }
            }
            item = NeChatInfo(neGroup)
        }
        val leaderId = neGroup.owner?.id
        val memberAdapter = MemberChatInfoAdapter(
            { user, _, _ ->
                run {
                    (viewModel as? BaseSDKViewModel)?.apply {
                        startCallingScreen.removeObservers(viewLifecycleOwner)
                    }
                    galleryFragment?.setIsShowMember(true)
                    viewModel.showSingleMember(neUser = user)
                }
            },
            {
                showSnackBar(R.string.text_message_this_is_you)
            },
            leaderId ?: 0,
            themeHelperImpl,
            { user ->
                viewModel.removeMember(user, neGroup)
            },
            preferencesHelperImpl,
            isChannel = neGroup.type == GroupType.GROUP_TYPE_CHANNEL
        )
        // add only 3 member
        neGroup.members?.let { listMember ->

            val threeMemberInGroup: MutableList<NeUser> = neGroup.owner?.let { owner ->
                val leader = listMember.firstOrNull { member ->
                    member.id == owner.id
                }
                leader?.let {
                    mutableListOf(
                        leader
                    )
                } ?: mutableListOf()
            } ?: mutableListOf()

            threeMemberInGroup.addAll(
                listMember.filter {
                    it.id != leaderId
                }.take(2).toList()
            )

            memberAdapter.display(threeMemberInGroup)

            // if list group only > 3 member => show text view "see all"
            binding.tvViewAll.isVisible = listMember.size > Constants.MEMBER_SHOW_GROUP_INFO
        }
        binding.rcvMember.apply {
            adapter = memberAdapter
        }

        binding.createGroupOrAddMember.clickDebounce {
            viewModel.showAddMemberDialog(neGroup)
        }

        binding.llBlockOrLeave.clickDebounce {
            DialogUtil.showMessage(
                context = requireContext(),
                title = R.string.leave_group,
                message = R.string.leave_group_confirm,
                okLabel = R.string.leave_group,
                cancelLabel = R.string.close,
                okFunc = {
                    viewModel.leaveGroupOrDeleteConversation(neGroup)
                },
                cancelFunc = {
                },
                themeHelperImpl = themeHelperImpl
            )
        }

        binding.tvViewAll.clickDebounce {
            viewModel.showMembersDialog(neGroup, leaderId ?: 0)
        }
    }

    private fun setupUserInfo(neUser: NeUser) {
        with(binding) {
            // Action for all
            neUser.let {
                item = NeChatInfo(it)
            }

            ivStartChat.clickDebounce {
                this@ChatInfoFragment.viewModel.startOneByOneChat(neUser)
            }

            imgVideoCall.clickDebounce {
                neUser.id?.let { id -> checkCall(id, isVideoEnable = true) }
            }

            imgVoiceCall.clickDebounce {
                neUser.id?.let { id -> checkCall(id, isVideoEnable = false) }
            }

            checkBlockUser(neGroup)

            llBlockOrLeave.clickDebounce {
                // ToastUtils.showShort(getString(R.string.common_function_is_building))
                if (neGroup.blockers != null) {
                    neGroup.blockers?.let {
                        if (it.isNotEmpty()) {
                            if (it[0].id == preferencesHelperImpl.getUserId) {
                                DialogUtil.show(
                                    requireContext(),
                                    messageRes = R.string.cannot_block_message,
                                    okLabel = R.string.popup_confirm_logout_ok,
                                    themeHelperImpl = themeHelperImpl
                                )
                            } else {
                                val titleUnblock = getString(R.string.unblock_alert_title, neUser.getDisplayName)
                                val messageUnblock = getString(R.string.unblock_alert_message, neUser.getDisplayName)
                                DialogUtil.show(
                                    context = requireContext(),
                                    titleStr = titleUnblock,
                                    messageStr = messageUnblock,
                                    cancelLabel = R.string.str_cancel,
                                    okLabel = R.string.unblock_alert_button,
                                    okFunc = {
                                        neUser.id?.let { blockIds ->
                                            viewModel.unblockUser(
                                                neGroup.id?.toLongOrNull() ?: 0,
                                                blockIds,
                                                object : CallbackResult<NeGroup> {
                                                    override fun callBackSuccess(neGroup: NeGroup) {
                                                        showSnackBar(R.string.unblock_user_success)
                                                    }

                                                    override fun callBackError(error: String?) {
                                                        showSnackBar(R.string.unblock_user_unsuccess)
                                                    }
                                                }
                                            )
                                        }
                                    },
                                    themeHelperImpl = themeHelperImpl
                                )
                            }
                        } else {
                            showBlockPopup(neUser)
                        }
                    }
                } else {
                    showBlockPopup(neUser)
                }
            }

            // Show UI for friend or not
            if (neUser.isDeleted) {
                binding.toolbar.hideActionRight()
                binding.layoutContact.isGone = true
                binding.layoutMakeFriend.isVisible = true
            } else {
                mainSdkViewModel.listContact.value?.peekContent()?.let {
                    neUser.id?.let { id ->
                        when {
                            ContactUtils.hasAddedFriend(id, it) -> {
                                binding.layoutContact.isVisible = true
                                binding.layoutMakeFriend.isGone = true

                                ivCreateGroupOrAddMember.setImageResource(R.drawable.ic_create_group_info)
                                tvCreateGroupOrAddMember.text = StringUtils.getString(R.string.message_create_group)
                                tvShareContactOrGroup.text = StringUtils.getString(R.string.message_share_contact)

                                if (neGroup.isSecretChat()) {
                                    tvSwitchSecretOrNormal.text = StringUtils.getString(R.string.text_message_start_normal_message)
                                    ivSecretChat.setImageResource(R.drawable.ic_chat_blue)
                                }

                                // binding.item = NeChatInfo(it.members?.get(0))

                                createGroupOrAddMember.clickDebounce {
                                    this@ChatInfoFragment.viewModel.openCreateGroupDialog(neUser)
                                }

                                llSecretMessage.clickDebounce {
                                    if (neGroup.isSecretChat()) {
                                        this@ChatInfoFragment.viewModel.startOneByOneChat(neUser)
                                    } else {
                                        mainSdkViewModel.startSecretChat(neUser)
                                    }
                                }
                            }
                            else -> {
                                binding.toolbar.hideActionRight()
                                binding.layoutContact.isGone = true
                                binding.layoutMakeFriend.isVisible = true
                            }
                        }
                    }
                }
            }

            binding.layoutMakeFriend.clickDebounce {
                viewModel.showAddContactDialog(neUser, TAG)
            }
        }
    }

    private fun checkNotification(neGroup: NeGroup) {
        binding.isMute = neGroup.isMute
    }

    private fun checkBlockUser(neGroup: NeGroup) {
        if (!viewModel.checkGroupExitsInDb(neGroup)) {
            binding.llBlockOrLeave.isGone = true
        }
        if (neGroup.blockers != null) {
            neGroup.blockers?.let {
                if (it.isNotEmpty()) {
                    binding.llActionCall.isGone = true
                    if (it[0].id != preferencesHelperImpl.getUserId) {
                        binding.tvLeaveOrBlock.text = getString(R.string.chat_un_block)
                    } else {
                        binding.tvLeaveOrBlock.text = getString(R.string.chat_block_user)
                    }
                } else {
                    binding.llActionCall.isVisible = true
                    binding.tvLeaveOrBlock.text = getString(R.string.chat_block_user)
                }
            }
        } else {
            binding.llActionCall.isVisible = true
            binding.tvLeaveOrBlock.text = getString(R.string.chat_block_user)
        }
    }

    private fun setupCallLogInfo() {
        with(binding) {
            llCallLog.isVisible = true

            tvTimeCall.text = DateTimeUtils.ParseDateTimeToString.parse(
                millis = millisCallLog.run {
                    if (millisCallLog.toString().length > 10) {
                        millisCallLog.toString().substring(0, 10).toLong() * 1000
                    } else {
                        millisCallLog * 1000
                    }
                },
                format = DateTimeUtils.ParseDateTimeToString.FULL_DATE_TIME_FORMAT_PATTERN_NO_GET_HOUR_MIN
            )

            val neUser: NeUser? = neGroup.members?.find { it ->
                it.id != viewModel.getUserId
            }

            neUser?.id.let { callLogViewModel.getCallLogsByUser(it.toString(), millisCallLog) }
            callLogViewModel.callLogsByUser.observeOnce {
                viewModel.getSortedCallLogsByType(it, isMissCall)
            }
            viewModel.callLogsSortedByUser.observeOnce {
                if (it.isNotEmpty()) {
                    if (it.size > Constants.CALL_LOGS_SHOW_INFO) {
                        tvViewAllCallLog.isVisible = true
                        viewSeeAll.isVisible = true

                        tvViewAllCallLog.clickDebounce {
                            viewModel.showCallLogByUser()
                        }

                        binding.rcvCallLogs.adapter = CallLogAdapter(
                            viewModel.getUser,
                            languageApp = LanguageUtils.getLanguage(),
                            true,
                            themeHelperImpl,
                            {},
                            {},
                            { item, _, _ ->
                                Logger.d("CallLogAdapter Item Click")
                            }
                        ).apply {
                            display(it.subList(0, Constants.CALL_LOGS_SHOW_INFO))
                        }
                    } else {
                        binding.tvViewAllCallLog.isGone = true
                        binding.viewSeeAll.isGone = true
                        binding.rcvCallLogs.adapter = CallLogAdapter(
                            viewModel.getUser,
                            languageApp = LanguageUtils.getLanguage(),
                            true,
                            themeHelperImpl,
                            {},
                            {},
                            { item, _, _ ->
                                Logger.d("CallLogAdapter Item Click")
                            }
                        ).apply {
                            display(it)
                        }
                    }
                } else llCallLog.isGone = true
            }
        }
    }

    override fun onResume() {
        super.onResume()
        galleryFragment?.setIsShowMember(false)
    }

    private fun showBlockPopup(blockUser: NeUser) {
        val titleBlock = getString(R.string.block_alert_title, blockUser.getDisplayName)
        val messageBlock = getString(R.string.block_alert_message, blockUser.getDisplayName)
        DialogUtil.show(
            context = requireContext(),
            titleStr = titleBlock,
            messageStr = messageBlock,
            cancelLabel = R.string.str_cancel,
            okLabel = R.string.block_alert_button,
            okFunc = {
                blockUser.id?.let { blockIds ->
                    viewModel.blockUser(
                        neGroup,
                        blockIds,
                        object : CallbackResult<Boolean> {
                            override fun callBackSuccess(result: Boolean) {
                                showSnackBar(R.string.block_user_success)
                            }

                            override fun callBackError(error: String?) {
                                showSnackBar(R.string.block_user_unsuccess)
                            }
                        }
                    )
                    binding.llChangeBackground.isVisible = viewModel.isLeaderInGroup(neGroup)
                }
            },
            themeHelperImpl = themeHelperImpl
        )
    }

    private fun setupGalleryFragment() {
        if (galleryFragment == null) {
            galleryFragment = GalleryFragment().apply {
                this.setGroup(neGroup)
                this.setIsViewMedia(false)
                this.setIsViewSingle(false)
            }
            childFragmentManager.beginTransaction().add(R.id.flGallery, galleryFragment!!).commit()
        }
    }
}
