package com.netacom.full.ui.main.group

import android.text.Editable
import android.text.TextWatcher
import android.view.animation.Animation
import android.view.animation.Transformation
import android.widget.LinearLayout
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import androidx.navigation.fragment.navArgs
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.android_utils.UriUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.base.chat.imageloader.loadAvatar
import com.netacom.base.chat.logger.Logger
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseCoreCallCameraBottomDialog
import com.netacom.full.databinding.ViewNewGroupPopupBinding
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.group.adapter.AddMemberAdapter
import com.netacom.full.ui.main.group.adapter.CreateGroupAdapter
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@AndroidEntryPoint
class CreateGroupDialog : BaseCoreCallCameraBottomDialog<ViewNewGroupPopupBinding, GroupViewModel>(R.layout.view_new_group_popup, GroupViewModel::class.java) {

    private lateinit var contactAdapter: CreateGroupAdapter

    private lateinit var memberAdapter: AddMemberAdapter

    override fun setLayoutHeight(): Int = DialogDef.LAYOUT_FULL

    override fun setViewModel(): Int = BR.viewModel

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()

    private val args: CreateGroupDialogArgs by navArgs()
    private var isShowMemberChecked = false
    private var avatarFile: String? = null

    override fun initViews() {
        initGroupAdapter()
        initAddMemberAdapter()
        setupDoneClickBtn()
        filterSearch()
    }

    private fun initGroupAdapter() {
        contactAdapter = CreateGroupAdapter(
            (
                { _, _, _ ->
                }
                ),
            { neContact: NeUser, _: Int, isCheck: Boolean ->

                if (isCheck) {
                    checkMembers(addMember(neContact))
                } else {
                    checkMembers(removeMember(neContact))
                }
            },
            themeHelperImpl,
            emptyCallBack = { isEmpty ->
                if (isEmpty) {
                    binding.layoutEmpty.ctlEmptyContainer.isVisible = true
                    binding.layoutEmpty.btnEmpty.clickDebounce {
                        viewModel.showAddContactDialog(null, null)
                    }
                }
            }
        )
        if (args.isChannel) {
            binding.tvNewGroupTitle.text = StringUtils.getString(R.string.text_create_new_channel)
        }

        binding.recyclerView.adapter = contactAdapter

        mainSdkViewModel.listContact.value?.let {
            contactAdapter.setData(
                neContacts = it.peekContent(),
                lifecycleScope = lifecycleScope
            )
            lifecycleScope.launch(Dispatchers.Main) {
                delay(100)
                args.neUser?.let { user ->
                    contactAdapter.checkItem(user)
                }
            }
        } ?: kotlin.run {
            mainSdkViewModel.listContact.observeOnce {
                contactAdapter.setData(
                    it,
                    lifecycleScope = lifecycleScope
                )
                lifecycleScope.launch(Dispatchers.Main) {
                    delay(100)
                    args.neUser?.let { user ->
                        contactAdapter.checkItem(user)
                    }
                }
            }
            mainSdkViewModel.syncContact(SyncType.LIST_SYNC_DB)
        }
    }

    private fun initAddMemberAdapter() {
        memberAdapter = AddMemberAdapter(
            { item, _, _ -> },
            // on remove item from member adapter
            { item, _, _ ->

                contactAdapter.unCheckItem(item)
                checkMembers(removeMember(item))
            },
            themeHelperImpl
        )

        binding.rcvNewMember.apply {
            layoutManager = LinearLayoutManager(context, RecyclerView.HORIZONTAL, false)
            adapter = memberAdapter
        }
    }

    private fun setupDoneClickBtn() {
        binding.tvDone.clickDebounce {
            Logger.d("member adapter = ${memberAdapter.getAll()}")
            if (memberAdapter.getAll().isNotEmpty()) {
                val ids = memberAdapter.getAll().map {
                    it.id ?: 0L
                }
                Logger.d("member adapter ids = $ids")
                viewModel.startGroupChat(
                    listIDs = ids,
                    avatarPath = avatarFile ?: "",
                    groupName = binding.groupName.text.toString().trim(),
                    isChannel = args.isChannel,
                    callbackResult = object : CallbackResult<Boolean> {
                        override fun callBackError(error: String?) {
                            error?.let {
                                CoroutineScope(Dispatchers.Main).launch {
                                    showCustomError(binding.tvError, it)
                                }
                            }
                        }

                        override fun callBackSuccess(result: Boolean) {
                            Logger.d("startGroupChat callBackSuccess")
                        }
                    }
                )
            }
        }
    }

    override fun initData() {
        capture.observeOnce { result ->
            result.let {
                viewModel.postImage(it)
            }
        }
        binding.icClose.clickDebounce {
            dismiss()
        }

        binding.imgChooseAvatar.clickDebounce {
            checkPermissionStorage(
                {
                    openSelectMediaDialog()
                },
                {
                }
            )
        }

        viewModel.editPhoto.observeOnce {
            avatarFile = UriUtils.uri2File(it).absolutePath
            binding.imgChooseAvatar.imageTintList = null
            binding.imgChooseAvatar.loadAvatar(it)
        }
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(imgChooseAvatar, icClose, tvDone, layoutEmpty.btnEmpty, layoutEmpty.tvEmpty)
            themeHelperImpl.setThemeColorForSubBackground(imgChooseAvatar)
            themeHelperImpl.setButtonBackgroundAndText(layoutEmpty.btnEmpty)
        }
    }

    private fun addMember(neUser: NeUser): List<NeUser> {
        memberAdapter.addMember(neUser)
        binding.rcvNewMember.smoothScrollToPosition(memberAdapter.itemCount)
        return memberAdapter.getAll()
    }

    private fun removeMember(neUser: NeUser): List<NeUser> {
        memberAdapter.removeMember(neUser)
        return memberAdapter.getAll()
    }

    private fun checkMembers(members: List<NeUser?>) {
        Logger.d("list check : $members")
        binding.tvNewGroupCount.text = String.format(
            requireContext().resources.getString(R.string.str_new_group_count),
            members.size
        )

        if (members.isNotEmpty() && !isShowMemberChecked) {
            showMemberCheckedAnimation()
        } else if (members.isEmpty()) {
            hideMemberCheckedAnimation()
        }
    }

    private fun showMemberCheckedAnimation() {
        isShowMemberChecked = true

        val viewHeight = AppUtils.dpToPx(context = requireContext(), dp = 88f)
        val newMargin = -AppUtils.dpToPx(context = requireContext(), dp = 24f)

        (binding.flBottom.layoutParams as? LinearLayout.LayoutParams)?.let {
            val a: Animation = object : Animation() {
                override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
                    it.bottomMargin = (viewHeight * (interpolatedTime - 1)).toInt() + newMargin
                    binding.flBottom.layoutParams = it
                }
            }
            a.duration = 300 // in ms
            binding.flBottom.startAnimation(a)
        }
    }

    private fun hideMemberCheckedAnimation() {
        isShowMemberChecked = false

        val viewHeight = AppUtils.dpToPx(context = requireContext(), dp = 88f)
        val newMargin = -AppUtils.dpToPx(context = requireContext(), dp = 24f)

        (binding.flBottom.layoutParams as? LinearLayout.LayoutParams)?.let {
            val a: Animation = object : Animation() {
                override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
                    it.bottomMargin = -(viewHeight * interpolatedTime).toInt() + newMargin
                    binding.flBottom.layoutParams = it
                }
            }
            a.duration = 300 // in ms
            binding.flBottom.startAnimation(a)
        }
    }

    private fun filterSearch() {
        binding.editSearch.addTextChangedListener(
            object : TextWatcher {
                private var searchFor = Constants.EMPTY
                override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
                }

                override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                    val searchText = s.toString().trim()
                    if (searchText == searchFor) {
                        return
                    }

                    searchFor = searchText

                    lifecycleScope.launch {
                        delay(500) // debounce timeOut
                        if (searchText != searchFor) {
                            return@launch
                        }
                        contactAdapter.filter(
                            filterBy = searchFor,
                            lifecycleScope = lifecycleScope
                        )
                    }
                }

                override fun afterTextChanged(p0: Editable?) {
                }
            }
        )
    }

    private fun openSelectMediaDialog() {
        viewModel.openChooseMediaFragment(isChat = false)
    }
}
