package com.netacom.full.ui.main.group

import android.content.ContentProviderOperation
import android.content.ContentProviderResult
import android.content.Context
import android.database.Cursor
import android.net.Uri
import android.provider.ContactsContract
import android.provider.ContactsContract.PhoneLookup
import android.text.InputFilter.LengthFilter
import androidx.activity.addCallback
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.core.widget.doAfterTextChanged
import androidx.fragment.app.activityViewModels
import androidx.navigation.NavController
import androidx.navigation.fragment.navArgs
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.imageloader.loadAvatar
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.getThemeColor
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseCoreCallCameraFragment
import com.netacom.full.databinding.FragmentEditGroupInfoBinding
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.utils.DialogUtil
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.ui.NeChatInfo
import com.netacom.lite.entity.ui.contact.ContactInfo
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.Constants
import com.netacom.lite.util.Constants.EMPTY
import dagger.hilt.android.AndroidEntryPoint

@AndroidEntryPoint
class EditInfoFragment :
    BaseCoreCallCameraFragment<FragmentEditGroupInfoBinding, GroupViewModel>(
        R.layout.fragment_edit_group_info,
        GroupViewModel::class.java
    ) {
    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()
    lateinit var navigationDispatcher: Dispatcher<(NavController) -> Unit>
    lateinit var neChatInfo: NeChatInfo
    private var hasChangedAvatar: Boolean = false
    private var avatarFile: Uri? = null
    private var tokenDeepLink: String? = null
    private val args: EditInfoFragmentArgs by navArgs()
    private val MAX_LENGTH_DESCRIPTION = 150
    override fun setViewModel(): Int = BR.viewModel

    override fun initViews() {
        neChatInfo = args.neChatInfo
        binding.item = neChatInfo
        // set ic
        binding.ivChangeAvatar.setImageResource(R.drawable.ic_icon_camera_avatar_group)
        neChatInfo.neGroup?.id?.toLongOrNull()?.let { groupId ->
            mainSdkViewModel.getDeepLink(groupId)
        }
        binding.toolbarInfo.setEnableActionRight(false)
        binding.toolbarInfo.setColorTextRight(getThemeColor(R.color.color_grey_959595))

        tokenDeepLink?.let {
            binding.tvDisableDeepLink.text = getString(R.string.deep_link_disable_link)
        } ?: let {
            binding.tvDisableDeepLink.text = getString(R.string.deep_link_create_link)
        }
    }

    override fun initData() {

        binding.toolbarInfo.setOnLeftClickListener {
            viewModel.onBack()
            args.neChatInfo.neGroup?.let {
                viewModel.updateGroup(it)
            }
        }

        requireActivity().onBackPressedDispatcher.addCallback(this) {
            viewModel.onBack()
            args.neChatInfo.neGroup?.let {
                viewModel.updateGroup(it)
            }
        }

        binding.toolbarInfo.setOnRightClickListener {
            when {
                neChatInfo.isGroup -> {
                    neChatInfo.neGroup?.let { neGroup ->
                        if (hasChangedAvatar) {
                            avatarFile?.let { filePath ->
                                binding.ivChangeAvatar.isGone = true
                                binding.prgUploadAva.isVisible = true
                                var name: String? = null
                                binding.txtName.text?.let {
                                    if (it.isNotEmpty() && it.toString() != neChatInfo.getDisplayName) {
                                        name = it.toString()
                                    }
                                }
                                viewModel.updateAvatarAndNameForGroup(
                                    neGroup,
                                    filePath,
                                    name,
                                    binding.prgUploadAva
                                )
                            }
                        } else {
                            val name = binding.txtName.text.toString().trim()
                            viewModel.updateNameForGroup(neGroup, name)
                        }
                    }
                }
                else -> {
                    checkPermissionContact(
                        {
                            val phoneNumber = neChatInfo.phoneNumber ?: ""
                            val newName = binding.txtName.text.toString()
                            val contactUpdate = mutableListOf(
                                ContactInfo(
                                    phone = phoneNumber,
                                    name = newName
                                )
                            )
                            if (contactExists(requireContext(), neChatInfo.phoneNumber)) {
                                updateContactName(newName, phoneNumber)
                            }
                            viewModel.updateSingleContact(
                                requireContext(),
                                contactUpdate,
                                object : CallbackResult<Boolean> {
                                    override fun callBackSuccess(result: Boolean) {
                                        neChatInfo.neUser?.let { user ->
                                            user.username = newName
                                            val neGroup = args.neGroup
                                            neGroup?.members?.forEachIndexed { index, value ->
                                                if (value.id == user.id) {
                                                    neGroup.members?.toMutableList()
                                                        ?.set(index, user)
                                                }
                                            }
                                            neGroup?.let { _neGroup ->
                                                viewModel.updateGroup(
                                                    _neGroup
                                                )
                                            }
                                            showSnackBar(R.string.message_update_contact_successful)
                                            viewModel.onBack()
                                        }
                                    }

                                    override fun callBackError(error: String?) {
                                        showSnackBar(
                                            String.format(
                                                getString(R.string.edit_contact_not_found),
                                                neChatInfo.getDisplayName
                                            )
                                        )
                                    }
                                },
                                neChatInfo.getUserId
                            )
                        },
                        {}
                    )
                }
            }
        }

        // check Max character
        binding.txtName.filters += LengthFilter(Constants.MAX_CHARACTER)

        neChatInfo.neGroup?.let { neGroup ->
            when (neGroup.type) {
                GroupType.GROUP_TYPE_CHANNEL -> {
                    hideDeleteContactUI(true)
                    binding.tvTextTitle.text = getString(R.string.text_channel_name)
                    binding.tvChangeOwner.text = getString(R.string.choose_admin)
                }
                GroupType.GROUP_TYPE_GROUP -> {
                    hideDeleteContactUI(true)
                    binding.tvChangeOwner.text = getString(R.string.choose_owner)
                }
                else -> {
                    hideDeleteContactUI(false)
                }
            }
        }

        binding.txtName.doAfterTextChanged {
            it ?: return@doAfterTextChanged
            try {
                when {
                    it.isEmpty() -> {
                        binding.toolbarInfo.setEnableActionRight(hasChangedAvatar)
                        binding.toolbarInfo.setColorTextRight(
                            if (hasChangedAvatar) {
                                getThemeColor(themeHelperImpl.mainColor)
                            } else {
                                getThemeColor(R.color.color_grey_959595)
                            }
                        )
                        binding.imgClearName.isGone = true
                    }
                    it.toString() == neChatInfo.getDisplayName -> {
                        binding.toolbarInfo.setEnableActionRight(hasChangedAvatar)
                        binding.toolbarInfo.setColorTextRight(
                            if (hasChangedAvatar) {
                                getThemeColor(themeHelperImpl.mainColor)
                            } else {
                                getThemeColor(R.color.color_grey_959595)
                            }
                        )
                    }
                    it.isNotEmpty() -> {
                        binding.toolbarInfo.setEnableActionRight(true)
                        binding.toolbarInfo.setColorTextRight(getThemeColor(themeHelperImpl.mainColor))
                        binding.imgClearName.isVisible = true
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
                FirebaseCrashlytics.getInstance().recordException(e)
            }
        }

        binding.imgClearName.clickDebounce {
            binding.txtName.setText("")
        }

        viewModel.editPhoto.observeOnce {
            avatarFile = it
            binding.imgAvatar.loadAvatar(it)
            hasChangedAvatar = true
            binding.toolbarInfo.setEnableActionRight(hasChangedAvatar)
            binding.toolbarInfo.setColorTextRight(getThemeColor(themeHelperImpl.mainColor))
        }

        mainSdkViewModel.createDeepLink.observe { deepLink ->
            deepLink?.let { _deepLink ->
                if (_deepLink.isNotEmpty()) {
                    binding.isShowDeepLink = true
                    tokenDeepLink =
                        _deepLink.substring(_deepLink.lastIndexOf("/") + 1, _deepLink.length)
                    binding.tvDeepLinkUrl.text = _deepLink
                    binding.tvDisableDeepLink.text = getString(R.string.deep_link_disable_link)
                } else {
                    binding.isShowDeepLink = false
                    binding.tvDisableDeepLink.text = getString(R.string.deep_link_create_link)
                }
                mainSdkViewModel._createDeepLink.postValue(null)
            }
        }

        mainSdkViewModel.disableDeepLink.observeOnce { isDisabled ->
            if (isDisabled) {
                binding.tvDeepLinkUrl.text = EMPTY
                binding.tvDisableDeepLink.text = getString(R.string.deep_link_create_link)
                tokenDeepLink = null
            }
        }

        binding.imgAvatar.clickDebounce {
            if (neChatInfo.isGroup) {
                checkPermissionStorage(
                    {
                        openSelectMediaDialog()
                    },
                    {
                    }
                )
            }
        }

        binding.tvChangeOwner.clickDebounce {
            neChatInfo.neGroup?.let {
                viewModel.showChangeLeaderDialog(neGroup = it)
            }
        }

        binding.tvDisableDeepLink.clickDebounce {
            tokenDeepLink?.let { token ->
                mainSdkViewModel.disableDeepLink(token)
            } ?: let {
                neChatInfo.neGroup?.id?.toLongOrNull()?.let { groupId ->
                    mainSdkViewModel.createDeepLink(groupId)
                }
            }
        }

        binding.tvCopyDeepLink.clickDebounce {
            Logger.d("___DeepLink = ")
            if (!binding.tvDeepLinkUrl.text.isNullOrBlank()) {
                AppUtils.copyTextToClipboard(
                    context = requireContext(),
                    binding.tvDeepLinkUrl.text.toString()
                )
            }
        }
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(
                toolbarInfo.getButtonLeft()!!,
                txtName,
                tvDeepLinkUrl
            )
        }
    }

    override fun syncEvent() {
        capture.observeOnce { result ->
            // viewModel.onBack()
            result.let {
                viewModel.postImage(it)
            }
        }

        binding.tvDeleteContact.clickDebounce {
            DialogUtil.show(
                context = requireContext(),
                titleStr = requireContext().resources.getString(R.string.delete_contact),
                messageStr = requireContext().resources.getString(R.string.popup_confirm_delete_contact),
                cancelLabel = R.string.str_cancel,
                okLabel = R.string.delete,
                okFunc = {
                    neChatInfo.neUser?.let { deletedUser ->
                        viewModel.deletedContact(
                            contactId = deletedUser.id ?: 0,
                            contactName = deletedUser.username ?: EMPTY,
                            contactPhone = deletedUser.phone ?: EMPTY,
                            callbackResult = { deletedResult ->
                                if (deletedResult) {
                                    mainSdkViewModel.syncContact(SyncType.LIST_SYNC_SERVER)
                                    // Note Logic: deleted user success => check user is have group => nav to chat screen with user, else nav to previous screen
                                    deletedUser.id?.let { userID ->
                                        viewModel.getGroupOneToOneByUserId(
                                            userID,
                                            object : CallbackResult<NeGroup> {
                                                override fun callBackSuccess(result: NeGroup) {
                                                    result.isDeletedContacts = true
                                                    viewModel.openChat(result)
                                                }

                                                override fun callBackError(error: String?) {
                                                    viewModel.onBack()
                                                }
                                            }
                                        )
                                    }
                                } else {
                                    showSnackBar(R.string.text_error_delete_contact)
                                    viewModel.onBack()
                                }
                            }
                        )
                    }
                },
                themeHelperImpl = themeHelperImpl
            )
        }
    }

    private fun openSelectMediaDialog() {
        viewModel.openChooseMediaFragment(isChat = false)
    }

    private fun contactExists(context: Context, number: String?): Boolean {
        // / number is the phone number
        val lookupUri = Uri.withAppendedPath(PhoneLookup.CONTENT_FILTER_URI, Uri.encode(number))
        val mPhoneNumberProjection = arrayOf(PhoneLookup._ID, PhoneLookup.NUMBER, PhoneLookup.DISPLAY_NAME)
        val cur: Cursor? = context.contentResolver.query(lookupUri, mPhoneNumberProjection, null, null, null)
        cur?.let { cursor ->
            if (cursor.moveToFirst()) {
                return true
            }
            cur.close()
            return false
        } ?: return false
    }

    private fun updateContactName(newName: String, phoneNumber: String): Boolean {
        val contactId = getContactId(requireContext(), phoneNumber)
        contactId?.let { id ->
            val where = ContactsContract.Data.CONTACT_ID + " = ? AND " + ContactsContract.Data.MIMETYPE + " = ?"

            val nameParam = arrayOf(id, ContactsContract.CommonDataKinds.StructuredName.CONTENT_ITEM_TYPE)

            val operations: ArrayList<ContentProviderOperation> = ArrayList()

            operations.add(
                ContentProviderOperation.newUpdate(ContactsContract.Data.CONTENT_URI)
                    .withSelection(where, nameParam)
                    .withValue(ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME, newName)
                    .withValue(ContactsContract.CommonDataKinds.StructuredName.MIDDLE_NAME, "")
                    .withValue(ContactsContract.CommonDataKinds.StructuredName.FAMILY_NAME, "")
                    .withValue(ContactsContract.CommonDataKinds.StructuredName.SUFFIX, "")
                    .build()
            )
            // delete nick name
            ContentProviderOperation.newDelete(ContactsContract.Data.CONTENT_URI).apply {
                val selection = "${ContactsContract.Data.RAW_CONTACT_ID} = ? AND ${ContactsContract.Data.MIMETYPE} = ? "
                val selectionArgs = arrayOf(id, ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME)
                withSelection(selection, selectionArgs)
                operations.add(build())
            }

            // add nickname
            ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI).apply {
                withValue(ContactsContract.Data.RAW_CONTACT_ID, id)
                withValue(
                    ContactsContract.Data.MIMETYPE,
                    ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME
                )
                withValue(ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME, newName)
                operations.add(build())
            }

            return try {
                val results: Array<ContentProviderResult> =
                    requireContext().contentResolver.applyBatch(
                        ContactsContract.AUTHORITY,
                        operations
                    )
                for (result in results) {
                    // ToastUtils.showLong("Contact is successfully edited")
                }
                true
            } catch (e: Exception) {
                e.printStackTrace()
                false
            }
        } ?: return false
    }

    private fun getContactId(context: Context, number: String): String? {
        val cursor = context.contentResolver.query(
            ContactsContract.CommonDataKinds.Phone.CONTENT_URI,
            arrayOf(
                ContactsContract.CommonDataKinds.Phone.CONTACT_ID,
                ContactsContract.CommonDataKinds.Phone.NUMBER
            ),
            ContactsContract.CommonDataKinds.Phone.NUMBER + "=?",
            arrayOf(number),
            null
        )
        if (cursor == null || cursor.count == 0) return null
        cursor.moveToFirst()
        val id = cursor.getString(cursor.getColumnIndex(ContactsContract.CommonDataKinds.Phone.CONTACT_ID))
        cursor.close()
        return id
    }

    private fun hideDeleteContactUI(isHide: Boolean = false) {
        if (isHide) {
            binding.tvDeleteContact.isGone = true
            binding.viewDelete.isGone = true
        } else {
            binding.tvDeleteContact.isVisible = true
            binding.viewDelete.isVisible = true
        }
    }
}
