/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.group

import android.database.ContentObserver
import android.net.Uri
import android.os.Handler
import android.os.Looper
import android.provider.ContactsContract
import androidx.core.view.isInvisible
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import com.netacom.base.chat.adapter.GroupViewPagination
import com.netacom.base.chat.android_utils.LanguageUtils
import com.netacom.base.chat.type.ScreenState
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDataFragment
import com.netacom.full.databinding.FragmentGroupBinding
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.group.adapter.ListGroupAdapter
import com.netacom.full.ui.sdk.NetAloSDK
import com.netacom.full.utils.DialogUtil
import com.netacom.full.widget.EmptyView
import com.netacom.lite.define.SyncType
import com.netacom.lite.repository.configs.GroupConfig
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
Created by vantoan on 28,July,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 */

@AndroidEntryPoint
class GroupFragment : BaseSDKDataFragment<FragmentGroupBinding, GroupViewModel>(
    R.layout.fragment_group,
    GroupViewModel::class.java
) {

    private lateinit var groupAdapter: ListGroupAdapter

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()

    override fun setViewModel(): Int = BR.viewModel

    private var groupViewPagination: GroupViewPagination? = null

    override fun initViews() {
        mainSdkViewModel.syncGroup()
        groupAdapter = ListGroupAdapter(
            currentId = preferencesHelperImpl.getUserId,
            languageApp = LanguageUtils.getLanguage(),
            groupClick = { neGroup ->
                viewModel.openChat(neGroup)
            },
            moreClick = { neGroup ->
                viewModel.openGroupMore(neGroup)
            },
            removeClick = { neGroup ->
                DialogUtil.show(
                    context = requireContext(),
                    titleRes = R.string.popup_confirm,
                    messageRes = R.string.popup_confirm_deleted_conversation,
                    okLabel = R.string.popup_confirm_logout_ok,
                    okFunc = {
                        viewModel.leaveGroupOrDeleteConversation(neGroup)
                    },
                    cancelLabel = R.string.popup_confirm_logout_no,
                    cancelable = true,
                    themeHelperImpl = themeHelperImpl
                )
            },
            themeHelperImpl = themeHelperImpl
        )
        with(binding) {
            with(rcvGroup) {
                setHasFixedSize(true)
                adapter = groupAdapter
            }
            with(toolbar) {
                if (this@GroupFragment.viewModel.checkHideCreateGroup) {
                    getButtonRight()?.isInvisible = true
                }
                setOnLeftClickListener {
                    activity?.onBackPressed()
                }
                setOnRightClickListener {
                    this@GroupFragment.viewModel.openNewMessageDialog()
                }
            }
            with(swipeLayout) {
                setOnRefreshListener {
                    isRefreshing = false
                    mainSdkViewModel.syncGroup(syncType = SyncType.LIST_SYNC_SERVER)
                }
            }
        }

        groupViewPagination = GroupViewPagination(
            recyclerView = binding.rcvGroup,
            loadMore = { page ->
                viewModel.loadMoreGroup(
                    GroupConfig(
                        lastMessageId = groupAdapter.getData.lastOrNull()?.lastMessage?.id,
                        index = page + 1,
                        isConnection = mainSdkViewModel.isConnectSocket.value,
                        currentListSize = groupAdapter.getData.size
                    )
                )
            },
        )

        binding.rcvGroup.itemAnimator = null
        mainSdkViewModel.cacheListGroup?.let {
            if (it.isNotEmpty()) {
                groupAdapter.display(it)
            }
        }
    }

    override fun initData() {
        mainSdkViewModel.listGroup.observeOnce { listNeGroup ->
            groupAdapter.display(listNeGroup)
            if (listNeGroup.isEmpty()) {
                binding.emptyView.setEmptyState()
            }
        }

        mainSdkViewModel.deleteSecretGroup.observe { groupId ->
            viewModel.removeGroupInDb(groupId.toString())
        }

        viewModel.onRemoveGroup.observeOnce { listNeGroup ->
            groupAdapter.display(listNeGroup)
            if (listNeGroup.isEmpty()) {
                binding.emptyView.setEmptyState()
            }
        }

        mainSdkViewModel.joinGroupWithDeepLink.observeOnce { neGroup ->
            neGroup?.let { _neGroup ->
                viewModel.openChat(_neGroup)
            }
        }

        binding.emptyView.newActionListener = object : EmptyView.NewActionListener {
            override fun onStartNewAction() {
                viewModel.openNewMessageDialog()
            }
        }

        // this case handle for new user join app
        checkPermissionContact(
            {
                requireContext().contentResolver?.registerContentObserver(
                    ContactsContract.Contacts.CONTENT_URI,
                    true,
                    object : ContentObserver(Handler(Looper.getMainLooper())) {
                        override fun onChange(selfChange: Boolean, uri: Uri?) {
                            super.onChange(selfChange, uri)
                            launch(Dispatchers.IO) {
                                mainSdkViewModel.getContactFromServer(context = requireContext())
                            }
                        }
                    }
                )
            },
            {
                lifecycleScope.launch(Dispatchers.IO) {
                    mainSdkViewModel.getContactFromServer(context = requireContext(), isSyncContact = false)
                }
            }
        )
    }

    override fun syncEvent() {
        mainSdkViewModel.isConnectSocket.observe { isConnected ->
            binding.toolbar.setCheckConnection(isConnected)
        }

        mainSdkViewModel.userStatus.observeOnce { userStatus ->
            groupAdapter.updateStatus(userStatus)
        }

        viewModel.groupSync.observeOnce {
            mainSdkViewModel.syncGroup(syncType = it)
        }

        viewModel.listGroupLoadMore.observeOnce { listNeGroup ->
            if (listNeGroup.isNotEmpty()) {
                val data = groupAdapter.getData.toMutableList()
                data.addAll(listNeGroup)
                groupAdapter.display(data)
            }
        }

        mainSdkViewModel.groupScreenState.observe { state ->
            if (state.state == ScreenState.EMPTY) {
                if (groupAdapter.getData.isEmpty()) {
                    binding.emptyView.setEmptyState()
                }
            } else {
                binding.emptyView.setScreenState(state)
            }
        }
    }

    override fun setupTheme() {
        themeHelperImpl.setThemeColorForViews(binding.toolbar)
        if (themeHelperImpl.isDarkMode) {
            themeHelperImpl.setThemeColorForViews(binding.toolbar.getButtonLeft()!!, binding.toolbar.getButtonRight()!!)
        }
    }

    override fun onResume() {
        super.onResume()
        NetAloSDK.fragmentDisplay = Constants.GROUP_FRAGMENT
    }

    override fun onPause() {
        super.onPause()
        NetAloSDK.fragmentDisplay = null
    }
}
