/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.ui.main.group

import android.content.Context
import android.net.Uri
import android.util.Log
import android.widget.ProgressBar
import androidx.core.view.isGone
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import androidx.navigation.NavController
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkInfo
import androidx.work.WorkManager
import androidx.work.workDataOf
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.android_utils.UriUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.define.ConfigDef.PAGE_SIZE_MEDIA_GET
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.network.ApiResponseError
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.network.ResultData
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKViewModel
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.full.extensions.navigateIfSafeClearStack
import com.netacom.full.model.CallInfoModel
import com.netacom.full.ui.main.chat.ForwardMessageDialogDirections
import com.netacom.full.ui.main.contact.ContactUtils
import com.netacom.full.worker.upload.UploadWorker
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MediaType
import com.netacom.lite.define.MessageType
import com.netacom.lite.define.SyncType
import com.netacom.lite.entity.database.attachment.NeAttachments
import com.netacom.lite.entity.socket.Message
import com.netacom.lite.entity.ui.NeChatInfo
import com.netacom.lite.entity.ui.call.NeCallLog
import com.netacom.lite.entity.ui.contact.ContactInfo
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.media.NeMedia
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.mapper.Media_SK_UI_Mapper
import com.netacom.lite.network.model.FileUpload
import com.netacom.lite.network.model.response.MediaInGroupResponse
import com.netacom.lite.network.model.response.UploadResult
import com.netacom.lite.repository.ContactRepository
import com.netacom.lite.repository.DownloadRepository
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.repository.SocketRepository
import com.netacom.lite.repository.base.State
import com.netacom.lite.repository.configs.GroupConfig
import com.netacom.lite.socket.request.DeleteSecretMessageRequest
import com.netacom.lite.socket.request.GroupRequest
import com.netacom.lite.util.CallbackResult
import com.netacom.lite.util.FileUtils
import com.netacom.lite.util.getUrlImage
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.withContext
import javax.inject.Inject

/**
Created by vantoan on 30,July,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

@HiltViewModel
class GroupViewModel @Inject constructor(
    private val socketRepository: SocketRepository,
    val downloadRepository: DownloadRepository,
    private val groupRepository: GroupRepository,
    private val jsonSerializer: JsonSerializer,
    private val contactRepository: ContactRepository,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>,
    private val mediaSkUiMapper: Media_SK_UI_Mapper
) : BaseSDKViewModel(socketRepository, groupRepository, navigationDispatcher) {
    private val isShowLog = true

    private var _groupSync = MutableLiveData<EventLiveData<SyncType>>()
    val groupSync: LiveData<EventLiveData<SyncType>> = _groupSync

    private var _editPhoto = MutableLiveData<EventLiveData<Uri>>()
    val editPhoto: LiveData<EventLiveData<Uri>> = _editPhoto

    private var _mediaData = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val mediaData: LiveData<EventLiveData<List<NeMedia>>> = _mediaData

    private var _callLogsSortedByUser = MutableLiveData<EventLiveData<List<NeCallLog>>>()
    val callLogsSortedByUser: LiveData<EventLiveData<List<NeCallLog>>> = _callLogsSortedByUser

    private var _onActionImage = MutableLiveData<EventLiveData<ViewMediaFragment.Action>>()
    val onActionImage: LiveData<EventLiveData<ViewMediaFragment.Action>> = _onActionImage

    private var _onRemoveGroup = MutableLiveData<EventLiveData<List<NeGroup>>>()
    val onRemoveGroup: LiveData<EventLiveData<List<NeGroup>>> = _onRemoveGroup

    private var _singleMediaData = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val singleMediaData: LiveData<EventLiveData<List<NeMedia>>> = _singleMediaData

    private val _listGroupLoadMore = MutableLiveData<EventLiveData<List<NeGroup>>>()
    val listGroupLoadMore: LiveData<EventLiveData<List<NeGroup>>> = _listGroupLoadMore

    val checkHidePhone get() = getPreferences.sdkConfig?.hidePhone ?: false
    val checkHideInfo get() = getPreferences.sdkConfig?.hideInfo ?: false
    val checkHideCreateGroup get() = getPreferences.sdkConfig?.hideCreateGroup ?: true

    private val _listNeMedia = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val listNeMedia: LiveData<EventLiveData<List<NeMedia>>> = _listNeMedia

    private val _listNeMediaLoadMore = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val listNeMediaLoadMore: LiveData<EventLiveData<List<NeMedia>>> = _listNeMediaLoadMore

    private val _mediaDetail = MutableLiveData<EventLiveData<MediaInGroupResponse>>()
    val mediaDetail: LiveData<EventLiveData<MediaInGroupResponse>> = _mediaDetail
    /**
     * Hot fix for issue ALO-1865
     * Will be replaced it by Coroutine Flow
     */
    private val _listNeMediaFragment = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val listNeMediaFragment: LiveData<EventLiveData<List<NeMedia>>> = _listNeMediaFragment

    private val _listNeMediaFragmentLoadMore = MutableLiveData<EventLiveData<List<NeMedia>>>()
    val listNeMediaFragmentLoadMore: LiveData<EventLiveData<List<NeMedia>>> = _listNeMediaFragmentLoadMore

    // Navigation
    fun openGroupMore(neGroup: NeGroup) {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                GroupFragmentDirections.actionGroupFragmentToGroupMoreFunction(
                    neGroup = neGroup
                )
            )
        }
    }

    fun openSearch() {
        navigationDispatcher.emit {
            it.navigateIfSafe(GroupFragmentDirections.actionGroupFragmentToContactSearch())
        }
    }

    fun openNewMessageDialog() {
        navigationDispatcher.emit {
            it.navigateIfSafe(GroupFragmentDirections.openNewMessageDialog())
        }
    }

    fun openCreateGroupDialog(neUser: NeUser? = null, isChannel: Boolean = false) {
        navigationDispatcher.emit {
            it.navigateIfSafe(NewMessageDialogDirections.openCreateGroup(neUser, isChannel))
        }
    }

    fun openCreateSecretGroupDialog(neUser: NeUser? = null) {
        navigationDispatcher.emit {
            it.navigateIfSafe(NewMessageDialogDirections.openCreateSecretGroup(neUser))
        }
    }

    fun openQrCodeDialog(neGroup: NeGroup, groupLink: String?) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.generateQrCode(neGroup, groupLink))
        }
    }

    fun showMembersDialog(neGroup: NeGroup, leaderId: Long) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.openMemberList(neGroup, leaderId))
        }
    }

    fun showAddMemberDialog(neGroup: NeGroup) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.addMemberToGroup(neGroup))
        }
    }

    fun showAddContactDialog(neUser: NeUser?, className: String?) {
        navigationDispatcher.emit {
            it.navigate(ChatInfoFragmentDirections.addContactPhone(neUser, className = className))
        }
    }

    fun showChangeLeaderDialog(neGroup: NeGroup, willLeaveGroup: Boolean = false) {
        navigationDispatcher.emit {
            it.navigateIfSafe(
                EditInfoFragmentDirections.changeOwner(
                    neGroup,
                    isChangeOwner = true,
                    willLeaveGroup = willLeaveGroup
                )
            )
        }
    }

    fun chooseImageDialog() {
        navigationDispatcher.emit {
            it.navigateIfSafe(EditInfoFragmentDirections.chooseImage())
        }
    }

    fun postImage(uri: Uri) {
        _editPhoto.post(EventLiveData(uri))
    }

    fun goToMediaView(neMedia: NeMedia, neGroup: NeGroup) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.viewMedia(neMedia, neGroup))
        }
    }

    fun displayImage(imageId: String?) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.displayImage(imageId))
        }
    }

    fun showImageActionPopup() {
        navigationDispatcher.emit {
            it.navigateIfSafe(ViewMediaFragmentDirections.showActionImage())
        }
    }

    fun showSaveImagePopup() {
        navigationDispatcher.emit {
            it.navigateIfSafe(ViewMediaFragmentDirections.saveImage())
        }
    }

    fun showAddMember(neUser: NeUser?, className: String? = null) {
        navigationDispatcher.emit {
            it.navigate(
                MemberInfoDialogDirections.addContactPhone(neUser = neUser, className = className)
            )
        }
    }

    fun openForwardDialog(filePath: String?) {
        filePath?.apply {
            navigationDispatcher.emit { nav ->
                nav.navigateIfSafeClearStack(ViewMediaFragmentDirections.actionMediaFunctionToForwardMessageDialog(this))
            }
        }
    }

    fun leaveGroupOrDeleteConversation(neGroup: NeGroup) {
        val groupId = neGroup.id ?: return
        val groupType = neGroup.type ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            when (groupType) {
                GroupType.GROUP_TYPE_GROUP -> {
                    /**
                     * leave group
                     * */
                    val leaveGroupRequest = GroupRequest(groupId = groupId.toLongOrNull())
                    getIOSocket.leaveGroup(leaveGroupRequest = leaveGroupRequest)
                        .let { result ->
                            when (result) {
                                is ResultData.Success -> result.data?.let {
                                    removeGroupInDb(groupId)
                                }
                                is ResultData.Failed -> Logger.e("error:${result.message}")
                                else -> {
                                }
                            }
                        }
                }
                GroupType.GROUP_TYPE_PRIVATE -> {
                    val deleteSecretMessageRequest = DeleteSecretMessageRequest(group_id = groupId.toLongOrNull())
                    getIOSocket.deleteSecretChat(deleteSecretMessageRequest = deleteSecretMessageRequest)
                        .let { result ->
                            when (result) {
                                is ResultData.Success -> result.data?.let {
                                    removeGroupInDb(groupId)
                                }
                                is ResultData.Failed -> {
                                    removeGroupInDb(groupId)
                                    Logger.e("error:${result.message}")
                                }
                                else -> {
                                }
                            }
                        }
                }
                else -> {
                    /**
                     * delete conversation
                     * */
                    val deleteConversation = GroupRequest(groupId = groupId.toLongOrNull())
                    getIOSocket.deleteConversation(
                        deleteConversation
                    ).let { result ->
                        when (result) {
                            is ResultData.Success -> result.data?.let {
                                removeGroupInDb(groupId)
                            }
                            is ResultData.Failed -> Logger.e("error:${result.message}")
                            else -> {
                            }
                        }
                    }
                }
            }
        }
    }

    fun deleteChannel(neGroup: NeGroup) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            val groupRequest = GroupRequest(groupId = groupId)
            getIOSocket.deleteConversation(groupRequest)
            getIOSocket.deleteGroup(groupRequest).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        if (it) {
                            removeGroupInDb(groupId.toString())
                        }
                    }
                    is ResultData.Failed -> {
                        Logger.e("error:${result.message}")
                    }
                    else -> {
                    }
                }
            }
        }
    }

    fun removeGroupInDb(groupId: String?) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            groupRepository.removeGroupByGroupId(
                groupId = groupId
            )?.let {
                if (it) {
                    groupId?.toLongOrNull()?.let { id ->
                        groupRepository.deleteAllMediaByGroupId(id)
                    }
                    groupRepository.deleteEncryptedFilesInGroup(groupId)
                    navigationDispatcher.emit { nav ->
                        nav.navigateIfSafe(ChatInfoFragmentDirections.leaveGroup())
                    }
                    _onRemoveGroup.post(EventLiveData(groupRepository.getListGroupDb()))
                }
            }
        }
    }

    fun groupActionPin(neGroup: NeGroup, callbackResult: CallbackResult<Boolean>) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            val pinGroupRequest = GroupRequest(groupId = groupId)
            if (neGroup.isPin) {
                /**
                 * pin -> unpin
                 * */
                getIOSocket.unPinGroup(pinGroupRequest).let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            groupRepository.updatePinGroupById(groupId = groupId, isPin = false)
                            _groupSync.post(EventLiveData(SyncType.LIST_SYNC_DB))
                            callbackResult.callBackSuccess(true)
                        }
                        is ResultData.Failed -> callbackResult.callBackError()
                        else -> {
                        }
                    }
                }
            } else {
                /**
                 * unpin -> pin
                 * */
                getIOSocket.pinGroup(pinGroupRequest).let { result ->
                    when (result) {
                        is ResultData.Success -> result.data?.let {
                            groupRepository.updatePinGroupById(groupId = groupId, isPin = true)
                            _groupSync.post(EventLiveData(SyncType.LIST_SYNC_DB))
                            callbackResult.callBackSuccess(true)
                        }
                        is ResultData.Failed -> callbackResult.callBackError()
                        else -> {
                        }
                    }
                }
            }
        }
    }

    fun checkPartnerId(partnerUid: Long, callbackResult: CallbackResult<NeGroup>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            checkGroup(partnerUin = partnerUid).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        convertToNeGroup(it).let { group ->
                            callbackResult.callBackSuccess(group)
                            navigationDispatcher.emit { nav ->
                                nav.navigateIfSafe(ForwardMessageDialogDirections.actionForwardMessageDialogToChatGroupFragment(group))
                            }
                        }
                    }
                    is ResultData.Failed -> callbackResult.callBackError()
                    else -> {
                    }
                }
            }
        }
    }

    fun addMemberToGroup(neGroup: NeGroup, newMembers: List<NeUser>) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            val owner = neGroup.owner?.id.toString()
            val pushList = newMembers.map {
                it.id ?: 0L
            }
            updateGroup(
                groupId = groupId,
                owner = owner,
                name = "",
                pushList = pushList
            ).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        groupRepository.saveGroupToDB(group = it)?.let { neGroup ->
                            onBack()
                            updateGroup(neGroup)
                            sendMessageAddOrRemoveMembersUpdateGroupNotification(
                                listAdd = pushList,
                                removeId = null,
                                isAddMember = true,
                                neGroup = neGroup
                            )
                        }
                    }
                    is ResultData.Failed -> screenStateError.postErrorOnce(result.message)
                    else -> {
                        // screenState.postError(message = result.message)
                    }
                }
            }
        }
    }

    fun removeMember(neUser: NeUser, neGroup: NeGroup) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        val userId = neUser.id ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            val groupName = neGroup.name
            val groupAvatar = neGroup.getDisplayAvatar
            val owner = neGroup.owner?.id.toString()
            val pullList = mutableListOf(userId)
            updateGroup(
                groupId = groupId,
                name = groupName,
                avatar = groupAvatar,
                owner = owner,
                pullList = pullList
            ).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        groupRepository.saveGroupToDB(group = it)?.let { neGroup ->
                            updateGroup(neGroup)
                            sendMessageAddOrRemoveMembersUpdateGroupNotification(
                                listAdd = emptyList(),
                                removeId = pullList.firstOrNull(),
                                isAddMember = false,
                                neGroup = neGroup
                            )
                        }
                    }
                    is ResultData.Failed -> screenStateError.postErrorOnce(result.message)
                    else -> {
                    }
                }
            }
        }
    }

    fun changeOwner(neGroup: NeGroup, owner: String, willLeaveGroup: Boolean = false) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            updateGroup(groupId = groupId, owner = owner, name = "").let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        groupRepository.saveGroupToDB(group = it)?.let { neGroup ->
                            navigationDispatcher.emit { nav ->
                                nav.navigateIfSafe(
                                    AddMemberDialogDirections.backToInfo(
                                        CallInfoModel(neGroup = neGroup)
                                    )
                                )
                            }
                            updateGroup(neGroup)
                            sendMessageUpdateGroupNotification(
                                neGroup = neGroup,
                                avatar = null,
                                name = null,
                                owner = owner
                            )

                            if (willLeaveGroup) {
                                leaveGroupOrDeleteConversation(neGroup)
                            }
                        }
                    }
                    is ResultData.Failed -> screenStateError.postErrorOnce(result.message)
                    else -> {
                    }
                }
            }
        }
    }

    fun updateNameForGroup(neGroup: NeGroup, groupName: String) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            updateGroup(groupId = groupId, name = groupName).let { result ->
                when (result) {
                    is ResultData.Success -> result.data?.let {
                        groupRepository.saveGroupToDB(group = it)?.let { neGroup ->
                            onBack()
                            updateGroup(neGroup)
                            sendMessageUpdateGroupNotification(
                                neGroup = neGroup,
                                avatar = null,
                                name = groupName,
                                owner = null
                            )
                        }
                    }
                    is ResultData.Failed -> {
                        Logger.e("updateNameForGroup :${result.message}")
                    }
                    else -> {
                    }
                }
            }
        }
    }

    private fun sendMessageUpdateGroupNotification(
        neGroup: NeGroup,
        avatar: String?,
        name: String?,
        owner: String?,
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val neMessage: NeMessage? = NeMessage()
            neMessage.let { updateGroup ->
                updateGroup?.type = MessageType.MESSAGE_TYPE_GROUP_UPDATE
                updateGroup?.attachment = NeAttachments(
                    type = "update_group",
                    updatedGroupAvatar = avatar,
                    updatedGroupName = name,
                    ownerUin = owner,
                )
                updateGroup?.let {
                    createUpdateGroup(
                        neGroup = neGroup,
                        updatedGroup = it,
                        isAddMember = false,
                        callbackResult = object : CallbackResult<Message> {
                            override fun callBackSuccess(result: Message) {
                            }

                            override fun callBackError(error: String?) {
                            }
                        }

                    )
                }
            }
        }
    }

    private fun sendMessageAddOrRemoveMembersUpdateGroupNotification(
        listAdd: List<Long>,
        removeId: Long?,
        isAddMember: Boolean,
        neGroup: NeGroup
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val neMessage: NeMessage? = NeMessage()
            neMessage.let { updateGroup ->
                updateGroup?.type = MessageType.MESSAGE_TYPE_GROUP_UPDATE
                if (isAddMember) {
                    updateGroup?.attachment = NeAttachments(
                        type = "update_group",
                        addedUins = listAdd.map { it.toString() }
                    )
                } else {
                    updateGroup?.attachment = NeAttachments(
                        type = "update_group",
                        removedUin = removeId.toString()
                    )
                }
                updateGroup?.let {
                    createUpdateGroup(
                        neGroup = neGroup,
                        updatedGroup = it,
                        isAddMember = isAddMember,
                        callbackResult = object : CallbackResult<Message> {
                            override fun callBackSuccess(result: Message) {
                            }

                            override fun callBackError(error: String?) {
                            }
                        }
                    )
                }
            }
        }
    }

    fun updateAvatarAndNameForGroup(
        neGroup: NeGroup,
        avatarFilePath: Uri,
        groupName: String?,
        loadingView: ProgressBar?
    ) {
        val groupId = neGroup.id?.toLongOrNull() ?: return
        launchOnViewModelScope(getPostExecutionThread.io) {
            uploadFile(
                UriUtils.uri2File(avatarFilePath).absolutePath,
                callbackSuccess = {
                    if (it is ResultData.Success) {
                        Log.d("updateAvatar", "avatar = $it")
                        launchOnViewModelScope(getPostExecutionThread.io) {
                            updateGroup(
                                groupId = groupId,
                                name = groupName,
                                avatar = it.data
                            ).let { result ->
                                withContext(getPostExecutionThread.main) {
                                    loadingView?.isGone = true
                                }
                                when (result) {
                                    is ResultData.Success -> result.data?.let { group ->
                                        groupRepository.saveGroupToDB(group = group)?.let { neGroup ->
                                            onBack()
                                            updateGroup(neGroup)
                                            sendMessageUpdateGroupNotification(
                                                neGroup = neGroup,
                                                avatar = group.avatar,
                                                name = groupName,
                                                owner = null
                                            )
                                        }
                                    }
                                    is ResultData.Failed -> {
                                        Logger.e("updateAvatarAndNameForGroup :${result.message}")
                                    }
                                    else -> {
                                    }
                                }
                            }
                        }
                    }
                },
                callbackError = {}
            )
        }
    }

    private suspend fun uploadFile(
        filePath: String,
        callbackSuccess: (ResultData<String>) -> Unit = {},
        callbackError: (ResultData<String>) -> Unit = {}
    ) {
        withContext(getPostExecutionThread.io) {
            val uploadWorkRequest = OneTimeWorkRequestBuilder<UploadWorker>().setInputData(
                workDataOf(
                    UploadWorker.KEY_THUMBNAIL_ARG to false,
                    UploadWorker.KEY_ENCRYPT_ARG to false,
                    UploadWorker.KEY_MEDIA_TYPE_ARG to jsonSerializer.asJson(MediaType.PHOTO, MediaType::class.java),
                    UploadWorker.KEY_UPLOAD_FILE_ARG to jsonSerializer.asJson(FileUpload(filePath), FileUpload::class.java)
                )
            ).build()
            WorkManager.getInstance(Utils.getApp()).enqueue(uploadWorkRequest)
            withContext(getPostExecutionThread.main) {
                WorkManager.getInstance(Utils.getApp()).getWorkInfoByIdLiveData(uploadWorkRequest.id).observeForever(object : Observer<WorkInfo> {
                    override fun onChanged(workInfo: WorkInfo?) {
                        if (workInfo != null && workInfo.state.isFinished) {
                            jsonSerializer.asObject(workInfo.outputData.getString(UploadWorker.KEY_UPLOAD_RESULT), UploadResult::class.java)?.let {
                                if (it.status == UploadResult.UploadResultStatus.SUCCESS) {
                                    callbackSuccess(ResultData.Success(data = it.fileUrl ?: ""))
                                } else if (it.status == UploadResult.UploadResultStatus.FAIL) {
                                    callbackError(ResultData.Failed(message = it.message))
                                }
                            }
                            WorkManager.getInstance(Utils.getApp()).getWorkInfoByIdLiveData(uploadWorkRequest.id).removeObserver(this)
                        }
                    }
                })
            }
        }
    }

    fun getMediaForGroup(
        groupId: Long,
        isLoadAttachments: Boolean = true,
        offset: Int = 0,
        limit: Int = PAGE_SIZE_MEDIA_GET,
        mediaType: Int = 2,
        isViewMediaFragment: Boolean = false
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val mediaQueryMap: HashMap<String, String> = hashMapOf()
            mediaQueryMap["group_id"] = groupId.toString()
            mediaQueryMap["media_type"] = mediaType.toString()
            mediaQueryMap["load_attachments"] = isLoadAttachments.toString()
            mediaQueryMap["offset"] = offset.toString()
            mediaQueryMap["limit"] = limit.toString()

            groupRepository.getMediaByGroupId(mediaQueryMap).collect { state ->
                when (state) {
                    is ApiResponseSuccess -> {
                        state.data?.let {
                            Logger.d("Refactor Gallery - getMediaByGroupId - ApiResponseSuccess - ${it.size}")
                            if (isViewMediaFragment) {
                                _listNeMediaFragment.postValue(EventLiveData(mediaSkUiMapper.mapFromEntityList(it)))
                            } else {
                                _listNeMedia.postValue(EventLiveData(mediaSkUiMapper.mapFromEntityList(it)))
                            }
                        }
                    }
                    is ApiResponseError -> {
                        Logger.d("Refactor Gallery - getMediaByGroupId - ApiResponseError")
                        if (isViewMediaFragment) {
                            _listNeMediaFragment.postValue(EventLiveData(emptyList()))
                        } else {
                            _listNeMedia.postValue(EventLiveData(emptyList()))
                        }
                    }
                }
            }
        }
    }

    fun startGroupChat(
        listIDs: List<Long>,
        groupName: String,
        avatarPath: String = "",
        isChannel: Boolean = false,
        callbackResult: CallbackResult<Boolean>
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            var avatarUid = ""
            if (avatarPath.isNotEmpty()) {
                uploadFile(
                    avatarPath,
                    callbackSuccess = {
                        launchOnViewModelScope(getPostExecutionThread.io) {
                            if (it is ResultData.Success) {
                                it.data?.let { url ->
                                    avatarUid = url
                                }
                            }
                            startCreateGroup(
                                listIDs = listIDs,
                                groupName = groupName,
                                avatarUid = avatarUid,
                                typeGroup = if (isChannel) GroupType.GROUP_TYPE_CHANNEL else GroupType.GROUP_TYPE_GROUP,
                                callbackResult = callbackResult
                            )
                        }
                    },
                    callbackError = {
                        launchOnViewModelScope(getPostExecutionThread.io) {
                            startCreateGroup(
                                listIDs = listIDs,
                                groupName = groupName,
                                avatarUid = avatarUid,
                                typeGroup = if (isChannel) GroupType.GROUP_TYPE_CHANNEL else GroupType.GROUP_TYPE_GROUP,
                                callbackResult = callbackResult
                            )
                        }
                    }
                )
            } else {
                startCreateGroup(
                    listIDs = listIDs,
                    groupName = groupName,
                    avatarUid = avatarUid,
                    typeGroup = if (isChannel) GroupType.GROUP_TYPE_CHANNEL else GroupType.GROUP_TYPE_GROUP,
                    callbackResult = callbackResult
                )
            }
        }
    }

    private suspend fun startCreateGroup(
        listIDs: List<Long>,
        groupName: String,
        avatarUid: String = "",
        typeGroup: Int,
        callbackResult: CallbackResult<Boolean>
    ) {
        createGroup(
            type = typeGroup,
            listIDs = listIDs,
            groupName = groupName,
            avatar = avatarUid
        ).let { result ->
            when (result) {
                is ResultData.Success -> result.data?.let {
                    groupRepository.saveGroupToDB(group = it)?.let { neGroup ->
                        openChat(neGroup)
                        callbackResult.callBackSuccess(true)
                    }
                }
                is ResultData.Failed -> {
                    Logger.e("Create Group Failed : ${result.message}")
                    callbackResult.callBackError(StringUtils.getString(R.string.text_create_group_fail))
                    // screenStateError.postErrorOnce(StringUtils.getString(R.string.text_create_group_fail))
                }
                else -> {
                }
            }
        }
    }

    fun getMediaListDbByGroupId(groupId: Long, isSingleUser: Boolean = false) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            groupRepository.getMediaListDbByGroupId(groupId).let { listNeMedia ->
                if (isShowLog) Logger.d("listNeMedia = ${jsonSerializer.asJson(listNeMedia, NeMedia::class.java)}")
                if (isSingleUser) {
                    _singleMediaData.post(EventLiveData(listNeMedia))
                } else {
                    _mediaData.post(EventLiveData(listNeMedia))
                }
            }
        }
    }

    fun goToEditGroupOrPartnerInfo(neChatInfo: NeChatInfo, neGroup: NeGroup? = null) {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.editInfo(neChatInfo, neGroup))
        }
    }

    fun showCallLogByUser() {
        navigationDispatcher.emit {
            it.navigateIfSafe(ChatInfoFragmentDirections.viewCallLogByUser())
        }
    }

    private fun updateContactName(context: Context, newName: String, phoneNumber: String) {
        ContactUtils.getContactIDByPhoneNumber(
            context,
            phoneNumber,
            object : CallbackResult<String> {
                override fun callBackSuccess(result: String) {
                    ContactUtils.updatePhoneContact(
                        context,
                        result,
                        newName,
                        object : CallbackResult<String> {
                            override fun callBackSuccess(result: String) {
                            }

                            override fun callBackError(error: String?) {
                            }
                        }
                    )
                }

                override fun callBackError(error: String?) {
                }
            }
        )
    }

    fun getSortedCallLogsByType(callLogs: List<NeCallLog>, missCall: Boolean) {
        launchOnViewModelScope(getPostExecutionThread.default) {
            if (callLogs.isNotEmpty()) {
                val listCallLogsForShow = callLogs.filter { log -> log.isMissCall == missCall }
                    .sortedByDescending { log -> log.createAt }
                if (listCallLogsForShow.isNotEmpty()) {
                    _callLogsSortedByUser.post(EventLiveData(listCallLogsForShow))
                }
            }
        }
    }

    fun postActionImage(actionImage: ViewMediaFragment.Action) {
        _onActionImage.post(EventLiveData(actionImage))
    }

    fun handleBackFm(neGroup: NeGroup, isChangeBackground: Boolean = false) {
        navigationDispatcher.emit { nav ->
            val previousDestinaionId = nav.previousBackStackEntry?.destination?.id
            previousDestinaionId?.let {
                if (previousDestinaionId == R.id.chatFragment) {
                    nav.navigateIfSafe(ChatInfoFragmentDirections.backToBaseChat(neGroup, isChangeBackground = isChangeBackground))
                } else {
                    onBack()
                }
            } ?: onBack()
        }
    }

    fun updateSingleContact(
        context: Context,
        listContacts: List<ContactInfo>,
        callbackResult: CallbackResult<Boolean>,
        userId: Long?
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            contactRepository.sendPhoneContacts(
                udid = "",
                isForce = 1,
                contactInfo = listContacts
            ).collect {
                if (it) {
                    userId?.let { id ->
                        val updateContact = listContacts[0]
                        updateContact.name?.let { name ->
                            if (updateContact.phone.isNotBlank()) {
                                updateContactName(context, name, updateContact.phone)
                            }
                            contactRepository.updateNameForContact(id, name).apply {
                                if (this == true) {
                                    callbackResult.callBackSuccess(true)
                                }
                            }
                        }
                    }
                } else {
                    callbackResult.callBackError("")
                }
            }
        }
    }

    fun getGroupOneToOneByUserId(userId: Long, callbackResult: CallbackResult<NeGroup>) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            groupRepository.getGroupOneToOneByUserId(userId)?.let { neGroup ->
                callbackResult.callBackSuccess(neGroup)
            } ?: kotlin.run {
                callbackResult.callBackError("")
            }
        }
    }

    fun isMute(groupId: String) = groupRepository.isMute(groupId)

    fun getGroupDbById(groupId: String?) = groupRepository.getGroupDbById(groupId)

    fun userHasConversation(neUser: NeUser): Boolean {
        return groupRepository.getGroupOneToOneByUserId(neUser.id ?: 0).isNotNull
    }

    fun downloadFile(fileUrl: String, callbackResult: (Boolean, String?) -> Unit?, isForward: Boolean = false) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            downloadRepository.downloadFiles(arrayListOf(fileUrl.getUrlImage())).collect {
                when (it) {
                    is ApiResponseSuccess -> {
                        it.data?.let { responseBody ->
                            val pathFile = FileUtils.saveFileToLocal(responseBody, isTemp = isForward, isSaveDocument = !isForward)
                            callbackResult(true, pathFile)
                        }
                    }
                    is ApiResponseError -> {
                        callbackResult(false, null)
                    }
                }
            }
        }
    }

    fun deletedContact(contactName: String, contactPhone: String, contactId: Long, callbackResult: (Boolean) -> Unit?) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val deletedInfo = ContactInfo(
                name = contactName,
                phone = contactPhone,
                destroy = 1
            )
            contactRepository.sendPhoneContacts(
                udid = "",
                isForce = 0,
                contactInfo = mutableListOf(deletedInfo)
            ).collect { isDeletedSuccess ->
                if (isDeletedSuccess) {
                    getDbManager.deletedContact(contactId).apply {
                        if (this != null && this) {
                            callbackResult(true)
                        } else {
                            callbackResult(false)
                        }
                    }
                } else {
                    callbackResult(false)
                }
            }
        }
    }

    // Group
    private var isListGroupCalling = false
    fun loadMoreGroup(groupConfig: GroupConfig) {
        if (isListGroupCalling) return
        isListGroupCalling = true
        launchOnViewModelScope(getPostExecutionThread.io) {
            socketRepository.getListGroup(groupConfig).collectLatest { state ->
                when (state) {
                    is State.SuccessMerged -> {
                        isListGroupCalling = false
                        withContext(getPostExecutionThread.main) {
                            _listGroupLoadMore.value = EventLiveData(state.data)
                        }
                    }
                    is State.Loading -> Unit
                    is State.Error -> isListGroupCalling = false
                    is State.SuccessLocal -> Unit
                }
            }
        }
    }

    private var isLoadMoreCalling = false
    fun loadMoreMedia(
        groupId: Long,
        isLoadAttachments: Boolean = true,
        offset: Int = 0,
        limit: Int = PAGE_SIZE_MEDIA_GET,
        mediaType: Int = 2,
        isViewMediaFragment: Boolean = false
    ) {
        if (isLoadMoreCalling) return
        isLoadMoreCalling = true
        launchOnViewModelScope(getPostExecutionThread.io) {
            val mediaQueryMap: HashMap<String, String> = hashMapOf()
            mediaQueryMap["group_id"] = groupId.toString()
            mediaQueryMap["media_type"] = mediaType.toString()
            mediaQueryMap["load_attachments"] = isLoadAttachments.toString()
            mediaQueryMap["offset"] = offset.toString()
            mediaQueryMap["limit"] = limit.toString()

            groupRepository.getMediaByGroupId(mediaQueryMap).collect { state ->
                when (state) {
                    is ApiResponseSuccess -> {
                        isLoadMoreCalling = false
                        state.data?.let {
                            if (isViewMediaFragment) {
                                _listNeMediaFragmentLoadMore.postValue(EventLiveData(mediaSkUiMapper.mapFromEntityList(it)))
                            } else {
                                _listNeMediaLoadMore.postValue(EventLiveData(mediaSkUiMapper.mapFromEntityList(it)))
                            }
                        }
                    }
                    is ApiResponseError -> {
                        isLoadMoreCalling = false
                        if (isViewMediaFragment) {
                            _listNeMediaFragmentLoadMore.postValue(EventLiveData(emptyList()))
                        } else {
                            _listNeMediaLoadMore.postValue(EventLiveData(emptyList()))
                        }
                    }
                }
            }
        }
    }

    fun getMediaDetailForGroup(
        groupId: Long,
        isLoadAttachments: Boolean = true,
        mediaType: Int = 2,
        mediaUid: String? = null,
        messageId: String? = null
    ) {
        launchOnViewModelScope(getPostExecutionThread.io) {
            val mediaQueryMap: HashMap<String, String> = hashMapOf()
            mediaQueryMap["group_id"] = groupId.toString()
            mediaQueryMap["media_type"] = mediaType.toString()
            mediaQueryMap["load_attachments"] = isLoadAttachments.toString()

            messageId?.let {
                mediaQueryMap["message_id"] = messageId
            }
            mediaUid?.let {
                mediaQueryMap["media_uid"] = it
            }
            groupRepository.getMediaDetailByGroupId(mediaQueryMap).collect { state ->
                when (state) {
                    is ApiResponseSuccess -> {
                        state.data?.let {
                            Logger.e("total media size == ${it.total} && media index == ${it.index}")
                            _mediaDetail.post(EventLiveData(it))
                        }
                    }
                    is ApiResponseError -> {
                    }
                }
            }
        }
    }
}
