package com.netacom.full.ui.main.group

import android.text.Editable
import android.text.TextWatcher
import androidx.core.view.isVisible
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKBottomDialog
import com.netacom.full.databinding.ViewNewMessagePopupBinding
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.group.adapter.NewMessageAdapter
import com.netacom.lite.define.SyncType
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@AndroidEntryPoint
class NewMessageDialog : BaseSDKBottomDialog<ViewNewMessagePopupBinding, GroupViewModel>(
    R.layout.view_new_message_popup,
    GroupViewModel::class.java
) {

    private lateinit var contactAdapter: NewMessageAdapter

    override fun setLayoutHeight(): Int = DialogDef.LAYOUT_FULL

    override fun setViewModel(): Int = BR.viewModel

    private val mainSdkViewModel: MainSdkViewModel by activityViewModels()

    override fun initViews() {
        contactAdapter = NewMessageAdapter(
            itemClick = { item, _, _ ->
                /**
                 * Open the message screen.
                 */
                // Log.d("TAG", "TAG_contactAdapter Item Click")
                val neContact = item.getItemUser()?._neContact
                neContact?.let {
                    viewModel.startOneByOneChat(it)
                }
            },
            emptyCallBack = { isEmpty ->
                if (isEmpty) {
                    binding.layoutEmpty.ctlEmptyContainer.isVisible = true
                    binding.layoutEmpty.btnEmpty.clickDebounce {
                        viewModel.showAddContactDialog(null, null)
                    }
                }
            }
        )
        binding.recyclerView.adapter = contactAdapter

        mainSdkViewModel.listContact.observeOnce { listNeUser ->
            contactAdapter.setData(
                neContacts = listNeUser,
                lifecycleScope = lifecycleScope
            )
        }
        mainSdkViewModel.syncContact(SyncType.LIST_SYNC_DB)
    }

    override fun initData() {

        binding.btnGroupChat.clickDebounce {
            mainSdkViewModel.listContact.removeObservers(viewLifecycleOwner)
            viewModel.openCreateGroupDialog()
        }

        binding.btnCreateChannel.clickDebounce {
            mainSdkViewModel.listContact.removeObservers(viewLifecycleOwner)
            viewModel.openCreateGroupDialog(isChannel = true)
        }

        binding.btnSecretChat.clickDebounce {
            mainSdkViewModel.listContact.removeObservers(viewLifecycleOwner)
            viewModel.openCreateSecretGroupDialog()
        }

        binding.icClose.clickDebounce {
            dismiss()
        }

        binding.editSearch.addTextChangedListener(
            object : TextWatcher {
                private var searchFor = Constants.EMPTY
                override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
                }

                override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                    val searchText = s.toString().trim()
                    if (searchText == searchFor) {
                        return
                    }

                    searchFor = searchText

                    lifecycleScope.launch {
                        delay(500) // debounce timeOut
                        if (searchText != searchFor) {
                            return@launch
                        }
                        contactAdapter.filter(
                            filterBy = searchFor,
                            lifecycleScope = lifecycleScope
                        )
                    }
                }

                override fun afterTextChanged(p0: Editable?) {
                }
            }
        )
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(tvGroupChat, ivGroupChat, tvSecretChat, ivSecretChat, icClose, ivCreateChannel, tvCreateChannel, layoutEmpty.tvEmpty)
            themeHelperImpl.setThemeColorForSubBackground(ivGroupChat, ivSecretChat, ivCreateChannel)
            themeHelperImpl.setButtonBackgroundAndText(layoutEmpty.btnEmpty)
        }
    }
}
