package com.netacom.full.ui.main.group

import android.content.Intent
import android.graphics.Bitmap
import android.view.View
import android.widget.ImageView
import androidx.core.content.ContextCompat
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.navigation.fragment.navArgs
import androidx.viewpager.widget.ViewPager
import com.netacom.base.chat.android_utils.ImageUtils
import com.netacom.base.chat.android_utils.IntentUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.define.DialogDef
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.DateTimeUtils
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDialog
import com.netacom.full.databinding.FragmentViewPhotoBinding
import com.netacom.full.ui.main.group.adapter.ViewPhotoPagerAdapter
import com.netacom.full.ui.main.group.adapter.ViewVideoPagerAdapter
import com.netacom.full.widget.video.PlayerExoHelper
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.MessageType
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.media.NeMedia
import com.netacom.lite.repository.DownloadRepository
import com.netacom.lite.util.FileUtils
import dagger.hilt.android.AndroidEntryPoint
import javax.inject.Inject

@AndroidEntryPoint
class ViewMediaFragment :
    BaseSDKDialog<GroupViewModel, FragmentViewPhotoBinding>(
        R.layout.fragment_view_photo,
        GroupViewModel::class.java
    ),
    GalleryFragment.GalleryCallBack {
    @Inject
    lateinit var playerExoHelper: PlayerExoHelper

    override fun setLayoutHeight() = DialogDef.LAYOUT_FULL

    override fun setViewModel(): Int = BR.viewModel

    private val args: ViewMediaFragmentArgs by navArgs()

    private var galleryList = arrayListOf<NeMedia>()

    private var neMedia: NeMedia? = null

    private var activePosition: Int? = null

    private var viewVideoPagerAdapter: ViewVideoPagerAdapter? = null

    private var viewPhotoPagerAdapter: ViewPhotoPagerAdapter? = null

    private var galleryTabType = MessageType.MESSAGE_TYPE_IMAGE

    private var totalSize: Int = 1

    private var indexMediaSelected: Int = 0

    private var listImages: MutableList<NeMedia> = mutableListOf()
    private var listVideos: MutableList<NeMedia> = mutableListOf()
    private var listFiles: MutableList<NeMedia> = mutableListOf()

    private var galleryFragment: GalleryFragment? = null

    override fun initViews() {
        neMedia = args.neMedia
        Logger.e("display image : ${neMedia?.mediaUid}")

        viewModel.mediaDetail.observeOnce {
            totalSize = it.total ?: 1
            indexMediaSelected = it.index ?: 0
        }

        neMedia?.let { media ->
            galleryTabType = media.mediaType

            args.neGroup.let { group ->
                setupGalleryFragment(group)
                group.id?.toLongOrNull()?.let { groupId ->
                    viewModel.getMediaDetailForGroup(groupId, messageId = neMedia?.messId, mediaType = media.mediaType, mediaUid = neMedia?.mediaUid)
                    viewModel.getMediaForGroup(groupId, mediaType = media.mediaType, isViewMediaFragment = true)
                }
            }
        }

        binding.tbHeader.setColorTextRight(ContextCompat.getColor(requireContext(), R.color.color_white))
    }

    private fun setupData() {
        when (galleryTabType) {
            MessageType.MESSAGE_TYPE_IMAGE -> {
                binding.clViewPhoto.isVisible = true
                binding.clViewVideo.isGone = true
                setupPhotoAdapter()
            }
            MessageType.MESSAGE_TYPE_VIDEO -> {
                binding.clViewVideo.isVisible = true
                binding.clViewPhoto.isGone = true
                setupVideoAdapter(viewModel.downloadRepository)
            }
        }
    }

    override fun initData() {
        binding.icAll.clickDebounce {
            showGallery()
        }

        binding.tbHeader.setOnRightClickListener {
            showGallery()
        }

        binding.ivSaveAction.clickDebounce {
            downloadImage()
        }

        binding.icClose.clickDebounce {
            viewModel.onBack()
        }

        binding.tbHeader.setOnLeftClickListener {
            viewModel.onBack()
        }

        binding.toolbarGallery.setOnLeftClickListener {
            viewModel.onBack()
        }

        binding.tvAnotherAction.clickDebounce {
            viewModel.showImageActionPopup()
        }

        viewModel.onActionImage.observeOnce {
            when (it) {
                Action.SHARE -> FileUtils.getImageUriShare(requireContext(), getCurrentImageBitmap())?.let { uri ->
                    val shareIntent = Intent.createChooser(IntentUtils.getShareImageIntent(uri), getString(R.string.app_name))
                    startActivity(shareIntent)
                }
                Action.FORWARD -> {
                    /*
                    * Action forward 1 media:
                    * => get current media path
                    * => show popup select group for forward
                    * => choose group want to forward
                    * => send path to popup
                    * */

                    /*
                       * Check url media is available
                       * => First step is get file from Glide Cache
                       * => if have file from cache => send to forward
                       * => if not => download from server and => send to forward
                       */
                    FileUtils.convertBitmapToUriTemp(getCurrentImageBitmap())?.apply {
                        try {
                            val filePath = FileUtils.getRealPathFromURI(requireContext(), this)
                            if (filePath.isNotNull) {
                                viewModel.openForwardDialog(filePath = filePath)
                            } else {
                                downloadImage(isForward = true)
                            }
                        } catch (e: Exception) {
                            downloadImage(isForward = true)
                        }
                    }
                }
                Action.DOWNLOAD -> downloadImage()
            }
        }

        viewModel.listNeMediaFragment.observeOnce {
            if (it.isNotEmpty()) {
                galleryList.clear()

                galleryList.addAll(it)

                galleryFragment?.setDataList(it.toMutableList(), false)
                when (galleryTabType) {
                    MessageType.MESSAGE_TYPE_IMAGE -> {
                        listImages = it.toMutableList()
                    }
                    MessageType.MESSAGE_TYPE_VIDEO -> {
                        listVideos = it.toMutableList()
                    }
                    MessageType.MESSAGE_TYPE_FILE -> {
                        listFiles = it.toMutableList()
                    }
                }
            }
            setupData()
        }

        viewModel.listNeMediaFragmentLoadMore.observeOnce {
            if (it.isNotEmpty()) {
                val oldLastPosition = it.size
                galleryList.addAll(it)

                viewPhotoPagerAdapter?.setData(galleryList)

                binding.viewPager.adapter?.notifyDataSetChanged()

                binding.viewPager.setCurrentItem(oldLastPosition, false)

                galleryFragment?.setDataList(it.toMutableList(), true)
                when (galleryTabType) {
                    MessageType.MESSAGE_TYPE_IMAGE -> {
                        listImages.addAll(it)
                    }
                    MessageType.MESSAGE_TYPE_VIDEO -> {
                        listVideos.addAll(it)
                    }
                    MessageType.MESSAGE_TYPE_FILE -> {
                        listFiles.addAll(it)
                    }
                }
            }
        }
    }

    override fun setupTheme() {
        with(binding) {
            themeHelperImpl.setThemeColorForViews(toolbarGallery.getButtonLeft()!!)
        }
    }

    private fun setupPhotoAdapter(isFirstInit: Boolean = true) {
        val photoList = arrayListOf<NeMedia>()
        galleryList.filter {
            it.mediaType == MessageType.MESSAGE_TYPE_IMAGE
        }.reversed().let {
            photoList.addAll(it)
        }

        viewPhotoPagerAdapter = ViewPhotoPagerAdapter(
            requireContext(),
            photoList,
            viewModel.getGroupDbById(neMedia?.groupId.toString())?.groupType == GroupType.GROUP_TYPE_PRIVATE
        ) {
            // long click
            viewModel.showSaveImagePopup()
        }

        binding.viewPager.adapter = viewPhotoPagerAdapter

        binding.viewPager.addOnPageChangeListener(
            object : ViewPager.SimpleOnPageChangeListener() {
                override fun onPageSelected(positionInPager: Int) {
                    // To do
                    Logger.e("position in pager ==$positionInPager")
                    val position = galleryList.size - positionInPager
                    activePosition = position
                    val item = galleryList[positionInPager]
                    binding.tvSenderName.text = item.owner?.getDisplayName ?: ""
                    binding.tvSentTime.text = DateTimeUtils.ParseDateTimeToString.parseJustDate(item.msgCreateAt)
                    updatePosition(position)

                    // load more data when swipe view pager reach to limit
                    if (positionInPager == 0) {
                        val groupId = neMedia?.groupId ?: return
                        viewModel.loadMoreMedia(groupId = groupId, offset = galleryList.size, mediaType = galleryTabType, isViewMediaFragment = true)
                    }
                }
            }
        )

        val position = photoList.reversed().find {
            neMedia?.mediaUid == it.mediaUid
        }?.let {
            photoList.indexOf(it)
        }

        Logger.d("display position : $position and size : ${photoList.size}")
        if (isFirstInit) {
            position?.let {
                binding.viewPager.currentItem = it
                activePosition = it
                binding.tvSenderName.text = neMedia?.owner?.getDisplayName ?: ""
                binding.tvSentTime.text =
                    DateTimeUtils.ParseDateTimeToString.parseJustDate(neMedia?.msgCreateAt ?: 0L)
                updatePosition(indexMediaSelected + 1)
            }
        }
    }

    private fun setupVideoAdapter(downloadRepository: DownloadRepository) {
        val videoList = galleryList.filter {
            it.mediaType == MessageType.MESSAGE_TYPE_VIDEO
        }.reversed()

        binding.viewPagerVideo.adapter = preferencesHelperImpl.token?.let {
            viewVideoPagerAdapter = ViewVideoPagerAdapter(
                requireContext(),
                videoList,
                it,
                viewModel.getGroupDbById(neMedia?.groupId.toString())?.groupType == GroupType.GROUP_TYPE_PRIVATE,
                downloadRepository,
                playerExoHelper
            )
            viewVideoPagerAdapter
        }

        binding.viewPagerVideo.addOnPageChangeListener(
            object : ViewPager.SimpleOnPageChangeListener() {
                override fun onPageSelected(position: Int) {
                    val selectedPosition: Int
                    if (position >= 0 && (videoList.size - 1) >= position) {
                        selectedPosition = position
                        videoList[position]
                    } else {
                        selectedPosition = videoList.size - 1
                        videoList.last()
                    }
                    updatePosition(selectedPosition)
                }
            }
        )
        val position = videoList.find {
            neMedia?.mediaUid == it.mediaUid
        }?.let {
            videoList.indexOf(it)
        }
        position?.let {
            binding.viewPagerVideo.currentItem = it
        }
    }

    private fun updatePosition(position: Int) {
        // reverse position
        binding.tbHeader.setTitle("$position/$totalSize")
    }

    private fun showMediaFromGallery(neMedia: NeMedia) {
        binding.clPhotoGallery.isGone = true
        val position = galleryList.filter {
            it.mediaType == neMedia.mediaType
        }.reversed().indexOf(neMedia)

        when (neMedia.mediaType) {
            MessageType.MESSAGE_TYPE_IMAGE -> {
                binding.clViewPhoto.isVisible = true
                binding.clViewVideo.isVisible = false

                binding.viewPager.currentItem = position
            }
            MessageType.MESSAGE_TYPE_VIDEO -> {
                binding.clViewVideo.isVisible = true
                binding.clViewPhoto.isVisible = false

                binding.viewPagerVideo.currentItem = position
            }
        }
    }

    private fun showGallery() {
        binding.clViewPhoto.isGone = true
        binding.clViewVideo.isGone = true
        binding.clPhotoGallery.isVisible = true
    }

    private fun downloadImage(isForward: Boolean = false) {
        neMedia?.mediaUid?.let { linkUrl ->
            viewModel.downloadFile(
                fileUrl = linkUrl,
                isForward = isForward,
                callbackResult = { isSuccess, tempFilePath ->
                    if (isForward) {
                        if (isSuccess) {
                            viewModel.openForwardDialog(tempFilePath)
                        }
                    } else {
                        if (isSuccess) {
                            showSnackBar(R.string.str_save_success)
                        } else {
                            showSnackBar(R.string.str_save_fail)
                        }
                    }
                }
            )
        } ?: showSnackBar(R.string.str_save_fail)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewVideoPagerAdapter?.releasePlayer()
        viewModel.mediaData.removeObservers(viewLifecycleOwner)
    }

    enum class Action {
        FORWARD,
        SHARE,
        DOWNLOAD
    }

    private fun getCurrentImageBitmap(): Bitmap? {
        activePosition?.let { _ ->
            val view: View = binding.viewPager.findViewWithTag(activePosition)
            val imageView: ImageView = view.findViewById(R.id.iv_view_photo_pager)
            return ImageUtils.view2Bitmap(imageView)
            // TODO: COIL
            /*val memoryCacheKey = imageView.metadata?.memoryCacheKey
            memoryCacheKey?.apply {
                return requireContext().imageLoader.memoryCache[this]
            }*/
        }
        return null
    }

    private fun setupGalleryFragment(group: NeGroup) {
        if (galleryFragment == null) {
            galleryFragment = GalleryFragment().apply {
                this.setGroup(group)
                this.setIsViewMedia(true)
                this.setIsViewSingle(false)
                this.setGalleryCallBack(this@ViewMediaFragment)
            }
            childFragmentManager.beginTransaction().add(R.id.flGallery, galleryFragment!!).commit()
        }
    }

    override fun onOpenMedia(media: NeMedia) {
        showMediaFromGallery(media)
    }
}
