package com.netacom.full.ui.main.group.adapter

import android.view.View
import android.view.ViewGroup
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.LifecycleCoroutineScope
import androidx.recyclerview.widget.DiffUtil
import com.netacom.base.chat.adapter.BaseMultiTypeListSimpleAdapter
import com.netacom.base.chat.adapter.BaseViewHolder
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.unAccentText
import com.netacom.full.R
import com.netacom.full.databinding.ItemRcvUserGroupHeaderBinding
import com.netacom.full.databinding.ItemRcvUserMessageBinding
import com.netacom.full.ui.main.contact.ContactUtils
import com.netacom.full.ui.main.contact.model.SubNeContact
import com.netacom.lite.entity.ui.user.NeUser
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class NewMessageAdapter(
    itemClick: ((SubNeContact, View, Int) -> Unit)?,
    val emptyCallBack: ((Boolean) -> Unit)?
) : BaseMultiTypeListSimpleAdapter<SubNeContact, ViewDataBinding>(itemClick, diff = NewMessageDiffUtil()) {

    companion object {
        private const val CONTACT_HEADER = 0
        private const val CONTACT_BODY = 1
    }

    private val neContacts = mutableListOf<NeUser>()

    override fun bindMultiType(position: Int): Int {
        return if (getItem(position)?.isHeader() == true) CONTACT_HEADER else CONTACT_BODY
    }

    fun setData(neContacts: List<NeUser>, lifecycleScope: LifecycleCoroutineScope) {
        lifecycleScope.launch(Dispatchers.IO) {
            this@NewMessageAdapter.neContacts.clear()
            this@NewMessageAdapter.neContacts.addAll(neContacts)
            val newList = ContactUtils.sortContacts(listContacts = neContacts)
            withContext(Dispatchers.Main) {
                display(newList)
            }
        }
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder<ViewDataBinding> {
        return when (viewType) {
            CONTACT_HEADER -> BaseViewHolder(
                inflateView(parent, R.layout.item_rcv_user_group_header) as ItemRcvUserGroupHeaderBinding
            )
            CONTACT_BODY -> BaseViewHolder(
                inflateView(parent, R.layout.item_rcv_user_message) as ItemRcvUserMessageBinding
            )
            else -> BaseViewHolder(
                inflateView(parent, R.layout.item_rcv_user_message) as ItemRcvUserMessageBinding
            )
        }
    }

    override fun onBindViewHolder(holder: BaseViewHolder<ViewDataBinding>, position: Int) {
        getItem(position)?.let { item ->
            with(holder) {
                when (getItemViewType(position)) {
                    CONTACT_HEADER -> bindContactHeader(
                        this,
                        subNeContact = item
                    )
                    CONTACT_BODY -> bindContactBody(
                        this,
                        subNeContact = item,
                        isDivider = (layoutPosition != (itemCount - 1)),
                        position = position
                    )
                }
                binding.executePendingBindings()
            }
        }
    }

    private fun bindContactHeader(
        holder: BaseViewHolder<ViewDataBinding>,
        subNeContact: SubNeContact
    ) {
        with(holder.binding as ItemRcvUserGroupHeaderBinding) {
            this.header = subNeContact._neHeader ?: "#"
        }
    }

    private fun bindContactBody(
        holder: BaseViewHolder<ViewDataBinding>,
        subNeContact: SubNeContact,
        isDivider: Boolean,
        position: Int
    ) {
        with(holder.binding as ItemRcvUserMessageBinding) {
            this.item = subNeContact._itemNeContact?._neContact
            this.hasDivider = isDivider

            itemClick?.let { click ->
                holder.itemView.clickDebounce {
                    click.invoke(subNeContact, holder.itemView, position)
                }
            }
        }
    }

    fun filter(filterBy: String, lifecycleScope: LifecycleCoroutineScope) {
        lifecycleScope.launch(Dispatchers.IO) {
            val termContact = neContacts.filter {
                var result: Boolean
                result = it.getDisplayName.unAccentText()?.contains(
                    filterBy,
                    ignoreCase = true
                ) ?: false
                if (!result) {
                    result = it.phone?.contains(filterBy, ignoreCase = true) == true
                }
                Logger.d("_InviteFriendAdapter result = $result")
                result
            }

            val newList = ContactUtils.sortContacts(listContacts = termContact)
            withContext(Dispatchers.Main) {
                emptyCallBack?.let { callback ->
                    callback(newList.isEmpty())
                }
                display(newList)
            }
        }
    }

    private class NewMessageDiffUtil : DiffUtil.ItemCallback<SubNeContact>() {
        override fun areItemsTheSame(oldItem: SubNeContact, newItem: SubNeContact): Boolean {
            return if (oldItem.isHeader() && newItem.isHeader()) {
                true
            } else !oldItem.isHeader() && !newItem.isHeader()
        }

        override fun areContentsTheSame(oldItem: SubNeContact, newItem: SubNeContact): Boolean {
            if (!oldItem.isHeader() && !newItem.isHeader()) {
                return oldItem._itemNeContact?._neContact?.phone == newItem._itemNeContact?._neContact?.phone
            }
            return false
        }
    }
}
