package com.netacom.full.ui.main.theme

import android.content.res.ColorStateList
import android.graphics.Color
import android.os.Build
import android.view.View
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.DrawableCompat
import com.netacom.base.chat.android_utils.BarUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.getThemeColor
import com.netacom.base.chat.util.resIdByNameColor
import com.netacom.base.chat.util.resIdByNameDrawable
import com.netacom.base.chat.util.setBackgroundTheme
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.basechat.BaseSDKDataFragment
import com.netacom.full.databinding.FragmentThemeBinding
import com.netacom.full.databinding.setImageColorTheme
import com.netacom.full.ui.main.theme.adapter.ColorAdapter
import com.netacom.full.utils.ThemeUtils
import com.netacom.lite.util.Constants
import dagger.hilt.android.AndroidEntryPoint

@AndroidEntryPoint
class ThemeFragment : BaseSDKDataFragment<FragmentThemeBinding, ThemeViewModel>(
    R.layout.fragment_theme,
    ThemeViewModel::class.java
) {

    private lateinit var colorAdapter: ColorAdapter

    override fun setViewModel(): Int = BR.viewModel

    override fun initViews() {
        colorAdapter = ColorAdapter(
            requireContext(),
            { neTheme, _, _ ->
                themeHelperImpl.neTheme = neTheme
                applyColorForPreviewChat(requireContext().resIdByNameColor(neTheme.mainColor))
                applyColorForToolbar(requireContext().resIdByNameDrawable(neTheme.toolbarDrawable)!!)
                setSwitchColor(requireContext().resIdByNameColor(neTheme.mainColor))
                activity?.apply {
                    if (binding.isDarkMode == true) {
                        BarUtils.setNavBarColor(this, ContextCompat.getColor(requireContext(), R.color.color_black))
                    } else {
                        BarUtils.setNavBarColor(this, getThemeColor(themeHelperImpl.mainColor))
                    }
                }
            },
            viewModel.appId(),
            themeHelperImpl.mainColor
        )

        binding.recyclerViewColor.apply {
            adapter = colorAdapter
        }

        when (themeHelperImpl.displayMode) {
            Constants.LIGHT_MODE -> {
                applyLightMode()
            }

            Constants.DARK_MODE -> {
                applyDarkMode()
            }

            Constants.SYSTEM_MODE -> {
                applyThemeFollowSystem()
                binding.switchAutoColor.isChecked = true
            }
        }
        binding.layoutPreviewLight.clickDebounce {
            applyLightMode()
            binding.switchAutoColor.isChecked = false
        }

        binding.layoutPreviewDark.clickDebounce {
            applyDarkMode()
            binding.switchAutoColor.isChecked = false
        }

        binding.toolbar.setOnLeftClickListener {
            viewModel.onBack()
        }

        binding.switchAutoColor.setOnCheckedChangeListener { compoundButton, checked ->
            if (checked) {
                applyThemeFollowSystem()
            }
        }

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            binding.rlAutoChange.visibility = View.GONE
        }

        setSwitchColor(themeHelperImpl.mainColor)
        applyColorForPreviewChat(themeHelperImpl.mainColor)
    }

    private fun applyThemeFollowSystem() {
        val darkModeOnSystem = ThemeUtils.isDarkModeInSystem

        if (darkModeOnSystem) {
            applyDarkMode()
        } else {
            applyLightMode()
        }

        themeHelperImpl.displayMode = Constants.SYSTEM_MODE
    }

    override fun initData() {
    }

    override fun syncEvent() {
    }

    private fun applyLightMode() {
        Logger.d("Liem - apply Light mode")

        binding.isDarkMode = false

        binding.layoutPreviewDark.setBackgroundResource(R.drawable.bg_theme_dark_setting)
        binding.layoutPreviewLight.setBackgroundResource(R.drawable.theme_preview_light_checked)

        themeHelperImpl.displayMode = Constants.LIGHT_MODE

        themeHelperImpl.setThemeColorForBorder(binding.layoutPreviewLight.background)

        applyColorForToolbar(themeHelperImpl.toolbarDrawableId)

        activity?.let {
            BarUtils.setNavBarColor(it, getThemeColor(themeHelperImpl.mainColor))
        }
    }

    private fun applyDarkMode() {
        Logger.d("Liem - apply Dark mode")

        binding.isDarkMode = true

        binding.layoutPreviewLight.setBackgroundResource(R.drawable.bg_theme_light_setting)
        binding.layoutPreviewDark.setBackgroundResource(R.drawable.theme_preview_dark_checked)

        themeHelperImpl.displayMode = Constants.DARK_MODE

        themeHelperImpl.setThemeColorForBorder(binding.layoutPreviewDark.background)

        applyColorForToolbar(themeHelperImpl.toolbarDrawableId)

        activity?.let {
            BarUtils.setNavBarColor(it, ContextCompat.getColor(requireContext(), R.color.color_black))
        }
    }

    private fun applyColorForPreviewChat(colorId: Int) {

        // change color for sender messsage
        binding.txtDarkSenderMessage.setBackgroundTheme(colorId)
        binding.txtLightSenderMessage.setBackgroundTheme(colorId)

        // change color for ic sent
        binding.imgSendLight.setImageColorTheme(colorId)
        binding.imgSendDark.setImageColorTheme(colorId)

        // change border color
        if (binding.isDarkMode == true) {
            themeHelperImpl.setThemeColorForBorder(binding.layoutPreviewDark.background)
        } else {
            themeHelperImpl.setThemeColorForBorder(binding.layoutPreviewLight.background)
        }
    }

    private fun applyColorForToolbar(drawableId: Int) {
        binding.toolbar.setBackgroundTheme(viewModel.appId(), themeHelperImpl.isDarkMode, drawableId)
    }

    override fun onStop() {
        super.onStop()
        ThemeUtils.enableDisplayMode(themeHelperImpl.displayMode)
    }

    private fun setSwitchColor(colorId: Int) {
        DrawableCompat.setTintList(
            binding.switchAutoColor.thumbDrawable,
            ColorStateList(
                arrayOf(
                    intArrayOf(-android.R.attr.state_checked), // Disabled
                    intArrayOf() // Enabled
                ),
                intArrayOf(
                    Color.WHITE, // The color for the Disabled state
                    Color.WHITE, // The color for the Enabled state
                )
            )
        )

        DrawableCompat.setTintList(
            binding.switchAutoColor.trackDrawable,
            ColorStateList(
                arrayOf(
                    intArrayOf(-android.R.attr.state_checked), // Disabled
                    intArrayOf() // Enabled
                ),
                intArrayOf(
                    Color.GRAY, // The color for the Disabled state
                    ContextCompat.getColor(requireContext(), colorId), // The color for the Enabled state
                )
            )
        )
    }

    override fun setupTheme() {
    }
}
