package com.netacom.full.ui.main.theme

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.drawable.Drawable
import android.graphics.drawable.GradientDrawable
import android.graphics.drawable.StateListDrawable
import android.util.StateSet
import android.view.View
import android.widget.ImageView
import android.widget.ProgressBar
import android.widget.TextView
import androidx.appcompat.widget.AppCompatButton
import androidx.appcompat.widget.AppCompatEditText
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import com.google.android.material.bottomnavigation.BottomNavigationView
import com.google.android.material.tabs.TabLayout
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.asColor
import com.netacom.base.chat.util.getThemeColor
import com.netacom.base.chat.util.isNotNull
import com.netacom.base.chat.util.isNotZero
import com.netacom.base.chat.util.isResource
import com.netacom.base.chat.util.resIdByNameColor
import com.netacom.base.chat.util.resIdByNameDrawable
import com.netacom.full.R
import com.netacom.full.databinding.setBackgroundColorTheme
import com.netacom.full.databinding.setButtonTheme
import com.netacom.full.databinding.setImageColorTheme
import com.netacom.full.databinding.setProgressTintListTheme
import com.netacom.full.databinding.setTabLayoutTheme
import com.netacom.full.databinding.setTextColorTheme
import com.netacom.full.databinding.setupNavigationTheme
import com.netacom.full.define.ChatMessageType
import com.netacom.full.utils.ThemeUtils
import com.netacom.full.widget.toolbar.CoreToolBar
import com.netacom.lite.define.GroupType.GROUP_TYPE_PRIVATE
import com.netacom.lite.entity.ui.theme.NeTheme
import com.netacom.lite.local.prefs.PreferencesHelperImpl
import com.netacom.lite.sdk.AppID
import com.netacom.lite.sdk.SdkConfig
import com.netacom.lite.util.Constants
import dagger.hilt.android.qualifiers.ApplicationContext
import javax.inject.Inject

class ThemeHelperImpl @Inject constructor(
    private val preferencesHelperImpl: PreferencesHelperImpl,
    @ApplicationContext private val application: Context
) {

    private fun sdkConfig() = preferencesHelperImpl.sdkConfig ?: SdkConfig()

    var neTheme: NeTheme?
        get() = preferencesHelperImpl.neTheme
        set(value) {
            preferencesHelperImpl.neTheme = value
        }

    val mainColor: Int
        get() = if (preferencesHelperImpl.neTheme.isNotNull && application.resIdByNameColor(preferencesHelperImpl.neTheme!!.mainColor).isNotZero) {
            application.resIdByNameColor(preferencesHelperImpl.neTheme!!.mainColor)
        } else if (preferencesHelperImpl.neTheme?.mainColor?.startsWith("#") == true) {
            preferencesHelperImpl.neTheme?.mainColor?.asColor ?: R.color.color_orange_f5783f
        } else when (sdkConfig().appId) {
            AppID.GALO -> R.color.color_green_gchat
            else -> R.color.color_orange_f5783f
        }

    val subColor: Int
        get() = if (preferencesHelperImpl.neTheme.isNotNull && application.resIdByNameColor(preferencesHelperImpl.neTheme!!.subColor).isNotZero) {
            application.resIdByNameColor(preferencesHelperImpl.neTheme!!.subColor)
        } else if (preferencesHelperImpl.neTheme?.subColor?.startsWith("#") == true) {
            preferencesHelperImpl.neTheme?.subColor?.asColor ?: R.color.color_orange_4df5783f
        } else when (sdkConfig().appId) {
            AppID.GALO -> R.color.color_green_sub_gchat
            else -> R.color.color_orange_4df5783f
        }

    val toolbarDrawableId: Int
        get() = preferencesHelperImpl.neTheme?.let {
            when {
                application.resIdByNameDrawable(it.toolbarDrawable)!!.isNotZero -> {
                    application.resIdByNameDrawable(it.toolbarDrawable)
                }
                application.resIdByNameColor(it.toolbarDrawable).isNotZero -> {
                    application.resIdByNameColor(it.toolbarDrawable)
                }
                it.toolbarDrawable.startsWith("#") -> {
                    Logger.e("toolbarDrawable====" + it.toolbarDrawable)
                    it.toolbarDrawable.asColor ?: R.drawable.bg_header_orange
                }
                else -> {
                    when (sdkConfig().appId) {
                        AppID.GALO -> R.color.color_green_gchat
                        else -> R.drawable.bg_header_orange
                    }
                }
            }
        } ?: when (sdkConfig().appId) {
            AppID.GALO -> R.color.color_green_gchat
            else -> R.drawable.bg_header_orange
        }

    val referralDrawableId: Int
        get() = if (preferencesHelperImpl.neTheme.isNotNull && preferencesHelperImpl.neTheme!!.colorIdReferral != null && application.resIdByNameDrawable(preferencesHelperImpl.neTheme!!.colorIdReferral)!!.isNotZero) {
            application.resIdByNameDrawable(preferencesHelperImpl.neTheme!!.colorIdReferral) ?: 0
        } else R.drawable.ic_bg_referral_code_orange

    private val menuColor: Int
        get() = if (preferencesHelperImpl.neTheme.isNotNull && application.resIdByNameColor(preferencesHelperImpl.neTheme!!.menuColor).isNotZero) {
            application.resIdByNameColor(preferencesHelperImpl.neTheme!!.menuColor)
        } else when (sdkConfig().appId) {
            AppID.GALO -> R.color.bottom_navigation_menu_green
            else -> R.color.bottom_navigation_menu_orange
        }

    val isDefaultColor: Boolean
        get() = preferencesHelperImpl.neTheme?.isDefaultColor ?: false

    var displayMode: Int
        get() = preferencesHelperImpl.displayMode
        set(value) {
            preferencesHelperImpl.displayMode = value
        }
    val isDarkMode: Boolean
        get() = when (displayMode) {
            Constants.DARK_MODE -> {
                true
            }
            Constants.SYSTEM_MODE -> {
                ThemeUtils.isOsDarkTheme(application)
            }
            else -> {
                false
            }
        }

    private val secretColor: Int
        get() = if (isDarkMode) {
            R.color.color_grey_a0a0a0
        } else {
            R.color.color_black_25272a
        }

    private val secretMessageColor: Int get() = R.color.color_black_4b4c4e

    fun setThemeColorForViews(vararg views: View) {
        views.forEach {
            when (it) {
                is AppCompatTextView -> it.setTextColorTheme(mainColor)
                is AppCompatEditText -> it.setTextColorTheme(mainColor)
                is ImageView -> it.setImageColorTheme(mainColor)
                is CoreToolBar -> it.setBackgroundTheme(
                    sdkConfig().appId,
                    isDarkMode,
                    toolbarDrawableId
                )
                is BottomNavigationView -> {
                    it.setupNavigationTheme(createThemeMenu())
                }
                is AppCompatButton -> {
                    it.setButtonTheme(mainColor)
                }
                is TabLayout -> {
                    it.setTabLayoutTheme(mainColor)
                }
                is ProgressBar -> {
                    it.setProgressTintListTheme(mainColor)
                }
                else -> {
                    it.setBackgroundColorTheme(mainColor)
                }
            }
        }
    }

    fun setThemeColorForSecretChat(vararg views: View) {
        try {
            views.forEach {
                when (it) {
                    is AppCompatTextView -> it.setTextColorTheme(secretColor)
                    is AppCompatEditText -> it.setTextColorTheme(secretColor)
                    is ImageView -> it.setImageColorTheme(secretColor)
                    is AppCompatButton -> {
                        it.setButtonTheme(secretColor)
                    }
                    is TabLayout -> {
                        it.setTabLayoutTheme(secretColor)
                    }
                    is ProgressBar -> {
                        it.setProgressTintListTheme(secretColor)
                    }
                    else -> {
                        it.setBackgroundColorTheme(secretColor)
                    }
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    fun setHeaderTheme(view: View) {
        if (displayMode == Constants.DARK_MODE) {
            view.setBackgroundColor(ContextCompat.getColor(Utils.getApp(), R.color.color_black_25272a))
        } else if (displayMode == Constants.LIGHT_MODE) {
            when {
                isResource(view.context, toolbarDrawableId) -> view.setBackgroundResource(toolbarDrawableId)
                else -> view.setBackgroundColor(toolbarDrawableId)
            }
        }
    }

    private fun setThemeColorForViews(views: List<View>) {
        views.forEach {
            when (it) {
                is TextView -> it.setTextColorTheme(mainColor)
                is ImageView -> it.setImageColorTheme(mainColor)
            }
        }
    }

    fun setThemeStateListForButtonAction(view: View) {
        val stateListDrawable = StateListDrawable()
        stateListDrawable.addState(
            intArrayOf(android.R.attr.state_enabled),
            createShapeDrawable(getThemeColor(mainColor))
        ) // if activated true
        if (displayMode == Constants.DARK_MODE) {
            // Logger.d("Liem - button for dark mode")
            stateListDrawable.addState(
                StateSet.WILD_CARD,
                createShapeDrawable(ContextCompat.getColor(application, R.color.color_grey_b7b7b7))
            )
        } else {
            // Logger.d("Liem - button for light mode")
            stateListDrawable.addState(
                StateSet.WILD_CARD,
                createShapeDrawable(
                    ContextCompat.getColor(application, R.color.color_disable_btn)
                )
            )
        } // rest all the state
        view.background = stateListDrawable
    }

    fun createThemeColorStateList(): StateListDrawable {
        val stateListDrawable = StateListDrawable()
        stateListDrawable.addState(
            intArrayOf(android.R.attr.state_enabled),
            createShapeDrawable(ContextCompat.getColor(application, mainColor))
        ) // if activated true
        stateListDrawable.addState(
            StateSet.WILD_CARD,
            createShapeDrawable(
                ContextCompat.getColor(application, R.color.color_grey_3325272A)
            )
        )
        return stateListDrawable
    }

    private fun createThemeMenu(): ColorStateList {
        return ColorStateList(
            arrayOf(
                intArrayOf(-android.R.attr.state_checked), // Disabled
                intArrayOf(android.R.attr.state_checked) // Enabled
            ),
            intArrayOf(
                ContextCompat.getColor(Utils.getApp(), R.color.color_grey_b7b7b7), // The color for the Disabled state
                getThemeColor(mainColor) // The color for the Enabled state
            )
        )
    }

    private fun createShapeDrawable(color: Int): GradientDrawable {
        val shape = GradientDrawable()
        shape.shape = GradientDrawable.RECTANGLE
        shape.setColor(color)
        shape.cornerRadius = 4f
        return shape
    }

    fun setThemeColorForDrawable(drawables: Drawable?): Drawable? {
        drawables?.colorFilter = PorterDuffColorFilter(getThemeColor(mainColor), PorterDuff.Mode.SRC_IN)
        return drawables
    }

    fun setThemeColorForSubBackground(vararg views: View) {
        views.forEach {
            when (it) {
                is AppCompatImageView -> {
                    it.setBackgroundColorTheme(subColor)
                }
                else -> {
                    it.setBackgroundColorTheme(subColor)
                }
            }
        }
    }

    fun setThemeColorForBackground(view: View) {
        view.setBackgroundColorTheme(mainColor)
    }

    fun setThemeColorForSecretBackground(view: View) {
        view.setBackgroundColorTheme(secretColor)
    }

    fun setButtonBackgroundAndText(button: AppCompatButton) {
        button.setButtonTheme(mainColor)
        button.setTextColor(getThemeColor(mainColor))
    }

    fun setThemeColorForMessage(view: View, messageType: Int, groupType: Int) {
        // only set background opacity for background , needn't change
        when (messageType) {
            ChatMessageType.MY_MESSAGE_TEXT, ChatMessageType.MY_MESSAGE_REPLY
            -> view.setBackgroundResource(R.drawable.bg_single_sender)
            ChatMessageType.MY_MESSAGE_VIDEO, ChatMessageType.MY_MESSAGE_AUDIO, ChatMessageType.MY_MESSAGE_CALL, ChatMessageType.MY_MESSAGE_FILE
            -> view.setBackgroundResource(R.drawable.bg_media_opacity_45)
            ChatMessageType.PARTNER_MESSAGE_VIDEO, ChatMessageType.PARTNER_MESSAGE_AUDIO, ChatMessageType.PARTNER_MESSAGE_CALL, ChatMessageType.PARTNER_MESSAGE_FILE
            -> view.setBackgroundResource(R.drawable.bg_media_opacity_30)
        }

        // set color for message chat
        if (groupType == GROUP_TYPE_PRIVATE) {
            view.setBackgroundColorTheme(secretMessageColor)
        } else {
            view.setBackgroundColorTheme(mainColor)
        }
    }

    fun setThemeColorForBorder(drawable: Drawable) {
        val strokeDrawable = drawable as? GradientDrawable
        strokeDrawable?.setStroke(2, ContextCompat.getColor(application, mainColor))
    }

    fun resetToDefault() {
        preferencesHelperImpl.neTheme = null
        preferencesHelperImpl.displayMode = Constants.LIGHT_MODE
        ThemeUtils.enableDisplayMode(Constants.LIGHT_MODE)
    }
}
