/*
  Created by NetaloTeamAndroid on 2020
 * Company: Netacom.

 */

package com.netacom.full.ui.sdk

import android.Manifest
import android.annotation.SuppressLint
import android.app.KeyguardManager
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Parcelable
import android.view.WindowManager
import androidx.annotation.Keep
import androidx.core.os.bundleOf
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.ViewModelProvider
import androidx.navigation.NavController
import androidx.navigation.findNavController
import androidx.navigation.ui.setupWithNavController
import com.fondesa.kpermissions.extension.permissionsBuilder
import com.fondesa.kpermissions.extension.send
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.LanguageUtils
import com.netacom.base.chat.android_utils.UriUtils
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.isNotNull
import com.netacom.base.chat.util.launchActivity
import com.netacom.full.BR
import com.netacom.full.R
import com.netacom.full.databinding.ActivityNetaloBinding
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.extensions.navigateIfSafe
import com.netacom.full.ui.login.LoginActivity
import com.netacom.full.ui.main.MainSdkActivity
import com.netacom.full.ui.main.MainSdkViewModel
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.full.utils.DialogUtil
import com.netacom.lite.define.NavigationDef
import com.netacom.lite.entity.socket.Call
import com.netacom.lite.entity.ui.user.NeUser
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import javax.inject.Inject

/**
Created by vantoan on 28,July,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 */

@Keep
@AndroidEntryPoint
class NetAloActivity : MainSdkActivity<ActivityNetaloBinding, NetAloViewModel>(
    R.layout.activity_netalo,
    NetAloViewModel::class.java
) {
    @Inject
    lateinit var navigationDispatcher: Dispatcher<(NavController) -> Unit>

    @Inject
    lateinit var themeHelper: ThemeHelperImpl

    private var neUser: NeUser? = null

    override fun setViewModel(): Int = BR.viewModel

    private val navController by lazy {
        findNavController(R.id.nav_host)
    }

    private val mainSdkViewModel: MainSdkViewModel by lazy {
        ViewModelProvider(this).get(MainSdkViewModel::class.java)
    }

    companion object {
        private val IS_SDK = "IS_SDK"
        private val USER_CHAT_NETALO = "USER_CHAT_NETALO"
        private val CALL = "CALL"
        private val NOTIFICATION = "NOTIFICATION"

        fun startSDK(
            context: Context,
            isSdk: Boolean = false,
            neUserChat: NeUser? = null,
            call: Call? = null,
            isNotification: Boolean = false
        ) {
            val intent = Intent(context, NetAloActivity::class.java)
            intent.putExtra(IS_SDK, isSdk)
            intent.putExtra(USER_CHAT_NETALO, neUserChat)
            intent.putExtra(CALL, call)
            intent.putExtra(NOTIFICATION, isNotification)
            call?.let {
                intent.addFlags(Intent.FLAG_ACTIVITY_EXCLUDE_FROM_RECENTS)
            }
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
            return
        }

        fun startGallery(context: Context, isShowDocumented: Boolean, maxSelections: Int, autoDismissOnMaxSelections: Boolean, galleryType: Long) {
            val intent = Intent(context, SdkActivity::class.java)
            intent.putExtra("maxSelections", maxSelections)
            intent.putExtra("isShowDocumented", isShowDocumented)
            intent.putExtra("autoDismissOnMaxSelections", autoDismissOnMaxSelections)
            intent.putExtra("galleryType", galleryType)
            context.startActivity(intent)
            return
        }

        fun initPermission(activity: FragmentActivity) {
            activity.permissionsBuilder(
                Manifest.permission.READ_CONTACTS, Manifest.permission.WRITE_CONTACTS, Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE, Manifest.permission.RECORD_AUDIO
            ).build().send { status ->
                Logger.e("initPermission==$status")
            }
        }
    }

    private val showNavFragments = listOf(R.id.groupFragment, R.id.callLogFragment, R.id.contactFragment, R.id.profileFragment)

    fun disConnectSocket() {
        try {
            mainSdkViewModel.disConnectSocket()
        } catch (e: Exception) {
            e.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(e)
        }
    }

    fun logOutSdk(callbackSuccess: () -> Unit) {
        try {
            mainSdkViewModel.logout(callbackSuccess)
        } catch (e: Exception) {
            e.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(e)
        }
    }

    private fun isNetAlo() = !intent.getBooleanExtra(IS_SDK, false)

    override fun initViews() {
        neUser = intent.extras?.getParcelable(NavigationDef.ARGUMENT_NEUSER) ?: intent.getParcelableExtra(USER_CHAT_NETALO)
        mainSdkViewModel.initSDK(neUser) {
            if (mainSdkViewModel.getPreferences.getUserId == 0L) {
                launchActivity<LoginActivity>()
                finishAffinity()
            }
        }
        initTheme(themeHelper)
        viewModel.checkAppUpdate(context = this, themeHelperImpl = themeHelper)
    }

    override fun onResume() {
        super.onResume()
        receiveShare()
        setupNavigationTheme()
        processReceiveCall()
    }

    override fun initData() {
        checkPermissionAll()
        setUpNavigation()
        navigationDispatcher.event.observe { event ->
            event.consume { command ->
                command(navController)
            }
        }
        viewModel.showReferral.observeOnce { popup ->
            if (popup.isNotNull && navController.graph.startDestination == navController.currentDestination?.id) {
                DialogUtil.showImage(
                    this,
                    popup.banner_url,
                    {
                        if (popup.click_url.isNotNull) {
                            viewModel.gotoReferral(popup.click_url)
                        }
                    },
                    {
                        viewModel.showNextReferral()
                    }
                )
            }
        }
        MainScope().launch {
            /*val flowList = netAloEvent.receive<ArrayList<LocalFileModel>>()
            flowList.collect { listPhoto ->
                Logger.e("SELECT_PHOTO_VIDEO==$listPhoto")
            }
            val flow = netAloEvent.receive<LocalFileModel>()
            flow.collect { document ->
                Logger.e("SELECT_FILE==$document")
            }*/
        }
        // TODO toan test crash
        // throw RuntimeException("Test Crash")
    }

    override fun onBackPressed() {
        if (neUser == null && navController.graph.startDestination == navController.currentDestination?.id) {
            Logger.e("onBackPressed11111=disConnectSocket:finish")
            if (isNetAlo()) {
                moveTaskToBack(true)
            } else {
                Logger.e("onBackPressed2222=disConnectSocket:finish")
                // disConnectSocket()
                finish()
            }
        } else if (neUser != null) {
            Logger.e("onBackPressed3333=disConnectSocket:finish")
            // disConnectSocket()
            finish()
        } else {
            Logger.e("onBackPressed44444=disConnectSocket:finish")
            super.onBackPressed()
        }
    }

    override fun onSupportNavigateUp(): Boolean = navController.navigateUp()

    public override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(newBase)
        LanguageUtils.attachBaseContext(this)
    }

    @SuppressLint("RestrictedApi")
    private fun setUpNavigation() {
        if (isNetAlo()) {
            with(navController) {
                binding.navView.setupWithNavController(this)
                addOnDestinationChangedListener { _, destination, _ ->
                    // Logger.e("addOnDestinationChangedListener=" + destination.displayName + "\n backStack=" + backStack.size)
                    backStack.size.let { _ ->
                        if (showNavFragments.contains(destination.id)) {
                            showNavBottom(isShow = true)
                            setupNavigationTheme()
                        } else {
                            showNavBottom(isShow = false)
                        }
                    }
                }
            }
        } else {
            showNavBottom(isShow = false)
        }
    }

    private fun setupNavigationTheme() {
        if (isNetAlo()) {
            themeHelper.setThemeColorForViews(binding.navView)
        }
    }

    private fun showNavBottom(isShow: Boolean) {
        // Logger.e("showNavBottom=$isShow" + navController.currentDestination?.navigatorName)
        with(binding.navView) {
            if (isShow) {
                if (isGone) {
                    isVisible = true
                }
            } else if (navController.currentDestination?.navigatorName != "dialog") {
                isGone = true
            }
        }
    }

    private fun processReceiveCall() {
        val call = (intent.getParcelableExtra(CALL) as? Call) ?: mainSdkViewModel.mainViewModelCall ?: return
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.O_MR1) {
            setShowWhenLocked(true)
            setTurnScreenOn(true)
            val keyguardManager = getSystemService(KEYGUARD_SERVICE) as KeyguardManager?
            keyguardManager?.requestDismissKeyguard(this, null)
        } else {
            @Suppress("DEPRECATION")
            window.addFlags(
                WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON or
                    WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED or
                    WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON or
                    WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD
            )
        }
        val isNotification = intent.getBooleanExtra(NOTIFICATION, false)
        navigationDispatcher.emit { navController ->
            call.let { _call ->
                mainSdkViewModel.mainViewModelCall = null
                navController.navigate(
                    R.id.inComeFragment,
                    bundleOf(
                        NavigationDef.ARGUMENT_INCOME to _call,
                        NavigationDef.ARGUMENT_NOTIFICATION to isNotification
                    )
                )
            }
        }
    }

    private fun receiveShare() {
        intent?.apply {
            when (action) {
                Intent.ACTION_SEND -> {
                    if (type?.startsWith("text/") == true) {
                        getStringExtra(Intent.EXTRA_TEXT)?.let {
                            navigationDispatcher.emit { nav ->
                                nav.navigateIfSafe(R.id.ForwardMessageDialog, "forwardMediaPath" to it)
                            }
                            Logger.e("handleSendText=text=$it")
                        }
                    } else if (type?.startsWith("image/") == true || type?.startsWith("audio/") == true || type?.startsWith("video/") == true) {
                        getParcelableExtra<Parcelable>(Intent.EXTRA_STREAM)?.let {
                            UriUtils.uri2File(it as? Uri)?.path?.apply {
                                Logger.e("image=url=$this")
                                navigationDispatcher.emit { nav ->
                                    nav.navigate(R.id.ForwardMessageDialog, bundleOf("forwardMediaPath" to this))
                                }
                            }
                        }
                    }
                }
                Intent.ACTION_SEND_MULTIPLE -> {
                    if (type?.startsWith("image/") == true || type?.startsWith("audio/") == true || type?.startsWith("video/") == true) {
                        getParcelableArrayListExtra<Parcelable>(Intent.EXTRA_STREAM)?.let {
                            UriUtils.uri2File(it[0] as? Uri)?.path?.apply {
                                Logger.e("image=url=$this")
                                navigationDispatcher.emit { nav ->
                                    nav.navigate(R.id.ForwardMessageDialog, bundleOf("forwardMediaPath" to this))
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
