/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */
package com.netacom.full.ui.sdk

import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.net.Uri
import androidx.annotation.Keep
import androidx.core.os.bundleOf
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import androidx.navigation.NavController
import com.netacom.base.chat.android_utils.LanguageUtils
import com.netacom.base.chat.android_utils.TimeUtils
import com.netacom.base.chat.base.BaseViewModel
import com.netacom.base.chat.livedata.EventLiveData
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.R
import com.netacom.full.dispatchers.Dispatcher
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.full.utils.DialogUtil
import com.netacom.lite.BuildConfig
import com.netacom.lite.config.EndPoint
import com.netacom.lite.hilt.executor.PostExecutionThread
import com.netacom.lite.local.prefs.PreferencesHelperImpl
import com.netacom.lite.network.model.response.CampaignResponse
import com.netacom.lite.network.model.response.Popup
import com.netacom.lite.repository.AuthRepository
import com.netacom.lite.sdk.AppID
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import javax.inject.Inject
import kotlin.time.ExperimentalTime
import kotlin.time.seconds

/**
Created by vantoan on 15,Sep,2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/

@Keep
@HiltViewModel
class NetAloViewModel @Inject constructor(
    private val authRepository: AuthRepository,
    private val postExecutionThread: PostExecutionThread,
    private val preferencesHelperImpl: PreferencesHelperImpl,
    private val navigationDispatcher: Dispatcher<(NavController) -> Unit>
) : BaseViewModel() {

    private var _showReferral = MutableLiveData<EventLiveData<Popup>>()
    val showReferral: LiveData<EventLiveData<Popup>> = _showReferral

    init {
        EndPoint.URL_CDN = preferencesHelperImpl.setting?.cdnEndpoint ?: BuildConfig.CDN_URL
        if (preferencesHelperImpl.sdkConfig?.appId == AppID.GALO) {
            getCampaign()
        } else {
            checkReferralCode()
        }
    }

    // val netAloSessionExpireEvent = socketRepository.flowSessionExpire
    private fun checkReferralCode() {
        viewModelScope.launch(postExecutionThread.io) {
            preferencesHelperImpl.referral = CampaignResponse(referral = false)
            authRepository.checkReferralCode().collect {
                if (it is ApiResponseSuccess) {
                    preferencesHelperImpl.referral = CampaignResponse(referral = true)
                }
            }
        }
    }

    @OptIn(ExperimentalTime::class)
    private fun getCampaign() {
        viewModelScope.launch(postExecutionThread.io) {
            preferencesHelperImpl.referral = CampaignResponse(referral = false)
            authRepository.getCampaign(LanguageUtils.getLanguage()).collect {
                if (it is ApiResponseSuccess) {
                    val listBanner = mutableListOf<Popup>()
                    var referral = false
                    it.data?.popups?.mapIndexed { index, popup ->
                        if (popup.banner_url?.contains("BANNER") == true) {
                            popup.click_url = "${popup.click_url}?token=${preferencesHelperImpl.token}"
                        }
                        when {
                            TimeUtils.getNowMills() < popup.start_at.seconds.toLongMilliseconds() -> {
                                referral = true
                                if (index == 0) {
                                    _showReferral.post(EventLiveData(popup))
                                    listBanner.add(popup.copy(banner_url = null))
                                } else {
                                    listBanner.add(popup)
                                }
                            }
                            TimeUtils.getNowMills() < popup.end_at.seconds.toLongMilliseconds() -> {
                                referral = true
                                if (index == 0) {
                                    _showReferral.post(EventLiveData(popup))
                                    listBanner.add(popup.copy(banner_url = null))
                                } else {
                                    listBanner.add(popup)
                                }
                            }
                            else -> {
                                listBanner.add(popup)
                            }
                        }
                    }
                    preferencesHelperImpl.referral = CampaignResponse(popups = listBanner, referral = referral)
                }
            }
        }
    }

    fun showNextReferral() {
        viewModelScope.launch(postExecutionThread.io) {
            preferencesHelperImpl.referral?.apply {
                val listBanner = mutableListOf<Popup>()
                popups?.map { popup ->
                    if (popup.banner_url.isNotNull) {
                        _showReferral.post(EventLiveData(popup))
                        listBanner.add(popup.copy(banner_url = null))
                    } else {
                        listBanner.add(popup)
                    }
                }
                // Logger.e("showNextReferral==" + listBanner.map { it })
                preferencesHelperImpl.referral = CampaignResponse(popups = listBanner, referral = referral)
            }
        }
    }

    fun gotoReferral(url: String) {
        navigationDispatcher.emit {
            it.navigate(R.id.campaignFragment, bundleOf("url" to url))
        }
    }

    fun checkAppUpdate(context: Context, themeHelperImpl: ThemeHelperImpl) {
        launchOnViewModelScope(postExecutionThread.io) {
            authRepository.checkAppUpdate().collect {
                if (it is ApiResponseSuccess) {
                    withContext(postExecutionThread.main) {
                        if (it.data?.forceUpdate == true) {
                            DialogUtil.show(
                                context = context,
                                titleRes = R.string.text_title_force_update,
                                messageRes = R.string.text_message_force_update,
                                okLabel = R.string.text_update,
                                okFunc = {
                                    openAppPlayStore(context)
                                },
                                cancelable = false,
                                ignoreOkDismiss = true,
                                themeHelperImpl = themeHelperImpl
                            )
                        } else if (it.data?.lastVersion == false && !preferencesHelperImpl.userIgnoreOptionUpdate) {
                            DialogUtil.show(
                                context = context,
                                titleRes = R.string.text_title_force_update,
                                messageRes = R.string.text_message_force_update,
                                okLabel = R.string.text_update,
                                cancelLabel = R.string.text_later,
                                okFunc = {
                                    openAppPlayStore(context)
                                },
                                cancelFunc = {
                                    preferencesHelperImpl.userIgnoreOptionUpdate = true
                                },
                                themeHelperImpl = themeHelperImpl
                            )
                        }
                    }
                }
            }
        }
    }

    private fun openAppPlayStore(context: Context) {
        try {
            context.startActivity(Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=${context.packageName}")))
        } catch (e: ActivityNotFoundException) {
            context.startActivity(Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/details?id=${context.packageName}")))
        }
    }
}
