package com.netacom.full.utils

import android.content.Context
import android.media.AudioManager
import android.media.MediaPlayer
import android.net.Uri
import android.os.CountDownTimer
import android.view.View
import android.widget.ProgressBar
import android.widget.TextView
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import com.netacom.base.chat.android_utils.ThreadUtils.runOnUiThread
import com.netacom.full.R
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.Constants
import com.netacom.lite.util.getUrlImage

/**
 * Create on 8/7/20
 * @author duonghd
 */

object AudioSupport {

    enum class AudioStatus {
        STATUS_NONE,
        STATUS_LOADING,
        STATUS_PLAYING
    }

    var messageId: Long? = null
    var status = AudioStatus.STATUS_NONE
    var audioView: View? = null
    private var timer: CountDownTimer? = null

    private val mediaPlayer = MediaPlayer().apply {
        setAudioStreamType(AudioManager.STREAM_MUSIC)
    }

    fun playAudio(context: Context, messageId: Long?, audioUrl: String, userToken: String? = null) {
        AudioSupport.messageId = messageId

        val uri = if (audioUrl.split("/").size > 2) {
            Uri.parse(audioUrl) ?: return
        } else {
            Uri.parse(audioUrl.getUrlImage()) ?: return
        }

        val headers: HashMap<String, String> = HashMap(3)
        userToken?.let { headers.put(Constants.AuthConstants.TC_TOKEN, it) }
        if (mediaPlayer.isPlaying) {
            stopAudio()
        }
        mediaPlayer.setDataSource(context, uri, headers)

        mediaPlayer.setOnCompletionListener {
            stopAudio()
        }

        mediaPlayer.setOnErrorListener { mp, what, extra ->
            /**
             * TODO something
             * @param mp the MediaPlayer the error pertains to
             * @param what the type of error that has occurred:
             * @param extra extra code, specific to the error. Typically implementation dependent.
             */
            false
        }

        mediaPlayer.setOnPreparedListener { player ->
            audioView?.findViewById<AppCompatImageView>(R.id.imgPlay)?.let {
                it.setImageResource(R.drawable.ic_record_pause)
                it.alpha = 1f
            }
            player.start()
            startTimer()
            status = AudioStatus.STATUS_PLAYING
        }

        audioView?.findViewById<AppCompatImageView>(R.id.imgPlay)?.let {
            it.setImageResource(R.drawable.ic_record_pause)
            it.alpha = 0.5f
        }

        status = AudioStatus.STATUS_LOADING
        mediaPlayer.prepareAsync()
    }

    fun setViewPlayer(view: View) {
        audioView = view
    }

    fun stopAudio() {
        timer?.cancel()
        timer = null
        audioView?.findViewById<AppCompatImageView>(R.id.imgPlay)?.let {
            it.setImageResource(R.drawable.ic_record_play)
            it.alpha = 1f
        }
        audioView?.findViewById<AppCompatTextView>(R.id.txtProgress)?.let {
            (it.tag as? String).let { tag ->
                if (tag != null) {
                    it.text = tag
                } else {
                    it.text = AppUtils.setAudioTextViewTime(time = (mediaPlayer.duration / 1000).toLong())
                }
            }
        }
        audioView?.findViewById<ProgressBar>(R.id.seekbar)?.let {
            it.progress = 0
        }
        mediaPlayer.reset()
        audioView = null
        messageId = null
        status = AudioStatus.STATUS_NONE
    }

    private fun startTimer() {
        val audioDurations = mediaPlayer.duration.toLong()
        audioView?.context?.let {
            timer = object : CountDownTimer(audioDurations + 1000, 20) {
                override fun onTick(ticks: Long) {
                    val remainTimes = audioDurations - mediaPlayer.currentPosition
                    val time = kotlin.run {
                        val minutes = (remainTimes / 1000) / 60
                        val seconds = (remainTimes / 1000) % 60
                        String.format("%02d:%02d", minutes, seconds)
                    }
                    val progress = kotlin.run {
                        ((audioDurations - remainTimes) * 1000F / audioDurations).toInt()
                    }

                    runOnUiThread {
                        audioView?.findViewById<TextView>(R.id.txtProgress)?.text = time
                        audioView?.findViewById<ProgressBar>(R.id.seekbar)?.progress = progress
                    }
                }

                override fun onFinish() {
                }
            }
            timer?.start()
        }
    }
}
