package com.netacom.full.widget

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.drawable.LayerDrawable
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.FrameLayout
import androidx.annotation.AttrRes
import androidx.core.graphics.drawable.DrawableCompat
import androidx.core.view.isVisible
import androidx.databinding.DataBindingUtil
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.databinding.LayoutScreenStateBinding
import com.netacom.base.chat.model.ScreenStateObj
import com.netacom.base.chat.type.ScreenState
import com.netacom.base.chat.util.getThemeColor
import com.netacom.full.R
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import dagger.hilt.android.AndroidEntryPoint
import dagger.hilt.android.WithFragmentBindings
import javax.inject.Inject

@WithFragmentBindings
@AndroidEntryPoint
class EmptyView : FrameLayout {
    private lateinit var mBinding: LayoutScreenStateBinding

    @set:Inject
    internal lateinit var themeHelperImpl: ThemeHelperImpl

    var newActionListener: NewActionListener? = null
    constructor(context: Context) : super(context) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        init(context)
    }

    private fun init(context: Context) {
        try {
            val inflater = context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater
            mBinding = DataBindingUtil.inflate(inflater, R.layout.layout_screen_state, this, true)
            with(mBinding.imageView) {
                val backgroundResource: Int = when (this@EmptyView.tag.toString()) {
                    "group" -> R.drawable.ic_chat_empty_background
                    "call" -> R.drawable.ic_call_empty_background
                    "contact" -> R.drawable.ic_contact_empty_background
                    else -> return
                }
                this.setBackgroundResource(backgroundResource)

                if (mBinding.imageView.background is LayerDrawable) {
                    val layerList = this.background as LayerDrawable

                    val unwrappedBackgroundDrawable =
                        layerList.findDrawableByLayerId(R.id.ic_empty_background).mutate()

                    val wrappedBackgroundDrawable = DrawableCompat.wrap(unwrappedBackgroundDrawable)

                    DrawableCompat.setTint(
                        wrappedBackgroundDrawable,
                        getThemeColor(themeHelperImpl.subColor)
                    )

                    val unwrappedIconDrawable = layerList.findDrawableByLayerId(R.id.ic_contain).mutate()

                    val wrappedIconDrawable = DrawableCompat.wrap(unwrappedIconDrawable)

                    DrawableCompat.setTint(
                        wrappedIconDrawable,
                        getThemeColor(themeHelperImpl.mainColor)
                    )
                }
            }
            when (this@EmptyView.tag.toString()) {
                "group" -> {
                    mBinding.tvEmpty.text = StringUtils.getString(R.string.text_message_empty_conversation)
                    mBinding.tvAction.isVisible = true
                    mBinding.tvAction.text = StringUtils.getString(R.string.text_message_start_new_message)
                    mBinding.tvAction.setTextColor(getThemeColor(themeHelperImpl.mainColor))
                    mBinding.tvAction.backgroundTintList = ColorStateList.valueOf(
                        getThemeColor(themeHelperImpl.mainColor)
                    )
                }
                "call" -> {
                    mBinding.tvEmpty.text = StringUtils.getString(R.string.text_message_empty_call)
                    mBinding.tvAction.isVisible = true
                    mBinding.tvAction.text = StringUtils.getString(R.string.text_call_new)
                    mBinding.tvAction.setTextColor(getThemeColor(themeHelperImpl.mainColor))
                    mBinding.tvAction.backgroundTintList = ColorStateList.valueOf(
                        getThemeColor(themeHelperImpl.mainColor)
                    )
                }
                "contact" -> {
                    mBinding.tvAction.text = StringUtils.getString(R.string.contact_make_friend_phone_book)
                    mBinding.tvAction.isVisible = true
                    mBinding.tvAction.text = StringUtils.getString(R.string.contact_make_friend_phone_book)
                    mBinding.tvAction.setTextColor(getThemeColor(themeHelperImpl.mainColor))
                    mBinding.tvAction.backgroundTintList = ColorStateList.valueOf(getThemeColor(themeHelperImpl.mainColor))
                }
                else -> return
            }
            mBinding.tvAction.clickDebounce {
                newActionListener?.onStartNewAction()
            }
        } catch (e: Exception) {
        }
    }

    fun setScreenState(screenState: ScreenStateObj) {
        mBinding.stateObj = screenState
    }

    fun setEmptyState() {
        mBinding.stateObj = ScreenStateObj(ScreenState.EMPTY)
    }

    interface NewActionListener {
        fun onStartNewAction()
    }
}
