package com.netacom.full.widget.avatar

import android.animation.ValueAnimator
import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapShader
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.LinearGradient
import android.graphics.Matrix
import android.graphics.Outline
import android.graphics.Paint
import android.graphics.Paint.ANTI_ALIAS_FLAG
import android.graphics.Rect
import android.graphics.RectF
import android.graphics.Shader
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.Drawable
import android.os.Build
import android.util.AttributeSet
import android.util.Log
import android.util.TypedValue
import android.view.MotionEvent
import android.view.View
import android.view.ViewOutlineProvider
import androidx.annotation.DrawableRes
import androidx.annotation.RequiresApi
import androidx.appcompat.widget.AppCompatImageView
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.util.isNotNull
import com.netacom.base.chat.util.unAccentText
import com.netacom.full.R
import java.util.Locale
import kotlin.math.floor
import kotlin.math.max
import kotlin.math.min
import kotlin.math.pow

class AvatarView : AppCompatImageView {
    companion object {
        const val TAG = "AvatarView"
    }

    private val avatarDrawableRect = RectF()
    private val middleRect = RectF()
    private val borderRect = RectF()
    private val arcBorderRect = RectF()
    private val initialsRect = Rect()

    private val shaderMatrix = Matrix()
    private val bitmapPaint = Paint()
    private val middlePaint = Paint()
    private val borderPaint = Paint()
    private val circleBackgroundPaint = Paint()
    private val initialsPaint = Paint(ANTI_ALIAS_FLAG).apply {
        color = Color.WHITE
        textSize = Utils.getApp().resources.getDimension(R.dimen.sp_14)
        setup()
    }
    private val badgePaint = Paint()
    private val badgeStrokePaint = Paint()

    private var middleThickness = 0f
    private val avatarInset
        get() = (if (isBorder) distanceToBorder else 0) + max(borderThickness.toFloat(), highlightedBorderThickness.toFloat())

    private var avatarDrawable: Bitmap? = null
    private var bitmapShader: BitmapShader? = null
    private var bitmapWidth = 0
    private var bitmapHeight = 0

    private var drawableRadius = 0f
    private var middleRadius = 0f
    private var borderRadius = 0f

    /**
     * The length (in degrees) available for the arches when animating.
     */
    var totalArchesDegreeArea = Defaults.ARCHES_DEGREES_AREA
        set(value) {
            field = value
            logWarningOnArcLengthIfNeeded()
            setup()
        }

    /**
     * The number of arches displayed across the border when animating.
     * This can be set to zero, if no secondary arches are wanted.
     */
    var numberOfArches = Defaults.NUMBER_OF_ARCHES
        set(value) {
            field = if (value <= 0) 1 else value
            logWarningOnArcLengthIfNeeded()
            setup()
        }

    /**
     * The length (in degrees) of each arch when animating.
     * Keep in mind that the arches may overlap if this value is too high
     * and [totalArchesDegreeArea] is too low.
     */
    var individualArcDegreeLength = Defaults.INDIVIDUAL_ARCH_DEGREES_LENGTH
        set(value) {
            field = value
            logWarningOnArcLengthIfNeeded()
            setup()
        }

    private fun logWarningOnArcLengthIfNeeded() {
        if (individualArcDegreeLength * numberOfArches > totalArchesDegreeArea) {
            Log.w(TAG, "The arches are too big for them to be visible. (i.e. individualArcLength * numberOfArches > totalArchesDegreeArea)")
        }
    }

    /**
     * The color of the gap between the border and the avatar.
     * Default: [Color.TRANSPARENT]
     */
    var middleColor = Defaults.MIDDLE_COLOR
        set(value) {
            field = value
            setup()
        }

    /**
     * The border color.
     * Remember: The border is colored using a gradient.
     * If you want a solid color, make sure that the [borderColorEnd] is set to the same value.
     */
    var borderColor = Defaults.BORDER_COLOR
        set(value) {
            borderColorEnd = value
            field = value
        }

    /**
     * The second color of the border gradient.
     * Remember: The border is colored using a gradient.
     * If you want a solid color, make sure that the [borderColor] is set to the same value.
     */
    var borderColorEnd = Defaults.BORDER_COLOR
        set(value) {
            field = value
            setup()
        }

    /**
     * The border color when highlighted.
     * Remember: The border is colored using a gradient.
     * If you want a solid color, make sure that the [highlightBorderColorEnd] is set to the same value.
     */
    var highlightBorderColor = Defaults.BORDER_COLOR_HIGHLIGHT
        set(value) {
            field = value
            setup()
        }

    /**
     * The second color of the border gradient when highlighted.
     * Remember: The border is colored using a gradient.
     * If you want a solid color, make sure that the [highlightBorderColor] is set to the same value.
     */
    var highlightBorderColorEnd = Defaults.BORDER_COLOR_HIGHLIGHT
        set(value) {
            field = value
            setup()
        }

    /**
     * The distance (in pixels) between the avatar and the border.
     * Keep in mind that as the [borderThickness] and [highlightedBorderThickness] may be different,
     * the highest value between them will be considered in order to keep a steady avatar when
     * switching between highlight modes.
     */
    var distanceToBorder = Defaults.DISTANCE_TO_BORDER
        set(value) {
            field = if (value <= 0) 0 else value
            setup()
        }

    /**
     * The border thickness (in pixels) when not highlighted
     */
    var borderThickness = Defaults.BORDER_THICKNESS
        set(value) {
            field = if (value <= 0) 0 else value
            setup()
        }

    /**
     * The border thickness (in pixels) when highlighted
     */
    var highlightedBorderThickness = Defaults.HIGHLIGHTED_BORDER_THICKNESS
        set(value) {
            field = if (value <= 0) 0 else value
            setup()
        }

    /**
     * The background color of the avatar.
     * Only visible if the image has any transparency.
     */
    var avatarBackgroundColor = Defaults.CIRCLE_BACKGROUND_COLOR
        set(value) {
            field = value
        }

    /**
     * The default onClick and onLongClick manifestation is a simple scale/zoom bounce.
     * It must have a onClickListener or onLongClickListener in order to happen.
     * If you want to turn it off by any reason, just set this to false.
     */
    var shouldBounceOnClick = true

    /**
     * Text for which the initial(s) has to be displayed
     */
    var text: String? = null
        set(value) {
            field = value
            findInitials()
        }

    /**
     * The radius (in dp) of the badge
     */
    var badgeRadius = TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, Defaults.BADGE_RADIUS, resources.displayMetrics)
        set(value) {
            field = value
            setup()
        }

    /**
     * The color of the badge.
     */
    var badgeColor = Defaults.BADGE_COLOR
        set(value) {
            field = value
            setup()
        }

    /**
     * The color of the stroke of the badge.
     */
    var badgeStrokeColor = Defaults.BADGE_STROKE_COLOR
        set(value) {
            field = value
            setup()
        }

    /**
     * The stroke width (in pixels) of the badge.
     */
    var badgeStrokeWidth = Defaults.BADGE_WIDTH
        set(value) {
            field = if (value <= 0) 0 else value
            setup()
        }

    /**
     * Flag to toggle visibility of the badge.
     */
    var showBadge = Defaults.SHOW_BADGE
        set(value) {
            field = value
        }

    /**
     * Position of the badge with respect to [borderRect]
     *
     * Should be one of:
     * @see [BadgePosition.BOTTOM_RIGHT]
     * @see [BadgePosition.BOTTOM_LEFT]
     * @see [BadgePosition.TOP_RIGHT]
     * @see [BadgePosition.TOP_LEFT]
     */

    private var initials: String? = null

    /**
     * Set if this view should be highlighted or not.
     * If highlighted, the values of [highlightedBorderThickness], [highlightBorderColor] and [highlightBorderColorEnd] will apply
     * Otherwise, [borderThickness], [borderColor] and [borderColorEnd] will come into play.
     */

    var isBorder = false
        set(value) {
            initialsPaint.textSize = (if (value) Utils.getApp().resources.getDimension(R.dimen.sp_12) else Utils.getApp().resources.getDimension(R.dimen.sp_14))
            field = value
        }

    var textSize = Utils.getApp().resources.getDimension(R.dimen.sp_14)
        set(value) {
            initialsPaint.textSize = value
            field = value
        }

    var isHighlighted = Defaults.IS_HIGHLIGHTED
        set(value) {
            field = value
            setup()
        }

    private val scaleAnimator = ValueAnimator.ofFloat(1f, 0.9f, 1f).apply {
        addUpdateListener {
            this@AvatarView.scaleX = it.animatedValue as Float
            this@AvatarView.scaleY = it.animatedValue as Float
        }
    }

    constructor(context: Context) : super(context) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet, defStyle: Int) : super(context, attrs, defStyle) {
        init()
    }

    override fun setImageBitmap(bm: Bitmap) {
        super.setImageBitmap(bm)
        initializeBitmap()
    }

    override fun setImageDrawable(drawable: Drawable?) {
        super.setImageDrawable(drawable)
        initializeBitmap()
    }

    override fun setImageResource(@DrawableRes resId: Int) {
        super.setImageResource(resId)
        initializeBitmap()
    }

    private fun init() {
        scaleType = Defaults.SCALE_TYPE
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            outlineProvider = OutlineProvider()
        }
        setup()
    }

    fun setup() {
        if (width == 0 && height == 0) {
            return
        }

        val avatarDrawable = this.avatarDrawable
        if (avatarDrawable == null) {
            setImageResource(android.R.color.transparent)
            return
        }

        bitmapHeight = avatarDrawable.height
        bitmapWidth = avatarDrawable.width

        bitmapShader = BitmapShader(avatarDrawable, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP)
        bitmapPaint.isAntiAlias = true
        bitmapPaint.shader = bitmapShader

        val currentBorderThickness = (
            if (isHighlighted) {
                highlightedBorderThickness
            } else borderThickness
            ).toFloat()
        borderRect.set(calculateBounds())
        borderRadius = min((borderRect.height() - currentBorderThickness) / 2.0f, (borderRect.width() - currentBorderThickness) / 2.0f)

        val currentBorderGradient = LinearGradient(
            0f,
            0f,
            borderRect.width(),
            borderRect.height(),
            if (isHighlighted) highlightBorderColor else borderColor,
            if (isHighlighted) highlightBorderColorEnd else borderColorEnd,
            Shader.TileMode.CLAMP
        )

        borderPaint.apply {
            if (isBorder) {
                shader = currentBorderGradient
            } else {
                color = Color.TRANSPARENT
            }
            strokeWidth = if (isBorder) currentBorderThickness else 0f
            isAntiAlias = true
            strokeCap = Paint.Cap.ROUND
            style = Paint.Style.STROKE
        }
        avatarDrawableRect.set(borderRect)
        avatarDrawableRect.inset(avatarInset, avatarInset)
        middleThickness = (borderRect.width() - currentBorderThickness * 2 - avatarDrawableRect.width()) / 2

        middleRect.set(borderRect)
        middleRect.inset(currentBorderThickness + middleThickness / 2, currentBorderThickness + middleThickness / 2)

        middleRadius = floor(middleRect.height() / 2.0).coerceAtMost(floor(middleRect.width() / 2.0)).toFloat()
        drawableRadius = (avatarDrawableRect.height() / 2.0f).coerceAtMost(avatarDrawableRect.width() / 2.0f)

        middlePaint.apply {
            style = Paint.Style.STROKE
            isAntiAlias = true
            color = middleColor
            strokeWidth = middleThickness
        }

        circleBackgroundPaint.apply {
            style = Paint.Style.FILL
            isAntiAlias = true
            color = avatarBackgroundColor
        }
        arcBorderRect.apply {
            set(borderRect)
            inset(currentBorderThickness / 2f, currentBorderThickness / 2f)
        }

        badgePaint.apply {
            style = Paint.Style.FILL
            isAntiAlias = true
            color = badgeColor
        }

        badgeStrokePaint.apply {
            style = Paint.Style.FILL
            isAntiAlias = true
            color = badgeStrokeColor
        }

        updateShaderMatrix()
        invalidate()
    }

    private fun updateShaderMatrix() {
        val scale: Float
        var dx = 0f
        var dy = 0f

        shaderMatrix.set(null)

        if (bitmapWidth * avatarDrawableRect.height() > avatarDrawableRect.width() * bitmapHeight) {
            scale = avatarDrawableRect.height() / bitmapHeight.toFloat()
            dx = (avatarDrawableRect.width() - bitmapWidth * scale) / 2f
        } else {
            scale = avatarDrawableRect.width() / bitmapWidth.toFloat()
            dy = (avatarDrawableRect.height() - bitmapHeight * scale) / 2f
        }

        shaderMatrix.setScale(scale, scale)
        shaderMatrix.postTranslate((dx + 0.5f).toInt() + avatarDrawableRect.left, (dy + 0.5f).toInt() + avatarDrawableRect.top)

        bitmapShader!!.setLocalMatrix(shaderMatrix)
    }

    private fun getBitmapFromDrawable(drawable: Drawable?): Bitmap? {
        if (drawable == null) {
            return null
        }

        if (drawable is BitmapDrawable) {
            return drawable.bitmap
        }

        return try {
            val bitmap = if (drawable is ColorDrawable) {
                Bitmap.createBitmap(Defaults.COLORDRAWABLE_DIMENSION, Defaults.COLORDRAWABLE_DIMENSION, Defaults.BITMAP_CONFIG)
            } else {
                if (drawable.intrinsicWidth > 0 && drawable.intrinsicHeight > 0) {
                    Bitmap.createBitmap(drawable.intrinsicWidth, drawable.intrinsicHeight, Defaults.BITMAP_CONFIG)
                } else {
                    return null
                }
            }
            val canvas = Canvas(bitmap)
            drawable.setBounds(0, 0, canvas.width, canvas.height)
            drawable.draw(canvas)
            bitmap
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    private fun initializeBitmap() {
        avatarDrawable = getBitmapFromDrawable(drawable)
        setup()
    }

    private fun calculateBounds(): RectF {
        val availableWidth = width - paddingLeft - paddingRight
        val availableHeight = height - paddingTop - paddingBottom

        val sideLength = min(availableWidth, availableHeight)

        val left = paddingLeft + (availableWidth - sideLength) / 2f
        val top = paddingTop + (availableHeight - sideLength) / 2f
        return RectF(left, top, left + sideLength, top + sideLength)
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        return inTouchableArea(event.x, event.y) && super.onTouchEvent(event)
    }

    private fun animateClick() {
        if (shouldBounceOnClick) scaleAnimator.start()
    }

    override fun performClick(): Boolean {
        animateClick()
        return super.performClick()
    }

    override fun performLongClick(): Boolean {
        animateClick()
        return super.performLongClick()
    }

    private fun inTouchableArea(x: Float, y: Float): Boolean {
        return (x - borderRect.centerX().toDouble()).pow(2.0) + (y - borderRect.centerY().toDouble()).pow(2.0) <= borderRadius.toDouble().pow(2.0)
    }

    override fun onDraw(canvas: Canvas) {
        if (avatarDrawable == null &&
            initials == null
        ) {
            return
        }
        if (avatarBackgroundColor != Color.TRANSPARENT) {
            canvas.drawCircle(avatarDrawableRect.centerX(), avatarDrawableRect.centerY(), drawableRadius, circleBackgroundPaint)
        }

        val avatarDrawable = this.avatarDrawable
        if (null != avatarDrawable && hasAvatar()) {
            canvas.drawCircle(avatarDrawableRect.centerX(), avatarDrawableRect.centerY(), drawableRadius, bitmapPaint)
        } else if (null != initials) {
            canvas.drawText(
                initials!!,
                width.div(2f) - initialsRect.width().div(2f),
                height.div(2f) + initialsRect.height().div(2f),
                initialsPaint
            )
        }
        if (middleThickness > 0) {
            canvas.drawCircle(middleRect.centerX(), middleRect.centerY(), middleRadius, middlePaint)
        }
        drawBorder(canvas)
        if (showBadge && badgeColor != Color.TRANSPARENT) {
            val badgeCx = borderRect.right - badgeRadius * 2.1f
            val badgeCy = borderRect.bottom - badgeRadius * 2.1f
            if (badgeStrokeWidth > 0) {
                canvas.drawCircle(badgeCx, badgeCy, badgeRadius, badgeStrokePaint)
            }
            canvas.drawCircle(badgeCx, badgeCy, badgeRadius - badgeStrokeWidth, badgePaint)
        }
    }

    private fun hasAvatar(): Boolean {
        val drawable = this.drawable
        val hasTransparentDrawable = drawable is ColorDrawable && drawable.alpha == 0
        return !hasTransparentDrawable
    }

    private fun drawBorder(canvas: Canvas) {
        canvas.drawCircle(borderRect.centerX(), borderRect.centerY(), borderRadius, borderPaint)
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        setup()
    }

    override fun setPadding(left: Int, top: Int, right: Int, bottom: Int) {
        super.setPadding(left, top, right, bottom)
        setup()
    }

    override fun setPaddingRelative(start: Int, top: Int, end: Int, bottom: Int) {
        super.setPaddingRelative(start, top, end, bottom)
        setup()
    }

    private fun findInitials() {
        try {
            text?.trim()?.unAccentText()?.let {
                if (it.isNotNull) {
                    var initialsCount = 1
                    val words = it.split("[,]".toRegex(), limit = 2)
                    if (words.size > 1) {
                        initialsCount = 2
                        initials = it.first().toString()
                        initials = "$initials${words[1].trim().first()}".toUpperCase(Locale.ROOT)
                    } else {
                        initials = if (it.length > 1) {
                            initialsCount = 2
                            it.substring(0, 2).toUpperCase(Locale.ROOT)
                        } else {
                            it.first().toUpperCase().toString()
                        }
                    }
                    initialsPaint.getTextBounds(initials, 0, initialsCount, initialsRect)
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            FirebaseCrashlytics.getInstance().recordException(e)
        }
    }

    /**
     * This section makes the elevation settings on Lollipop+ possible,
     * drawing a circlar shadow around the border
     */
    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    private inner class OutlineProvider : ViewOutlineProvider() {

        override fun getOutline(view: View, outline: Outline) = Rect().let {
            borderRect.roundOut(it)
            outline.setRoundRect(it, it.width() / 2.0f)
        }
    }
}
