/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.widget.chat_footer

import android.annotation.SuppressLint
import android.content.Context
import android.content.res.ColorStateList
import android.graphics.drawable.AnimationDrawable
import android.text.Editable
import android.text.TextWatcher
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.FrameLayout
import androidx.annotation.AttrRes
import androidx.appcompat.widget.AppCompatImageView
import androidx.core.content.ContextCompat
import androidx.core.view.isGone
import androidx.core.view.isVisible
import androidx.core.widget.ImageViewCompat
import androidx.databinding.DataBindingUtil
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.FileUtils
import com.netacom.base.chat.android_utils.KeyboardUtils
import com.netacom.base.chat.android_utils.SizeUtils
import com.netacom.base.chat.android_utils.StringUtils
import com.netacom.base.chat.android_utils.TimeUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.binding.clickDebounce
import com.netacom.base.chat.imageloader.loadImage
import com.netacom.base.chat.library.mention.common.MentionSpan
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.isNotNull
import com.netacom.full.R
import com.netacom.full.databinding.ViewChatFooterBinding
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_AUDIO
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_FILE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_FORWARD
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_IMAGE
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_REPLY
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_STICKER
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_TEXT
import com.netacom.lite.define.MessageType.MESSAGE_TYPE_VIDEO
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.message.NeMessage
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.local.prefs.PreferencesHelperImpl
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.Constants
import com.netacom.lite.util.getUrlImage
import dagger.hilt.android.AndroidEntryPoint
import dagger.hilt.android.WithFragmentBindings
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import java.util.Timer
import java.util.TimerTask
import javax.inject.Inject

@WithFragmentBindings
@AndroidEntryPoint
class ChatFooterView : FrameLayout {

    private lateinit var mBinding: ViewChatFooterBinding

    @set:Inject
    internal lateinit var themeHelperImpl: ThemeHelperImpl

    @set:Inject
    internal lateinit var preferencesHelperImpl: PreferencesHelperImpl

    private var mCallback: ViewChatFooterCallBack? = null
    private var isTyping = false
    private var isReplying = false
    private var isBlockShow = false
    private var isForwarding = false
    private var isWasBlocked = false
    private var isClickVoice = false
    private var isChangeMention = false
    private var replyOrForwardMessage: NeMessage? = null

    // Record
    private var countTimerToggle: Timer? = null
    private var timeToggle = 0L
    private var typingAnimation: AnimationDrawable? = null

    constructor(context: Context) : super(context) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        init(context)
    }

    constructor(context: Context, attrs: AttributeSet?, @AttrRes defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        init(context)
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun init(context: Context) {
        val inflater = context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater
        mBinding = DataBindingUtil.inflate(inflater, R.layout.view_chat_footer, this, true)

        mBinding.ivTyping.setBackgroundResource(R.drawable.ic_typing_animation)
        typingAnimation = mBinding.ivTyping.background as AnimationDrawable

        with(mBinding) {
            buttonSend.clickDebounce {
                when (buttonSend.tag) {
                    "message" -> {
                        val content = mBinding.revInput.text.toString()
                        if (content.isNotEmpty()) {
                            content.replace("@1 ", "@All ")
                            replyOrForwardMessage?.let { neMessage ->
                                if (isReplying) {
                                    mCallback?.onSendReplyMessage(content, neMessage)
                                    isReplying = false
                                } else if (isForwarding) {
                                    mCallback?.onSendForwardMessage(content, neMessage)
                                    isForwarding = false
                                }
                                mBinding.isReplyOrForward = false
                                mBinding.revInput.setText("")
                                replyOrForwardMessage = null
                            } ?: let {
                                mCallback?.onSendMessage(content)
                                isReplying = false
                                mBinding.isReplyOrForward = false
                                mBinding.revInput.setText("")
                            }
                        }
                    }
                    "sticker" -> {
                        mCallback?.onSendSticker()
                    }
                }
            }
            revInput.addTextChangedListener(
                object : TextWatcher {
                    var isTyping = false
                    var timer = Timer()

                    override fun afterTextChanged(newText: Editable) {
                        Logger.d("__ChatFooterView afterTextChanged")
                        mCallback?.onDraftMessage(content = newText.toString())

                        if (newText.trim().isEmpty()) {
                            ivSentMessage.isVisible = false
                            ivSentSticker.isVisible = true
                            buttonSend.tag = "sticker"
                        } else {
                            ivSentMessage.isVisible = true
                            ivSentSticker.isVisible = false
                            buttonSend.tag = "message"
                        }
                        if (!isTyping) {
                            isTyping = true
                            mCallback?.isTyping(true)
                        }
                        try {
                            timer.cancel()
                            timer = Timer().apply {
                                schedule(
                                    object : TimerTask() {
                                        override fun run() {
                                            isTyping = false
                                            mCallback?.isTyping(false)
                                        }
                                    },
                                    2000
                                )
                            }
                        } catch (e: Exception) {}
                    }

                    override fun beforeTextChanged(
                        text: CharSequence,
                        start: Int,
                        count: Int,
                        after: Int
                    ) {
                        if (text.length > start + count - 1 && count - after == 1 && text.substring(start + after, start + count) == "@") {
                            mCallback?.closeMention()
                        }
                    }

                    override fun onTextChanged(
                        text: CharSequence,
                        start: Int,
                        before: Int,
                        count: Int
                    ) {
                        if (!isChangeMention) {
                            val cursorPosition = mBinding.revInput.selectionStart
                            if (cursorPosition > 0 && text.subSequence(cursorPosition - 1, cursorPosition).toString() == "@") {
                                mCallback?.openMention()
                            } else if (cursorPosition > 1) {
                                var startMention: Int? = null
                                for (i in cursorPosition downTo 1) {
                                    if (text.subSequence(i - 1, i).toString() == " ") {
                                        break
                                    } else if (text.subSequence(i - 1, i).toString() == "@") {
                                        startMention = i - 1
                                        break
                                    }
                                }
                                if (startMention != null) {
                                    val mention =
                                        text.subSequence(startMention, cursorPosition).toString()
                                    mCallback?.openMentionFilter(
                                        mention.substring(
                                            1,
                                            mention.length
                                        )
                                    )
                                } else {
                                    mCallback?.closeMention()
                                }
                            } else {
                                mCallback?.closeMention()
                            }
                        } else {
                            mCallback?.closeMention()
                        }
                    }
                }
            )

            mBinding.revInput.setOnFocusChangeListener { _, b ->
                if (b) {
                    closeSticker(view = mBinding.buttonSticker, openKeyboard = false)
                }
            }

            mBinding.buttonSticker.clickDebounce {
                if (it.tag == "sticker") {
                    openSticker(view = mBinding.buttonSticker)
                } else {
                    closeSticker(view = mBinding.buttonSticker, openKeyboard = true)
                }
            }

            buttonPhoto.clickDebounce {
                isShowRecord = false
                mCallback?.onPickMediaClicked()
            }

            buttonVoice.clickDebounce {
                isClickVoice = !isClickVoice
                if (isClickVoice) {
                    isShowRecord = true
                    KeyboardUtils.hideSoftInput(mBinding.revInput)
                    closeSticker()
                    btnRecording.isVisible = true
                    btnCloseRecord.isGone = true
                    btnSendRecord.isGone = true
                    mCallback?.onStartRecordVoice()
                } else {
                    isShowRecord = false
                    mCallback?.onStopRecordVoice()
                    isClickVoice = false
                }
            }

            btnUnBlock.clickDebounce {
                mCallback?.onUnBlock()
            }

            containReplyOrForward.imgClear.clickDebounce {
                isReplying = false
                isReplyOrForward = false
                replyOrForwardMessage = null
            }

            btnRecording.clickDebounce {
                mCallback?.onStopRecordVoice()
                it.isGone = true
                btnCloseRecord.isVisible = true
                btnSendRecord.isVisible = true
            }

            btnCloseRecord.clickDebounce {
                isShowRecord = false
                isClickVoice = false
                mCallback?.onResetRecordVoice()
            }

            btnSendRecord.clickDebounce {
                isShowRecord = false
                isClickVoice = false
                mCallback?.processSendAudioMessage()
            }

            btnUnBlock.clickDebounce {
                mCallback?.onUnBlock()
            }

            flTimer.clickDebounce {
                mCallback?.openTimer()
            }

            setupTheme()
        }
    }

    fun setTyping(isTyping: Boolean) {
        if (isTyping) {
            typingAnimation?.start()
        } else {
            typingAnimation?.stop()
        }
        mBinding.isTyping = isTyping
    }

    @SuppressLint("SimpleDateFormat")
    fun startAudioRecord(isStart: Boolean) {
        with(mBinding.chronometer) {
            var date = TimeUtils.millis2String(timeToggle)
            if (isStart) {
                text = date
                countTimerToggle = Timer()
                countTimerToggle?.scheduleAtFixedRate(
                    object : TimerTask() {
                        override fun run() {
                            timeToggle += 1000
                            date = TimeUtils.millis2String(timeToggle)
                            if (timeToggle >= Constants.MAX_RECORDING) {
                                countTimerToggle?.cancel()
                                countTimerToggle = null
                                mCallback?.onStopRecordVoice()
                            } else {
                                MainScope().launch {
                                    text = date
                                }
                            }
                        }
                    },
                    1000,
                    1000
                )
            } else {
                date = TimeUtils.millis2String(timeToggle)
                countTimerToggle?.cancel()
                countTimerToggle = null
                timeToggle = 0L
                MainScope().launch {
                    text = date
                    with(mBinding) {
                        try {
                            btnRecording.isGone = true
                            btnCloseRecord.isVisible = true
                            btnSendRecord.isVisible = true
                        } catch (e: Exception) {
                            e.printStackTrace()
                            FirebaseCrashlytics.getInstance().recordException(e)
                        }
                    }
                }
            }
        }
    }

    fun addMention(neUser: NeUser) {
        isChangeMention = true
        var text = mBinding.revInput.text ?: return
        val cursorPosition = mBinding.revInput.selectionStart
        var startMention: Int? = null
        for (i in cursorPosition downTo 1) {
            if (text.subSequence(i - 1, i).toString() == " ") {
                break
            } else if (text.subSequence(i - 1, i).toString() == "@") {
                startMention = i - 1
                break
            }
        }
        startMention?.let {
            val mentionReply = MentionSpan.build(
                "@${neUser.getDisplayName}",
                themeHelperImpl.mainColor
            )
            text =
                text.replace(it, cursorPosition, mentionReply).insert(it + mentionReply.length, " ")
            mBinding.revInput.text = text
            mBinding.revInput.setSelection(it + mentionReply.length + 1)
        }
        isChangeMention = false
    }

    fun showOwnerBlocked(neUser: NeUser) {
        with(mBinding) {
            layoutBlock.isVisible = true
            tvWasBlocked.isVisible = false
            tvTitleBlock.isVisible = true
            tvDescriptionBlock.isVisible = true
            llBtnBlock.isVisible = true

            val titleBlock = String.format(
                context.resources.getString(R.string.str_blocked),
                neUser.getDisplayName
            )
            tvTitleBlock.text = titleBlock

            llInput.isGone = true
            tvTitleBlock.isVisible = true
            tvDescriptionBlock.isVisible = true
            llBtnBlock.isVisible = true
            tvWasBlocked.isVisible = false
        }
    }

    fun showMemberOfChannel(neGroup: NeGroup) {
        with(mBinding) {
            layoutBlock.isVisible = true
            tvWasBlocked.isVisible = true
            if (neGroup.isMute) {
                tvWasBlocked.text = StringUtils.getString(R.string.turn_on_notification)
            } else {
                tvWasBlocked.text = StringUtils.getString(R.string.turn_off_notification)
            }
            themeHelperImpl.setThemeColorForViews(tvWasBlocked)
            tvWasBlocked.clickDebounce {
                mCallback?.onNotificationGroup()
            }
            llInput.isGone = true
        }
    }

    fun showUserWasBlocked() {
        with(mBinding) {
            layoutBlock.isVisible = true
            tvWasBlocked.isVisible = true

            llInput.isGone = true
            tvTitleBlock.isGone = true
            tvDescriptionBlock.isGone = true
            llBtnBlock.isGone = true
        }
    }

    fun showInputChat() {
        with(mBinding) {
            layoutBlock.isVisible = false
            llInput.isVisible = true
        }
    }

    fun showReplying(isEncrypt: Boolean, message: NeMessage, members: List<NeUser>) {
        if (!isBlockShow && !isWasBlocked) {
            this.isReplying = true
            mBinding.isReplyOrForward = true
            mBinding.containReplyOrForward.txtNameReceiver.text =
                message.owner?.getDisplayName ?: message.senderName ?: context.resources.getString(R.string.unknown_user)
            replyOrForwardMessage = message

            when (replyOrForwardMessage?.type) {
                MESSAGE_TYPE_TEXT, MESSAGE_TYPE_REPLY, MESSAGE_TYPE_FORWARD -> {
                    mBinding.containReplyOrForward.txtMessage.text = AppUtils.getMentionContent(
                        input = replyOrForwardMessage?.content.orEmpty(),
                        neMembers = members
                    )
                    mBinding.containReplyOrForward.rcvPhotoReply.isGone = true
                    mBinding.containReplyOrForward.ivStickerReply.isGone = true
                }
                MESSAGE_TYPE_IMAGE -> {
                    mBinding.containReplyOrForward.txtMessage.text =
                        context.resources.getString(R.string.str_photo)
                    replyOrForwardMessage?.let { neMessage ->
                        setupReplyImage(isEncrypt, neMessage)
                        mBinding.containReplyOrForward.rcvPhotoReply.isVisible = true
                        mBinding.containReplyOrForward.ivStickerReply.isGone = true
                    }
                }
                MESSAGE_TYPE_STICKER -> {
                    mBinding.containReplyOrForward.txtMessage.text =
                        StringUtils.getString(R.string.str_sticker)
                    replyOrForwardMessage?.let { neMessage ->
                        mBinding.containReplyOrForward.ivStickerReply.loadImage(neMessage.attachment?.stickerUrl)
                        mBinding.containReplyOrForward.rcvPhotoReply.isGone = true
                        mBinding.containReplyOrForward.ivStickerReply.isVisible = true
                    }
                }
                MESSAGE_TYPE_VIDEO -> {
                    mBinding.containReplyOrForward.txtMessage.text =
                        context.resources.getString(R.string.str_video)
                    replyOrForwardMessage?.let { neMessage ->
                        setupReplyVideo(isEncrypt, neMessage)
                        mBinding.containReplyOrForward.rcvPhotoReply.isVisible = true
                    }
                }
                MESSAGE_TYPE_AUDIO -> {
                    mBinding.containReplyOrForward.txtMessage.text =
                        context.resources.getString(R.string.str_audio)
                    mBinding.containReplyOrForward.rcvPhotoReply.isGone = true
                }
                MESSAGE_TYPE_FILE -> {
                    replyOrForwardMessage?.attachment?.file?.name?.let {
                        mBinding.containReplyOrForward.txtMessage.text = it
                    }
                }
            }
        }
    }

    private fun setupReplyVideo(isEncrypt: Boolean, message: NeMessage) {
        val thumbnailImg = if (isEncrypt) {
            val localThumbnailSecretPath =
                "${com.netacom.lite.util.FileUtils.secretFolderPath}/${message.id}/${Constants.VIDEO_THUMBNAIL}"
            if (FileUtils.isFileExists(localThumbnailSecretPath)) {
                localThumbnailSecretPath
            } else {
                message.attachment?.video?.thumbnailUrl ?: Constants.EMPTY
            }
        } else {
            message.attachment?.video?.thumbnailUrl ?: Constants.EMPTY
        }

        if (thumbnailImg.isNotNull) {
            mBinding.containReplyOrForward.rcvPhotoReply.setHasFixedSize(true)
            mBinding.containReplyOrForward.rcvPhotoReply.adapter = ReplyImageMessageAdapter()

            mBinding.containReplyOrForward.rcvPhotoReply.layoutManager = LinearLayoutManager(
                mBinding.containReplyOrForward.rcvPhotoReply.context,
                LinearLayoutManager.VERTICAL,
                false
            )
            (mBinding.containReplyOrForward.rcvPhotoReply.adapter as? ReplyImageMessageAdapter)?.display(
                mutableListOf(
                    thumbnailImg
                )
            )
        }
    }

    private fun setupReplyImage(isEncrypt: Boolean, message: NeMessage) {
        val images = message.attachment?.images?.map {
            if (isEncrypt) {
                it.url ?: ""
            } else {
                it.url?.getUrlImage() ?: ""
            }
        } ?: emptyList()

        mBinding.containReplyOrForward.rcvPhotoReply.setHasFixedSize(true)
        mBinding.containReplyOrForward.rcvPhotoReply.adapter = ReplyImageMessageAdapter()

        when {
            images.size < 2 -> {
                mBinding.containReplyOrForward.rcvPhotoReply.layoutManager =
                    LinearLayoutManager(mBinding.containReplyOrForward.rcvPhotoReply.context)
            }
            images.size == 2 -> {
                mBinding.containReplyOrForward.rcvPhotoReply.layoutManager =
                    GridLayoutManager(mBinding.containReplyOrForward.rcvPhotoReply.context, 2)
            }
            else -> {
                mBinding.containReplyOrForward.rcvPhotoReply.layoutManager =
                    GridLayoutManager(mBinding.containReplyOrForward.rcvPhotoReply.context, 3)
            }
        }

        (mBinding.containReplyOrForward.rcvPhotoReply.adapter as? ReplyImageMessageAdapter)?.display(
            images
        )
    }

    fun setDraftMessage(draftContent: String) {
        mBinding.revInput.setText(draftContent)
    }

    fun registerCallBack(callback: ViewChatFooterCallBack) {
        mCallback = callback
    }

    private fun openSticker(view: AppCompatImageView) {
        view.tag = "keyboard"
        view.setImageResource(R.drawable.ic_keyboard_default_blue)
        mCallback?.openStickerView()
    }

    private fun closeSticker(view: AppCompatImageView, openKeyboard: Boolean) {
        view.tag = "sticker"
        view.setImageResource(R.drawable.ic_emoij_default_blue)
        if (openKeyboard) {
            mBinding.revInput.requestFocus()
            KeyboardUtils.showSoftInput(mBinding.revInput)
        }
        mCallback?.closeStickerView()
    }

    fun closeSticker() {
        closeSticker(view = mBinding.buttonSticker, openKeyboard = false)
    }

    fun initForSecret() {
        with(mBinding) {
            flTimer.isVisible = true
            revInput.setPadding(
                SizeUtils.dp2px(8f),
                SizeUtils.dp2px(8f),
                SizeUtils.dp2px(14f),
                SizeUtils.dp2px(8f)
            )
            ImageViewCompat.setImageTintList(
                buttonSticker,
                ColorStateList.valueOf(
                    ContextCompat.getColor(
                        Utils.getApp(),
                        android.R.color.black
                    )
                )
            )

            buttonPhoto.setImageResource(R.drawable.ic_photo_chat_secret)
            buttonVoice.setImageResource(R.drawable.ic_vector_voice_chat_secret)
            ivSentSticker.setImageResource(R.drawable.ic_sticker_secret)
            ivSentMessage.setBackgroundResource(R.drawable.all_style_circle_chat_sercet)
            buttonSticker.setImageResource(R.drawable.ic_emoji_secret)
            themeHelperImpl.setThemeColorForSecretChat(
                buttonPhoto,
                buttonVoice,
                buttonSticker,
                ivSentSticker,
                containReplyOrForward.divider,
                containReplyOrForward.txtNameReceiver,
                btnSendRecord,
                btnUnBlock,
                ivTimer,
                tvTimer
            )
            themeHelperImpl.setThemeColorForSecretBackground(ivSentMessage)
        }
    }

    @SuppressLint("SetTextI18n")
    fun setTimer(timer: Int) {
        if (timer == 0) {
            mBinding.ivTimer.isVisible = true
            mBinding.tvTimer.isGone = true
        } else {
            mBinding.ivTimer.isGone = true
            mBinding.tvTimer.text = "${timer}s"
            mBinding.tvTimer.isVisible = true
        }
    }

    fun setupTheme() {
        mBinding.apply {
            themeHelperImpl.setThemeColorForViews(
                buttonPhoto,
                buttonVoice,
                buttonSticker,
                ivSentSticker,
                containReplyOrForward.divider,
                containReplyOrForward.txtNameReceiver,
                btnSendRecord,
                btnUnBlock
            )
            themeHelperImpl.setThemeColorForBackground(ivSentMessage)
        }
    }

    fun showForwarding(message: NeMessage?, members: List<NeUser>?) {
        if (message == null || members == null) return
        if (!isBlockShow && !isWasBlocked) {
            this.isForwarding = true
            mBinding.isReplyOrForward = true
            mBinding.containReplyOrForward.txtNameReceiver.text =
                message.owner?.getDisplayName ?: message.senderName ?: context.resources.getString(R.string.unknown_user)
            replyOrForwardMessage = message

            when (replyOrForwardMessage?.type) {
                MESSAGE_TYPE_TEXT, MESSAGE_TYPE_REPLY, MESSAGE_TYPE_FORWARD -> {
                    mBinding.containReplyOrForward.txtMessage.text = AppUtils.getMentionContent(
                        input = replyOrForwardMessage?.content.orEmpty(),
                        neMembers = members
                    )
                }
                MESSAGE_TYPE_IMAGE -> {
                    mBinding.containReplyOrForward.txtMessage.text =
                        context.resources.getString(R.string.str_photo)
                    replyOrForwardMessage?.let { neMessage ->
                        setupReplyImage(false, neMessage)
                    }
                }
            }
        }
    }
}
