package com.netacom.full.widget.video

import android.content.Context
import android.net.Uri
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import androidx.annotation.AttrRes
import androidx.annotation.NonNull
import androidx.annotation.Nullable
import androidx.core.net.toUri
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.DefaultLoadControl
import com.google.android.exoplayer2.LoadControl
import com.google.android.exoplayer2.MediaItem
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.SimpleExoPlayer
import com.google.android.exoplayer2.extractor.DefaultExtractorsFactory
import com.google.android.exoplayer2.source.ExtractorMediaSource
import com.google.android.exoplayer2.source.MediaSource
import com.google.android.exoplayer2.source.ProgressiveMediaSource
import com.google.android.exoplayer2.source.dash.DashMediaSource
import com.google.android.exoplayer2.source.hls.HlsMediaSource
import com.google.android.exoplayer2.source.smoothstreaming.SsMediaSource
import com.google.android.exoplayer2.trackselection.AdaptiveTrackSelection
import com.google.android.exoplayer2.trackselection.DefaultTrackSelector
import com.google.android.exoplayer2.ui.PlayerView
import com.google.android.exoplayer2.upstream.DefaultDataSourceFactory
import com.google.android.exoplayer2.upstream.DefaultHttpDataSource
import com.google.android.exoplayer2.util.Util
import com.netacom.base.chat.android_utils.AppUtils
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.util.isLocalLink
import com.netacom.full.R

class ExoPlayerView(@NonNull context: Context, @Nullable attrs: AttributeSet?, @AttrRes defStyleAttr: Int) :
    FrameLayout(context, attrs, defStyleAttr) {

    private var playerView: PlayerView? = null
    private var exoPlayer: SimpleExoPlayer? = null
    private lateinit var dataSourceFactory: DefaultHttpDataSource.Factory
    private lateinit var trackSelector: DefaultTrackSelector
    private lateinit var loadControl: LoadControl

    @JvmOverloads
    constructor(@NonNull context: Context, @Nullable attrs: AttributeSet? = null) : this(context, attrs, 0)

    fun initializePlayer() {
        if (playerView == null) {
            val context = context.applicationContext
            LayoutInflater.from(context).inflate(R.layout.view_video_player, this)
            playerView = findViewById(R.id.exoPlayer)
            dataSourceFactory = DefaultHttpDataSource.Factory().setUserAgent(Util.getUserAgent(context, this::class.java.simpleName))
        }
    }

    fun createExoPlayer(playerExoHelper: PlayerExoHelper) {
        if (exoPlayer == null) {
            loadControl = DefaultLoadControl.Builder()
//                .setAllocator(DefaultAllocator(true, 16))
                .setBufferDurationsMs(
                    DefaultLoadControl.DEFAULT_MAX_BUFFER_MS,
                    DefaultLoadControl.DEFAULT_MAX_BUFFER_MS,
                    DefaultLoadControl.DEFAULT_BUFFER_FOR_PLAYBACK_MS,
                    DefaultLoadControl.DEFAULT_BUFFER_FOR_PLAYBACK_AFTER_REBUFFER_MS
                )
//                .setPrioritizeTimeOverSizeThresholds(true)
                .build()
            trackSelector = DefaultTrackSelector(context, AdaptiveTrackSelection.Factory())
            exoPlayer = SimpleExoPlayer.Builder(context).setTrackSelector(trackSelector).setLoadControl(loadControl).build()
            exoPlayer?.apply {
                addVideoListener(playerExoHelper)
                addListener(playerExoHelper)
            }
            playerView?.apply {
                player = exoPlayer
                setControllerVisibilityListener {
                    playerExoHelper.getPlayerClickListener()?.isShowController(it == View.VISIBLE)
                }
                setPlaybackPreparer(playerExoHelper)
            }
        }
    }

    fun checkPlayer(): Boolean = exoPlayer != null

    fun getPlayer(): Player? = exoPlayer

    fun getTrackSelector() = trackSelector

    fun setMutePlayer() {
        exoPlayer?.volume = 0f
    }

    fun setUnMutePlayer() {
        exoPlayer?.volume = 1f
    }

    fun runExoPlayer(linkUrl: String?, subtitle: String?, position: Long? = 0L, apphash: String, token: String) {
        if (linkUrl.isNullOrEmpty()) {
            return
        }
        Logger.e("runExoPlayer = $linkUrl subtitle = $subtitle")
        exoPlayer?.let { player ->
            player.setMediaSource(buildMediaResource(linkStream = linkUrl.toUri(), apphash = apphash, token = token))
            player.prepare()
            if (position != null && position != 0L) {
                player.seekTo(position)
            }
            player.playWhenReady = true
            Logger.d("check header = ${dataSourceFactory.defaultRequestProperties}")
        }
    }

    private fun buildMediaResource(linkStream: Uri, apphash: String, token: String): MediaSource {
        dataSourceFactory.defaultRequestProperties.set("TC-Token", token)
        return when (@C.ContentType val type = Util.inferContentType(linkStream)) {
            C.TYPE_DASH -> {
                dataSourceFactory.defaultRequestProperties.set("app", AppUtils.getAppPackageName())
                dataSourceFactory.defaultRequestProperties.set("apphash", apphash)
                DashMediaSource.Factory(dataSourceFactory).createMediaSource(MediaItem.fromUri(linkStream))
            }
            C.TYPE_SS -> {
                dataSourceFactory.defaultRequestProperties.set("app", AppUtils.getAppPackageName())
                dataSourceFactory.defaultRequestProperties.set("apphash", apphash)
                SsMediaSource.Factory(dataSourceFactory).createMediaSource(MediaItem.fromUri(linkStream))
            }
            C.TYPE_HLS -> {
                dataSourceFactory.defaultRequestProperties.set("app", AppUtils.getAppPackageName())
                dataSourceFactory.defaultRequestProperties.set("apphash", apphash)
                HlsMediaSource.Factory(dataSourceFactory).createMediaSource(MediaItem.fromUri(linkStream))
            }
            C.TYPE_OTHER -> {
                if (linkStream.toString().isLocalLink()) {
                    val dataSourceFactory = DefaultDataSourceFactory(context, Util.getUserAgent(context, this::class.java.simpleName))
                    ExtractorMediaSource(linkStream, dataSourceFactory, DefaultExtractorsFactory(), null, null)
                } else {
                    dataSourceFactory.defaultRequestProperties.set("app", AppUtils.getAppPackageName())
                    dataSourceFactory.defaultRequestProperties.set("apphash", apphash)
                    ProgressiveMediaSource.Factory(dataSourceFactory).createMediaSource(MediaItem.fromUri(linkStream))
                }
            }
            else -> throw IllegalStateException("Unsupported type: $type")
        }
    }

    fun releasePlayer() {
        stopExoPlayer()
        exoPlayer?.release()
        exoPlayer = null
    }

    fun isPlaying(): Boolean = exoPlayer != null && exoPlayer?.isPlaying == true

    fun isBuffering(): Boolean = exoPlayer != null && exoPlayer?.playbackState == Player.STATE_BUFFERING

    fun isShowController(): Boolean = playerView != null && playerView?.isControllerVisible ?: false

    fun setPlayWhenReady(shouldAutoPlay: Boolean) {
        exoPlayer?.playWhenReady = shouldAutoPlay
    }

    fun retry() = exoPlayer?.prepare()

    fun pauseExoPlayer() {
        exoPlayer?.apply {
            pause()
        }
    }

    fun stopExoPlayer() {
        exoPlayer?.apply {
            seekTo(0)
            stop()
        }
    }

    fun replay() = exoPlayer?.seekTo(0)

    fun seekTo(position: Long) = exoPlayer?.seekTo(position)

    fun setResizeModeRaw(resizeMode: Int) {
        playerView?.resizeMode = resizeMode
    }

    fun switchTargetView(oldPlayerView: ExoPlayerView?, newPlayerView: ExoPlayerView?) {
        if (oldPlayerView === newPlayerView) {
            return
        }
        if (newPlayerView != null) newPlayerView.exoPlayer = exoPlayer
        if (oldPlayerView != null) oldPlayerView.exoPlayer = null
    }

    fun hideController() {
        playerView?.useController = false
    }
}
