/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.worker.message

import android.annotation.SuppressLint
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.media.RingtoneManager
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.core.app.TaskStackBuilder
import androidx.core.os.bundleOf
import androidx.hilt.work.HiltWorker
import androidx.navigation.NavDeepLinkBuilder
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.netacom.base.chat.android_utils.ActivityUtils
import com.netacom.base.chat.android_utils.NotificationUtils
import com.netacom.base.chat.android_utils.NotificationUtils.IMPORTANCE_HIGH
import com.netacom.base.chat.android_utils.UriUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.imageloader.loadBitmapSync
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.base.chat.logger.Logger
import com.netacom.base.chat.network.ApiResponseError
import com.netacom.base.chat.network.ApiResponseSuccess
import com.netacom.base.chat.util.getThemeColor
import com.netacom.full.R
import com.netacom.full.ui.main.theme.ThemeHelperImpl
import com.netacom.full.ui.sdk.NetAloActivity
import com.netacom.lite.BuildConfig
import com.netacom.lite.config.EndPoint
import com.netacom.lite.define.FirebaseDefine
import com.netacom.lite.define.GroupType
import com.netacom.lite.define.NavigationDef
import com.netacom.lite.define.NotificationType.NOTIFICATION_TYPE_DELETE_SECRET
import com.netacom.lite.define.NotificationType.NOTIFICATION_TYPE_GROUP_UPDATED
import com.netacom.lite.define.NotificationType.NOTIFICATION_TYPE_START_SECRET
import com.netacom.lite.entity.fcm.FcmNotification
import com.netacom.lite.entity.socket.Group
import com.netacom.lite.entity.socket.Secret
import com.netacom.lite.entity.ui.group.NeGroup
import com.netacom.lite.entity.ui.secret.NePublicKey
import com.netacom.lite.entity.ui.user.NeUser
import com.netacom.lite.local.prefs.PreferencesHelperImpl
import com.netacom.lite.mapper.Group_DB_UI_Mapper
import com.netacom.lite.repository.GroupRepository
import com.netacom.lite.util.CallbackResult
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch

/**
Created by vantoan on 28/Sep/2020
Company: Netacom.
Email: huynhvantoan.itc@gmail.com
 **/
@HiltWorker
class MessageWorker @AssistedInject constructor(
    @Assisted context: Context,
    @Assisted workerParameters: WorkerParameters,
    private val groupDbUiMapper: Group_DB_UI_Mapper,
    private val jsonSerializer: JsonSerializer,
    private val preferencesHelperImpl: PreferencesHelperImpl,
    private val groupRepository: GroupRepository,
    private val themeHelperImpl: ThemeHelperImpl
) : CoroutineWorker(context, workerParameters) {

    override suspend fun doWork(): Result {
        return try {
            jsonSerializer.asObject(inputData.getString(FirebaseDefine.KEY_DATA), FcmNotification::class.java)?.let { fcmNotification ->
                if (fcmNotification.type == NOTIFICATION_TYPE_START_SECRET) {
                    val neGroup = NeGroup().apply {
                        id = fcmNotification.group_id.toString()
                        type = GroupType.GROUP_TYPE_PRIVATE
                        owner = groupRepository.getUserById(fcmNotification.mSenderUin)
                        members = mutableListOf<NeUser>().apply {
                            owner?.let { this.add(it) }
                            preferencesHelperImpl.user?.let { this.add(it) }
                        }
                        fcmNotification.secret?.let {
                            partnerPublicKey = NePublicKey().apply {
                                pUin = it.uin
                                pDeviceId = it.device_id
                                pIdentityKey = it.identity_key
                                pBaseKey = it.base_key
                                pOneTimePreKey = it.one_time_pre_key
                            }
                        }
                    }
                    checkNotification(fcmNotification, neGroup)
                } else if (fcmNotification.type == NOTIFICATION_TYPE_DELETE_SECRET) {
                    checkNotification(fcmNotification, NeGroup())
                } else if (fcmNotification.type == NOTIFICATION_TYPE_GROUP_UPDATED) {
                    Logger.d("handle message type ==$NOTIFICATION_TYPE_GROUP_UPDATED")
                    updateGroupToDB(
                        Group(groupId = fcmNotification.group_id),
                        fcmNotification = fcmNotification,
                        callbackResult = object : CallbackResult<NeGroup?> {
                            override fun callBackSuccess(result: NeGroup?) {
                                result?.let {
                                    CoroutineScope(Dispatchers.IO).launch {
                                        checkNotification(fcmNotification, it)
                                    }
                                }
                            }

                            override fun callBackError(error: String?) {
                            }
                        }
                    )
                } else {
                    val dbGroupEntity = groupRepository.getGroupDbById(fcmNotification.group_id.toString())
                    if (dbGroupEntity == null) {
                        updateGroupToDB(
                            Group(groupId = fcmNotification.group_id),
                            fcmNotification = fcmNotification,
                            callbackResult = object : CallbackResult<NeGroup?> {
                                override fun callBackSuccess(result: NeGroup?) {
                                    result?.let {
                                        CoroutineScope(Dispatchers.IO).launch {
                                            checkNotification(fcmNotification, it)
                                        }
                                    }
                                }

                                override fun callBackError(error: String?) {
                                }
                            }
                        )
                    } else {
                        checkNotification(fcmNotification, groupDbUiMapper.mapFromDbToUIGroup(dbGroupEntity))
                    }
                }
            }
            Result.success()
        } catch (e: Exception) {
            FirebaseCrashlytics.getInstance().recordException(e)
            Log.d("qqq", "!! exception.message:${e.message}")
            Result.failure()
        }
    }

    private suspend fun checkNotification(fcmNotification: FcmNotification, neGroup: NeGroup) {
        neGroup.isFcm = fcmNotification.isFcm
        if (fcmNotification.type == NOTIFICATION_TYPE_START_SECRET || fcmNotification.type == NOTIFICATION_TYPE_DELETE_SECRET) {
            if (fcmNotification.getDisplayAvatar != null) {
                loadBitmapSync(
                    applicationContext,
                    fcmNotification.getDisplayAvatar,
                    { bitmap ->
                        createNotification(fcmNotification = fcmNotification, bitmap = bitmap, neGroup = neGroup)
                    }
                )
            } else {
                createNotification(fcmNotification = fcmNotification, bitmap = null, neGroup = neGroup)
            }
        } else {
            loadBitmapSync(
                applicationContext,
                fcmNotification.getDisplayAvatar,
                { bitmap ->
                    createNotification(fcmNotification = fcmNotification, bitmap = bitmap, neGroup = neGroup)
                }
            )
        }
    }

    private fun createNotification(fcmNotification: FcmNotification, bitmap: Bitmap?, neGroup: NeGroup) {
        EndPoint.URL_CDN = preferencesHelperImpl.setting?.cdnEndpoint ?: BuildConfig.API_URL

        val pendingIntent = if (fcmNotification.type == NOTIFICATION_TYPE_DELETE_SECRET) {
            gotoGroup()
        } else {
            gotoChat(neGroup)
        }
        pendingIntent?.let { intent ->
            val defaultSoundUri = UriUtils.res2Uri("raw/notification_sound") ?: RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION)
            val channelConfig = NotificationUtils.ChannelConfig(Utils.getApp().packageName, Utils.getApp().packageName, IMPORTANCE_HIGH)
            NotificationUtils.notify(100, channelConfig) {
                it.setSmallIcon(R.drawable.ic_notification)
                    .setColor(getThemeColor(themeHelperImpl.mainColor))
                    .setLargeIcon(bitmap)
                    .setContentTitle(fcmNotification.title)
                    .setContentText(fcmNotification.content)
                    .setAutoCancel(true)
                    .setSound(defaultSoundUri)
                    .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)
                    .setDefaults(NotificationCompat.DEFAULT_ALL)
                    .setPriority(NotificationCompat.PRIORITY_MAX)
                    .setContentIntent(intent)
                    .build()
            }
        }
    }

    @SuppressLint("UnspecifiedImmutableFlag")
    private fun gotoChat(neGroup: NeGroup): PendingIntent? {
        val classSdk = preferencesHelperImpl.sdkConfig?.classMainActivity ?: kotlin.run { null }
        val sdkActive = classSdk?.run { ActivityUtils.isActivityExistsInStack(this) } ?: kotlin.run { false }
        if (sdkActive) {
            val intent = Intent(applicationContext, NetAloActivity::class.java)
            intent.action = Intent.ACTION_MAIN
            intent.addCategory(Intent.CATEGORY_LAUNCHER)
            intent.flags = Intent.FLAG_ACTIVITY_BROUGHT_TO_FRONT
            intent.putExtras(bundleOf(NavigationDef.ARGUMENT_NEUSER to neGroup.owner))
            return PendingIntent.getActivity(applicationContext, 0, intent, PendingIntent.FLAG_ONE_SHOT)
        } else {
            return if (classSdk != null) {
                TaskStackBuilder.create(applicationContext).run {
                    addNextIntentWithParentStack(
                        Intent(applicationContext, Class.forName(classSdk)).run {
                            putExtras(bundleOf(NavigationDef.ARGUMENT_NEUSER to neGroup.owner))
                        }
                    )
                    getPendingIntent(0, PendingIntent.FLAG_UPDATE_CURRENT)
                }
            } else {
                null
            }
        }
    }

    private fun gotoGroup(): PendingIntent? {
        val classSdk = preferencesHelperImpl.sdkConfig?.classMainActivity ?: kotlin.run { null }
        val sdkActive = classSdk?.run { ActivityUtils.isActivityExistsInStack(this) } ?: kotlin.run { false }
        if (sdkActive) {
            return NavDeepLinkBuilder(applicationContext)
                .setComponentName(NetAloActivity::class.java)
                .setGraph(R.navigation.nav_main)
                .setDestination(R.id.groupFragment)
                .createPendingIntent()
        } else {
            return if (classSdk != null) {
                TaskStackBuilder.create(applicationContext).run {
                    addNextIntentWithParentStack(
                        Intent(applicationContext, Class.forName(classSdk)).run {
                            putExtras(bundleOf(NavigationDef.ARGUMENT_NEUSER to null))
                        }
                    )
                    getPendingIntent(0, PendingIntent.FLAG_UPDATE_CURRENT)
                }
            } else {
                null
            }
        }
    }

    private suspend fun updateGroupToDB(
        group: Group,
        fcmNotification: FcmNotification,
        callbackResult: CallbackResult<NeGroup?>
    ) {
        groupRepository.getGroupInfoById(group.groupId).collect { state ->
            when (state) {
                is ApiResponseSuccess -> {
                    state.data?.let { infoGroupResponse ->
                        infoGroupResponse.partnerInfos?.let { groupRepository.saveListPartner(it) }
                        infoGroupResponse.group?.let { group ->
                            if (fcmNotification.type == NOTIFICATION_TYPE_START_SECRET) {
                                fcmNotification.secret?.let {
                                    group.secrets = listOf(
                                        Secret().apply {
                                            uin = it.uin
                                            groupId = group.groupId
                                            device_id = it.device_id
                                            identity_key = it.identity_key
                                            base_key = it.base_key
                                            one_time_pre_key = it.one_time_pre_key
                                        }
                                    )
                                }
                            }
                            groupRepository.saveMessageToDB(group)
                            groupRepository.saveGroupToDB(group)?.let {
                                callbackResult.callBackSuccess(it)
                            }
                        }
                    }
                }
                is ApiResponseError -> callbackResult.callBackSuccess(null)
            }
        }
    }
}
