/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.worker.upload

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import androidx.hilt.work.HiltWorker
import androidx.work.Data
import androidx.work.ListenableWorker
import androidx.work.WorkerParameters
import androidx.work.impl.utils.futures.SettableFuture
import androidx.work.workDataOf
import com.abedelazizshe.lightcompressorlibrary.CompressionListener
import com.abedelazizshe.lightcompressorlibrary.VideoCompressor
import com.abedelazizshe.lightcompressorlibrary.VideoQuality
import com.google.common.util.concurrent.ListenableFuture
import com.netacom.base.chat.android_utils.ImageUtils
import com.netacom.base.chat.android_utils.Utils
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.base.chat.util.ConvertUtil.convertSize
import com.netacom.lite.define.MediaType
import com.netacom.lite.network.model.FileUpload
import com.netacom.lite.util.AppUtils
import com.netacom.lite.util.CompressUtil
import com.netacom.lite.util.Constants
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.launch
import java.util.concurrent.Executors

@HiltWorker
class CompressWorker @AssistedInject constructor(
    @Assisted context: Context,
    @Assisted workerParameters: WorkerParameters,
    private val jsonSerializer: JsonSerializer
) : ListenableWorker(context, workerParameters) {

    companion object {
        const val KEY_MEDIA_TYPE_ARG = "KEY_MEDIA_TYPE_ARG"
        const val KEY_UPLOAD_FILE_ARG = "KEY_UPLOAD_FILE_ARG"
        const val KEY_INDEX_ARG = "KEY_INDEX_ARG"
        const val KEY_COMPRESS_RATIO_ARG = "KEY_COMPRESS_RATIO_ARG"
        const val KEY_COMPRESS_RESULT = "KEY_COMPRESS_RESULT"
        const val KEY_COMPRESS_PROGRESS = "KEY_COMPRESS_PROGRESS"
    }

    @SuppressLint("RestrictedApi")
    override fun startWork(): ListenableFuture<Result> {
        val future: SettableFuture<Result> = SettableFuture.create()
        val fileUpload = jsonSerializer.asObject(inputData.getString(KEY_UPLOAD_FILE_ARG), FileUpload::class.java)
        val mediaType = jsonSerializer.asObject(inputData.getString(KEY_MEDIA_TYPE_ARG), MediaType::class.java)
        if (fileUpload == null) {
            Result.failure()
        } else {
            CoroutineScope(Executors.newFixedThreadPool(Constants.THREAD_UPLOAD_LIMIT).asCoroutineDispatcher()).launch {
                if (mediaType == MediaType.VIDEO) {
                    val compressRatio = inputData.getFloat(KEY_COMPRESS_RATIO_ARG, 0F)
                    if (compressRatio == 0F) {
                        val output: Data = workDataOf(KEY_COMPRESS_RESULT to inputData.getString(KEY_UPLOAD_FILE_ARG))
                        future.set(Result.success(output))
                    } else {
                        var currentPercent = -1
                        val destPath = CompressUtil.createTempVideoFile()?.path ?: ""
                        VideoCompressor.start(
                            fileUpload.file,
                            destPath,
                            object : CompressionListener {
                                override fun onProgress(percent: Float) {
                                    val intPercent = percent.toInt()
                                    if (!isStopped && currentPercent < intPercent) {
                                        setProgressAsync(workDataOf(KEY_COMPRESS_PROGRESS to intPercent))
//                            Log.e("VideoCompressor", "VideoCompressor   $intPercent")
                                        currentPercent = intPercent
                                    }
                                }

                                override fun onStart() {
                                    Log.e("Start Compressing", "Start Compressing")
                                }

                                override fun onSuccess() {
                                    Log.e("End Compressing", "End Compressing")
                                    Log.e("Origin Video", "Origin Video Size:   ${fileUpload.getFileSize().convertSize()}")
                                    fileUpload.file = destPath
                                    Log.e("Compressed Video", "Compressed Video Size:   ${fileUpload.getFileSize().convertSize()}")
                                    val output: Data = workDataOf(KEY_COMPRESS_RESULT to jsonSerializer.asJson(fileUpload, FileUpload::class.java))
                                    future.set(Result.success(output))
                                }

                                override fun onFailure(failureMessage: String) {
                                    VideoCompressor.cancel()
                                    fileUpload.file = destPath
                                    val output: Data = workDataOf(KEY_COMPRESS_RESULT to jsonSerializer.asJson(fileUpload, FileUpload::class.java))
                                    future.set(Result.failure(output))
                                }

                                override fun onCancelled() {
                                }
                            },
                            quality = if (compressRatio == Constants.HIGH_VIDEO_SIZE) {
                                VideoQuality.VERY_HIGH
                            } else {
                                VideoQuality.HIGH
                            },
                            isMinBitRateEnabled = false,
                            keepOriginalResolution = true
                        )
                    }
                } else if (mediaType == MediaType.PHOTO) {
                    val file = CompressUtil.compressImage(fileUpload.file)

                    val imageSize = ImageUtils.getSize(file)
                    ImageUtils.rotate(
                        ImageUtils.getBitmap(file),
                        ImageUtils.getRotateDegree(fileUpload.file),
                        (imageSize[0] / 2).toFloat(),
                        (imageSize[1] / 2).toFloat(),
                        true
                    )?.apply {
                        AppUtils.createFileFromBitmap(Utils.getApp(), this)?.let { file ->
                            fileUpload.file = file.absolutePath
                        }
                    }

                    val output: Data = workDataOf(KEY_COMPRESS_RESULT to jsonSerializer.asJson(fileUpload, FileUpload::class.java))
                    future.set(Result.success(output))
                } else {
                    val output: Data = workDataOf(KEY_COMPRESS_RESULT to inputData.getString(KEY_UPLOAD_FILE_ARG))
                    future.set(Result.success(output))
                }
            }
        }
        return future
    }
}
