/*
 * *Created by NetaloTeamAndroid on 2020
 * Company: Netacom.
 *  *
 */

package com.netacom.full.worker.upload

import android.content.Context
import androidx.hilt.work.HiltWorker
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.netacom.base.chat.json.JsonSerializer
import com.netacom.lite.define.MediaType
import com.netacom.lite.network.model.FileUpload
import com.netacom.lite.network.model.response.UploadCallBack
import com.netacom.lite.network.model.response.UploadResult
import com.netacom.lite.repository.UploadRepository
import com.netacom.lite.util.Constants.THREAD_UPLOAD_LIMIT
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.withContext
import java.util.concurrent.Executors

@HiltWorker
class UploadWorker @AssistedInject constructor(
    @Assisted context: Context,
    @Assisted workerParameters: WorkerParameters,
    private val uploadRepository: UploadRepository,
    private val jsonSerializer: JsonSerializer
) : CoroutineWorker(context, workerParameters) {

    companion object {
        const val KEY_THUMBNAIL_ARG = "KEY_THUMBNAIL_ARG"
        const val KEY_ENCRYPT_ARG = "KEY_ENCRYPT_ARG"
        const val KEY_MEDIA_TYPE_ARG = "KEY_MEDIA_TYPE_ARG"
        const val KEY_UPLOAD_FILE_ARG = "KEY_UPLOAD_FILE_ARG"
        const val KEY_INDEX_ARG = "KEY_INDEX_ARG"
        const val KEY_UPLOAD_RESULT = "KEY_UPLOAD_RESULT"
    }

    override suspend fun doWork(): Result {
        return withContext(Executors.newFixedThreadPool(THREAD_UPLOAD_LIMIT).asCoroutineDispatcher()) {
            val isThumbnail = inputData.getBoolean(KEY_THUMBNAIL_ARG, false)
            val isEncryptedFile = inputData.getBoolean(KEY_ENCRYPT_ARG, false)
            val mediaType = jsonSerializer.asObject(inputData.getString(KEY_MEDIA_TYPE_ARG), MediaType::class.java)
            var fileUpload = jsonSerializer.asObject(inputData.getString(KEY_UPLOAD_FILE_ARG), FileUpload::class.java)
            jsonSerializer.asObject(inputData.getString(CompressWorker.KEY_COMPRESS_RESULT), FileUpload::class.java)?.let {
                fileUpload = it
            }
            val index = inputData.getInt(KEY_INDEX_ARG, 0)

            if (fileUpload == null || mediaType == null) {
                return@withContext Result.failure()
            } else {
                val result = uploadRepository.uploadFiles(
                    isThumbnail,
                    isEncryptedFile,
                    mediaType,
                    fileUpload!!,
                    index,
                    object : UploadCallBack {
                        override fun onResult(result: UploadResult) {
                            if (!isThumbnail && !isStopped) {
                                setProgressAsync(workDataOf(KEY_UPLOAD_RESULT to jsonSerializer.asJson(result, UploadResult::class.java)))
                            }
                        }
                    }
                )
                val output: Data = workDataOf(KEY_UPLOAD_RESULT to jsonSerializer.asJson(result, UploadResult::class.java))
                if (result.status == UploadResult.UploadResultStatus.FAIL) {
                    return@withContext Result.failure(output)
                } else {
                    return@withContext Result.success(output)
                }
            }
        }
    }
}
